/*
 # vim: ts=4 sw=4 et
 */

////////////////////////////////////////////////////////////////////////////////
// Copyright 2023-2025 by NI SP Software GmbH, All rights reserved.
// Copyright 1999-2023 by Nice, srl., All rights reserved.
//
// This software includes confidential and proprietary information
// of NI SP Software GmbH ("Confidential Information").
// You shall not disclose such Confidential Information
// and shall use it only in accordance with the terms of
// the license agreement you entered into with NI SP Software.
////////////////////////////////////////////////////////////////////////////////


/**
 * Submit the EnginFrame action by previously activating the
 * Multiple File Upload Applet if it's present on the page
 *
 * Changes global state:
 *     _isWorking set to 'true' as soon as invoked is entered
 *     _fileUploadForm set to 'form' input, i.e. the service form being submitted
 *     _formApplets set to '_getAll' invocation, i.e. Applets on page
 *
 * Uses global state:
 *     _forceSubmit that is changed in 'efApplets.fileUploadCallback'
 *
 * Returns:
 *     true when global '_forceSubmit' is true
 *     false when global '_isWorking' is true
 *     false if there's an Applet and it has been invoked
 *     true otherwise
 */
var efApplets = {

    _isWorking: false,
    _forceSubmit: false,
    _fileUploadForm: '',
    _formApplets: '',

    _isMSIE: function () {
        var userAgent = navigator.userAgent.toLowerCase();
        return (userAgent.indexOf("msie") > 0 || userAgent.indexOf("trident/") > 0) && userAgent.indexOf("opera") == -1;
    },

    _done: [],
    _doneIdx: 0,

    submitEFAction: function (form) {
        var argv = arguments,
            argc = arguments.length;

        if (efApplets._forceSubmit) {
            return true;
        }

        if (efApplets._isWorking) {
            return false;
        }

        efApplets._fileUploadForm = form;
        efApplets._formApplets = efApplets._getAll(efApplets._getAppletName(argc > 1 ? argv[1] : 'UploadApplet'), true);

        if (efApplets._formApplets.length > 0) {
            efApplets._isWorking = true;
            // Pass in 'null' the first time, because surely
            // no one has set EF_REUSE_SPOOLER yet.
            try {
                efApplets._formApplets[0].upload(null, "efApplets.fileUploadCallback");
            }
            catch(e) {
                efApplets._isWorking = false;
                return true;
            }
            return false;
        }
        return true;
    },

    _getAppletName: function (name) {
        var appletName = name;
        if (efApplets._isMSIE()) {
            appletName = "IE_" + appletName;
        }
        return appletName;
    },

    //Applets show/hide utility functions
    hideAll: function () {
        var i,
            appletList = efApplets._getAll(efApplets._getAppletName("UploadApplet"));
        for (i = 0; i < appletList.length ; i++) {
            appletList[i].className += ' ef-applet-hidden';
        }
    },

    showAll: function () {
        var i,
            appletList = efApplets._getAll(efApplets._getAppletName("UploadApplet"));
        for (i = 0; i < appletList.length ; i++) {
            appletList[i].className = appletList[i].className.replace(' ef-applet-hidden', '');
        }
    },

    /**
     * Callback function used by the file upload applet to control
     * the upload status at the end of each upload action. This
     * function calls 'document.getElementById(targetActionId).click()'
     * that returns to 'submitEFAction' invocation. But this time the
     * global variable '_forceSubmit' was set to 'true' so the form
     * is really submitted to the server.
     *
     * Changes global state:
     *     _doneIdx incremented for each Applet invoked on page
     *     _forceSubmit set to 'true' when all Applets finished with success
     *     _done set to '_formApplets' based on '_doneIdx'
     *     _fileUploadForm.EF_REUSE_SPOOLER is reset when any of the Applets fail
     *
     * Uses global state:
     *     _formApplets that is read to store Applets to call in case of error recovery
     *     _fileUploadForm.EF_REUSE_SPOOLER.value passed along to Applets
     *
     * Returns:
     *     true when global '_forceSubmit' is true
     *     false when global '_isWorking' is true
     *     false if there's an Applet and it has been invoked
     *     true otherwise
     */
    fileUploadCallback: function (status) {
        if (status) {
            efApplets._done[efApplets._doneIdx] = efApplets._formApplets[efApplets._doneIdx];
            efApplets._doneIdx++;
            if (efApplets._formApplets.length > efApplets._doneIdx) {
                efApplets._formApplets[efApplets._doneIdx].upload(efApplets._fileUploadForm.EF_REUSE_SPOOLER.value, "efApplets.fileUploadCallback");
            }
            else {
                efApplets._forceSubmit = true;
                if (typeof(jQuery) === "function") {
                    // Temporarily disable the "double submission prevention" mechanism
                    // See efPreventDoubleSubmission in com.enginframe.system.js
                    jQuery(efApplets._fileUploadForm).data('EFLastSubmitTime', 0);
                }
                document.getElementById(targetActionId).click();
            }
        }
        else {
            // If upload fails, let previous Applets recover,
            // reset EF_REUSE_SPOOLER, and bail out
            for (var i = 0; i < efApplets._doneIdx; i++) {
                efApplets._done[i].recover();
            }
            efApplets._doneIdx = 0;
            efApplets._fileUploadForm.EF_REUSE_SPOOLER.value = "";
            efApplets._isWorking = false;
        }
    },

    _isVisible: function (elem){
        return (elem.offsetWidth > 0 || elem.offsetHeight > 0) &&
            !elem.className.match(/\bef-applet-hidden-disabled\b/) !== null;
    },

    /**
     * Returns the Applets whose name is 'appletName'.
     * The returned array is never null.
     *
     * @param appletName Used in document.getElementsByName
     * @param onlyVisible Only returns visible applets (default: false)
     * @return JS Array of Applets whose name is 'appletName'.
     */
    _getAll: function(appletName, onlyVisible) {
        var i, elem,
            theApplets = [],
            allApplets = document.getElementsByName(appletName);
        onlyVisible = (onlyVisible || false) === true;
        if (allApplets) {
            for (i = 0; i < allApplets.length; i++) {
                elem = allApplets.item(i);
                if (!onlyVisible || efApplets._isVisible(elem)) {
                    theApplets[i] = elem;
                }
            }
        }
        return theApplets;
    },

    /**
     * Sets the field specified by 'id' to its new 'value'.
     *
     * @param id Used in document.getElementById
     * @param vale The new value
     */
    setAppletField: function (id, value) {
        document.getElementById(id).value = value;
    },

    /**
     * Open a new window that contains the HTML to start the MFD Applet.
     * The new window contains the 'closeWindow' function that is used
     * by the Applet itself to close '_MFDWindow'.
     *
     * Changes global state:
     *     _MFDWindow is closed if already open; otherwise it's opened
     *                  with the HTML contents that start the MFD Applet.
     *
     * @param root_context The EnginFrame root context
     * @param spoolerURI The location where to download files from
     * @param fileList The list, separated by ',', of files to download
     * @param fileSizes The list, separated by ',', of byte sizes of the files to download
     */
    _MFDWindow: 0,

    openMFDWindow: function (root_context, spoolerURI, plugin, fileList, fileSizes) {
        var myOptions;

        if (efApplets._MFDWindow && !efApplets._MFDWindow.closed) {
            efApplets._MFDWindow.close();
        }

        myOptions = "";
        myOptions = myOptions +       "location=no";
        myOptions = myOptions + "," + "menubar=no";
        myOptions = myOptions + "," + "directories=no";
        myOptions = myOptions + "," + "width=470";
        myOptions = myOptions + "," + "height=180";
        myOptions = myOptions + "," + "toolbar=no";
        myOptions = myOptions + "," + "status=no";
        myOptions = myOptions + "," + "titlebar=no";
        myOptions = myOptions + "," + "resizable=no";
        myOptions = myOptions + "," + "scrollbars=no";
        myOptions = myOptions + "," + "copyhistory=no";

        efApplets._MFDWindow = window.open('', 'MFD', myOptions);
        efApplets._MFDWindow.document.writeln('<html>');
        efApplets._MFDWindow.document.writeln('  <head>');
        efApplets._MFDWindow.document.writeln('    <title>EnginFrame MFD</title>');
        efApplets._MFDWindow.document.writeln('    <script type="text/javascript">');
        efApplets._MFDWindow.document.writeln('      function closeWindow() { window.focus(); window.close(); }');
        efApplets._MFDWindow.document.writeln('    </script>');
        efApplets._MFDWindow.document.writeln('  </head>');
        efApplets._MFDWindow.document.writeln('  <body>');
        efApplets._MFDWindow.document.writeln('    <applet archive  = "mfd.jar"'+
                                              '            codebase = "/' + root_context + '/client"'+
                                              '            code     = "com.enginframe.client.FileDownloadApplet"'+
                                              '            name     = "DownloadApplet"'+
                                              '            width    = "450"'+
                                              '            height   = "160"'+
                                              '            hspace   = "0"'+
                                              '            vspace   = "0"'+
                                              '            align    = "top"'+
                                              '           MAYSCRIPT = "true">');
        efApplets._MFDWindow.document.writeln('      <PARAM name="scriptable" value="true" />');
        efApplets._MFDWindow.document.writeln('      <PARAM name="url"        value="/download" />');
        efApplets._MFDWindow.document.writeln('      <PARAM name="spoolerURI" value="' + spoolerURI + '" />');
        efApplets._MFDWindow.document.writeln('      <PARAM name="plugin"     value="' + plugin + '" />');
        efApplets._MFDWindow.document.writeln('      <PARAM name="fileList"   value="' + fileList + '" />');
        efApplets._MFDWindow.document.writeln('      <PARAM name="fileSizes"  value="' + fileSizes + '" />');
        efApplets._MFDWindow.document.writeln('    </applet>');
        efApplets._MFDWindow.document.writeln('  </body>');
        efApplets._MFDWindow.document.writeln('</html>');
    }
};
