#!/bin/bash

# Job script

# Command to execute Jupyter Notebook
JUPYTER_CMD="jupyter"

# Function to parse the nat.conf file and find matching mapping
transform_hostport() {
    local input="$1"
    local nat_conf="${NAT_CONF_FILE}"

    if [[ -z "${nat_conf}" || ! -f "${nat_conf}" ]] ; then
        echo "${input}"
        return
    fi

    local input_lower

    # Convert input to lowercase for case-insensitive matching
    input_lower=$(echo "$input" | tr '[:upper:]' '[:lower:]')
    # Extract host and port
    if [[ $input_lower =~ ^([^:]+):([0-9]+)$ ]]; then
        local input_host="${BASH_REMATCH[1]}"
        local input_port="${BASH_REMATCH[2]}"

        while IFS= read -r line || [[ -n "$line" ]]; do
            # Skip comments and empty lines
            [[ $line =~ ^[[:space:]]*#.*$ || -z "${line// }" ]] && continue

            # Split the line into source and destination (handling multiple spaces)
            read -r src dst <<< "$(echo "$line" | tr -s ' ')"
            [[ -z "$src" || -z "$dst" ]] && continue

            # Parse source host and port
            if [[ $src =~ ^([^:]+)(:([0-9]+))?$ ]]; then
                local src_host="${BASH_REMATCH[1]}"
                local src_port="${BASH_REMATCH[3]}"
        
                # Parse destination host and port
                if [[ $dst =~ ^([^:]+)(:([0-9]+))?$ ]]; then
                    local dst_host="${BASH_REMATCH[1]}"
                    local dst_port="${BASH_REMATCH[3]}"

                    # Convert to lowercase for comparison
                    if [[ ${src_host,,} == "$input_host" ]]; then
                        # If source port matches input port
                        if [[ -z $src_port || $src_port == "$input_port" ]]; then
                            # If no destination port specified, use input port
                            if [[ -z $dst_port ]]; then
                                echo "$dst_host:$input_port"
                            else
                                echo "$dst_host:$dst_port"
                            fi

                            return 0
                        fi
                    fi
                fi
            fi
        done < "$nat_conf"
    fi

    # If no mapping found, return original input
    echo "$input"
}

# set -x 

# check if port is available
# suggested port is in variable $port

if [[ -z $port ]]; then
    port=8899
fi

_=$(which ss) # check if ss command is available
if [[ $? == 0 && -n $port ]] ; then
    for ((p=port; p<=65535; p++)); do
        if ! ss -tulnH "( sport = :$p )" | grep "$p" > /dev/null; then
            # echo "Port $p is available."
            break
        fi
    done
    port=$p
    echo "We will use port $port as it is available."
else
    _=$(which netstat) # check if netstat command is available
    if [[ $? == 0 && -n $port ]] ; then
        for ((p=port; p<=65535; p++)); do
            if ! netstat -tuln | grep ":$p " > /dev/null; then
                break
            fi
        done
        port=$p
        echo "We will use port $port as it is available (netstat)."
    fi
fi

input_hostport="$(hostname):$port" 
hostport=$(transform_hostport "$input_hostport")
echo Using "$hostport" after NAT translation

function show_url () {
    count=0
    max_retries=30

    # Try to get the token
    token=$($JUPYTER_CMD notebook list | grep -v "Currently" | awk '{print $1}' | awk -F'token=' '{print $2}')

    # If token is not available, poll for it
    while [ -z "$token" ]; do
        count=$((count + 1))
        if [[ $count -gt $max_retries ]]; then
            echo "Failed to retrieve a valid token after $max_retries attempts."
            exit 1
        fi

        echo "Waiting for token... (Attempt ${count})"
        sleep 2
        # Try again 
        token=$($JUPYTER_CMD notebook list | grep -v "Currently" | awk '{print $1}' | awk -F'token=' '{print $2}')
    done

    echo "http://${hostport}/notebooks/${file}?token=$token" > notebook.url 
    
    echo
    echo "Notebook URL:"
    echo "http://${hostport}/notebooks/${file}?token=$token"  # show URL in the job output
    echo "<a target=\"_new\" href=\"http://${hostport}/notebooks/${file}?token=${token}\">Jupyter Notebook ${file}</a>"  # show URL in the job output
    echo 
}

ip="0.0.0.0"

if [ "${file}" != "" ] ; then 
    
    # show_url
    (sleep 3; show_url) &
    
    $JUPYTER_CMD notebook --port="$port" --no-browser --ip="$ip" \
        --notebook-dir="$EF_SPOOLER" \
        "${EF_SPOOLER}/${file}" | tee -a jupyter-output.txt

elif [ "${rfb_file}" != "" ] ; then 
    echo Copying "${rfb_file}" to local directory ...
    cp  "${rfb_file}" .
    file=$(basename "${rfb_file}")
    
    # show_url
    (sleep 3; show_url) &
    
    $JUPYTER_CMD notebook  --port="$port" --no-browser --ip="$ip" \
        --notebook-dir="$EF_SPOOLER" \
        "${EF_SPOOLER}/${file}" | tee -a jupyter-output.txt

else
    echo It seems no file was selected...
fi
