////////////////////////////////////////////////////////////////////////////////
// Copyright 2023-2025 by NI SP Software GmbH, All rights reserved.
// Copyright 1999-2023 by Nice, srl., All rights reserved.
//
// This software includes confidential and proprietary information
// of NI SP Software GmbH ("Confidential Information").
// You shall not disclose such Confidential Information
// and shall use it only in accordance with the terms of
// the license agreement you entered into with NI SP Software.
////////////////////////////////////////////////////////////////////////////////

/*jslint
    maxlen: 160
    white: true
    plusplus: true
*/
/*
global jQuery, console, window, alert, qq, targetActionId, File, FileReader, SHA1
*/

jQuery.widget("ef.fileupload", {
    options: {
        optionId: "",
        serviceOptionId: "",
        serviceUri: "",
        sdfUrl: window.location.protocol + '//' + window.location.host + window.location.pathname,
        windowId: "",
        spoolerOptionId: "serviceopt_EF_REUSE_SPOOLER",
        efRootContext: "enginframe",
        fileSeparator: "\n",
        fineUploaderElementId: "",
        fineUploaderTemplate: "ef-fileupload-template",
        multiple: true,
        maxConnections: undefined,
        autoUpload: true,
        acceptAttribute: "",
        fileFilters: [],
        debug: false
    },

    _status: Object.freeze({
        INIT: "init",
        ACTIVE: "active",
        DONE: "done"
    }),

    _toString: function () {
        return "[ef.fileupload(" + this.options.optionId + ")]";
    },

    _log: function () {
        if (this.options.debug) {
//            [].splice.call(arguments, 0, 0, this._toString());
            [].unshift.call(arguments, this._toString());
            console.log.apply(console, arguments);
        }
    },

    _err: function () {
        if (this.options.debug) {
            if (console.error) {
                [].unshift.call(arguments, this._toString(), "  ERROR  ");
                console.error.apply(console, arguments);
            } else {
                [].unshift.call(arguments, "  ERROR  ");
                this._log(arguments);
            }
        }
    },

    _initChecks: function () {
        if (!this.options.optionId || !this.options.serviceOptionId || !this.options.serviceUri || !this.options.windowId ||
                !this.options.sdfUrl || !this.options.efRootContext || !this.options.fineUploaderElementId) {
            // TODO: initialization error message and break or exception?
            window.alert("Initialization error: invalid input options");
            this._log("Initialization error: invalid input options", this.options);

            throw new Error("Initialization error: invalid input options.");
        }
        // TODO: check support for HTML5 File API.
    },

    _create: function () {
        var self = this,
            fineUploaderElementSelector;

        this._initChecks();

        fineUploaderElementSelector = '#' + this.options.fineUploaderElementId;

        this._log("_create()");

        // Private attributes of the ef.fileupload widget instance
        this.status = this._status.INIT;
        this._uploadProtocol = this._createUploadProtocol();
        this._tidFileMap = {};
        this._cachedFileIds = {};
        this._hashers = [];
        this._initValidators();
        this._fineUploader = this._createFineUploader();
        // ---

        // set file list height
        jQuery('.ef-fileupload-list', fineUploaderElementSelector)
            .height(jQuery(fineUploaderElementSelector).height() - jQuery('.ef-fileupload-header', fineUploaderElementSelector).outerHeight());

        if (this.options.multiple) {
            // bind Cancel All button
            jQuery('.qq-upload-button.qq-upload-cancel-all', fineUploaderElementSelector).click(function () {
                self._fineUploader.cancelAll();
                jQuery(this).addClass('qq-hide');
            });
            // bind Delete All button
            jQuery('.qq-upload-button.qq-upload-delete-all', fineUploaderElementSelector).click(function () {
                self._deleteAll();
                jQuery(this).addClass('qq-hide');
            });
            // set drop area text
            self._setDropAreaTextUI('Drop files and folders here');
        } else {
            jQuery('.qq-total-progress-bar-container').css('visibility', 'hidden');
            // set drop area text
            self._setDropAreaTextUI('Drop a file here');
        }
    },

    _init: function () {
        var efUploader = this;

        this._log("_init()");
        //
        // FIXME:what about passing this.options?
        this._uploadProtocol._init({
            client: "fineuploader",
            efRootContext: this.options.efRootContext,
            onInitSuccess: function (session) {
                var efReuseSpooler = jQuery('#' + efUploader.options.spoolerOptionId);
                // var efReuseSpooler = jQuery('input[name=EF_REUSE_SPOOLER]');
                if (!efReuseSpooler.val()) {
                    efReuseSpooler.val(session.spooler);
                }
            },
            debug: this.options.debug
        });

        this._uploadProtocol.ucInit(this.options.serviceOptionId, this.options.serviceUri, this.options.sdfUrl, this.options.windowId);
    },

    _destroy: function () {},

    doUpload: function () {
        this._fineUploader.uploadStoredFiles();
    },

    // onStart is triggered when an upload or cancel operation is triggered
    onStart: function (event, fileupload) {},

    // onCompletion when all the upload or the cancel operations are done
    onCompletion: function (event, fileupload) {},

    cancel: function () {
        if (this.status === this._status.DONE || this.status === this._status.INIT) {
            this._uploadProtocol.ucClose("true");
        } else {
            this._err("It cannot be closed since it's not in 'INIT' or 'DONE' status.");
        }
    },

    close: function () {
        if (this.status === this._status.DONE || this.status === this._status.INIT) {
            this._uploadProtocol.ucClose();
            var inputField = '', i, uploadItem,
                uploads = this._fineUploader.getUploads();
            for (i = 0; i < uploads.length; i += 1) {
                uploadItem = uploads[i];
                if (uploadItem.status === qq.status.UPLOAD_SUCCESSFUL ||
                        uploadItem.status === qq.status.DELETE_FAILED ||
                        (uploadItem.status === qq.status.REJECTED && uploadItem.id in this._cachedFileIds)) {
                    if (inputField.length > 0) {
                        inputField += this.options.fileSeparator;
                    }
                    inputField += this._getFilePath(this._fineUploader.getFile(uploadItem.id));
                }
            }
            this._log("Filling input option:", this.options.optionId, " \n  value:", inputField);
            jQuery("#" + this.options.optionId).val(inputField);
        } else {
            this._err("It cannot be closed since it's not in 'INIT' or 'DONE' status.");
        }
    },

    _evaluateStatus: function () {
        var submitting, active, deleting, done, failed, cached, finished;

        // TODO Evaluate if is more efficient to perform a single getUploads query
        submitting = this._fineUploader.getUploads({
            status: [qq.status.SUBMITTING]
        }).length;
        active = this._fineUploader.getUploads({
            status: [qq.status.SUBMITTING, qq.status.SUBMITTED, qq.status.QUEUED,
                    qq.status.UPLOADING, qq.status.UPLOAD_RETRYING, qq.status.DELETING, qq.status.PAUSED]
        }).length;
        deleting = this._fineUploader.getUploads({
            status: [qq.status.DELETING]
        }).length;
        done = this._fineUploader.getUploads({
            status: [qq.status.UPLOAD_SUCCESSFUL, qq.status.DELETE_FAILED]
        }).length;
        failed = this._fineUploader.getUploads({
            status: [qq.status.UPLOAD_FAILED]
        }).length;
        cached = Object.keys(this._cachedFileIds).length;
        finished = done + failed + cached;

        this._log("evaluateStatus()  submitting:", submitting, ", active:", active, ", deleting:", deleting, ", done:", done, ", failed:", failed, ", cached:", cached);

        if (active === 0) {
            if (finished === 0) {
                this._setStatus(this._status.INIT);
            }
            else {
                this._setStatus(this._status.DONE);
            }
        }
        else {
            this._setStatus(this._status.ACTIVE);
        }
        this._updateUI(submitting, active, deleting, failed, finished);
    },

    _setStatus: function (newStatus) {
        var oldStatus = this.status;
        var newStatusProp = newStatus.toUpperCase();

        this.status = this._status.hasOwnProperty(newStatusProp) ? this._status[newStatusProp] : this.status;
        this._log("setStatus()  oldStatus:", oldStatus, ", newStatus:", this.status);

        switch (this.status) {
            case this._status.ACTIVE:
                if (oldStatus === this._status.INIT || oldStatus === this._status.DONE) {
                    this._trigger("onStart", null, this);
                }
                break;
            case this._status.INIT:
            case this._status.DONE:
                if (oldStatus === this._status.ACTIVE) {
                    this._trigger("onCompletion", null, this);
                }
                break;
        }
    },

    _updateUI: function (submitting, active, deleting, failed, finished) {
        switch (this.status) {
            case this._status.INIT:
                this._setDropAreaTextUI(this.options.multiple ? 'Drop files and folders here' : 'Drop a file here');
                this._hideItemUI('.qq-drop-processing');
                if (this.options.multiple) {
                    this._hideItemUI('.qq-total-progress-bar');
                    this._hideItemUI('.qq-upload-cancel-all');
                    this._hideItemUI('.qq-upload-delete-all');
                }
                else {
                    this.enableUploadUI();
                }
                break;
            case this._status.ACTIVE:
                if (submitting > 0) {
                    this._showItemUI('.qq-drop-processing');
                }
                else {
                    this._hideItemUI('.qq-drop-processing');
                }
                if (this.options.multiple) {
                    this._setDropAreaTextUI('');
                    // assert: finished >= failed
                    if (finished === 0 || finished === failed) {
                        // no uploaded files to delete OR there are only failed uploads
                        this._hideItemUI('.qq-upload-delete-all');
                    }
                    else {
                        this._showItemUI('.qq-upload-delete-all');
                    }
                    // assert: active >= submitting && active >= deleting
                    if (active === submitting || active === deleting) {
                        // ACTIVE because is ONLY calculating hash or deleting files
                        this._hideItemUI('.qq-total-progress-bar');
                        this._hideItemUI('.qq-upload-cancel-all');
                    }
                    else {
                        // some upload in progress
                        this._showItemUI('.qq-total-progress-bar');
                        this._showItemUI('.qq-upload-cancel-all');
                    }
                }
                else {
                    this.disableUploadUI();
                }
                break;
            case this._status.DONE:
                this._setDropAreaTextUI('');
                this._hideItemUI('.qq-drop-processing');
                if (this.options.multiple) {
                    this._hideItemUI('.qq-total-progress-bar');
                    this._hideItemUI('.qq-upload-cancel-all');
                    // assert: finished > 0 && finished >= failed
                    if (finished === failed) {
                        // there are only failed uploads, that cannot be deleted
                        this._hideItemUI('.qq-upload-delete-all');
                    }
                    else {
                        this._showItemUI('.qq-upload-delete-all');
                    }
                }
                else {
                    this.disableUploadUI();
                }
                break;
        }
    },

    _showItemUI: function (selector) {
        jQuery(selector, '#' + this.options.fineUploaderElementId).removeClass('qq-hide');
    },

    _hideItemUI: function (selector) {
        jQuery(selector, '#' + this.options.fineUploaderElementId).addClass('qq-hide');
    },

    _setDropAreaTextUI: function (text) {
        jQuery('.qq-uploader', '#' + this.options.fineUploaderElementId).attr('qq-drop-area-text', text);
    },

    disableUploadUI: function () {
        // disable Select Files buttons
        jQuery('.qq-upload-button input', '#' + this.options.fineUploaderElementId).prop('disabled', true);
        jQuery('.qq-upload-button', '#' + this.options.fineUploaderElementId).css('opacity', '0.5');
        // disable Drag & Drop
        jQuery('.qq-upload-drop-area', '#' + this.options.fineUploaderElementId).css('visibility', 'hidden');
        this._setDropAreaTextUI('');
    },

    enableUploadUI: function () {
        // enable Select Files button
        jQuery('.qq-upload-button input', '#' + this.options.fineUploaderElementId).prop("disabled", false);
        jQuery('.qq-upload-button', '#' + this.options.fineUploaderElementId).css('opacity', '');
        // enable Drag & Drop
        jQuery('.qq-upload-drop-area', '#' + this.options.fineUploaderElementId).css('visibility', '');
        this._setDropAreaTextUI(this.options.multiple ? 'Drop files and folders here' : 'Drop a file here');
    },

    _addCachedItemUI: function (fileId, fileName) {
        var encName = efEncodeHtml(fileName);
        // Note: we need to add \n to align presentation with the XSL template
        jQuery('.qq-upload-list-selector.qq-upload-list', '#' + this.options.fineUploaderElementId).append(
            '<li class="ef-fileupload-cached qq-file-id-' + fileId + ' qq-upload-success" qq-file-id="' + fileId + '">' +
                '<div class="qq-upload-file-selector qq-upload-file" title="' + encName + '">' +
                    encName +
                '</div>\n' +
                '<div class="qq-upload-size-selector qq-upload-size">' +
                    this._formatSize(this._fineUploader.getSize(fileId)) +
                '</div>\n' +
                '<button class="qq-btn qq-upload-delete-selector qq-upload-delete" type="button">Delete</button>' +
                '<span class="ef-fileupload-cached-label">Cached</span>' +
            '</li>');
    },

    _addFilePathUI: function (id) {
        var file, filePath, fileElem;

        file = this._fineUploader.getFile(id);
        filePath = this._getFilePath(file);
        fileElem = jQuery('.qq-file-id-' + id + ' .qq-upload-file', '#' + this.options.fineUploaderElementId);
        fileElem.text(filePath);
        fileElem.prop('title', filePath);
    },

    _getFilePath: function (file) {
        var filePath = "";

        if (file.qqPath) {
            filePath = file.qqPath + file.name;
        }
        else if (file.webkitRelativePath) {
            filePath = file.webkitRelativePath;
        }
        else {
            filePath = file.name;
        }
        return filePath;
    },

    _createFineUploader: function () {
        var efUploader, fineUploaderElementId;

        efUploader = this;
        fineUploaderElementId = this.options.fineUploaderElementId;

        this._log("Init FineUploader on element:", fineUploaderElementId, " with template:", this.options.fineUploaderTemplate);

        return new qq.FineUploader({
            debug: this.options.debug,
            autoUpload: this.options.autoUpload,
            element: document.getElementById(fineUploaderElementId),
            template: this.options.fineUploaderTemplate,
            multiple: this.options.multiple,
            maxConnections: (this.options.maxConnections ? this.options.maxConnections : 3),
            deleteFile: {
                enabled: true,
                endpoint: "/" + this.options.efRootContext + "/uc/delete",
                method: "POST"
            },
            request: {
                endpoint: "/" + this.options.efRootContext + "/ut",
                paramsInBody: false
            },
            callbacks: {
                onSubmit: function (id, name) {
                    var i, transaction, hash,
                        promise = new qq.Promise(),
                        fineUploader = this,
                        file = fineUploader.getFile(id),
                        filePath = efUploader._getFilePath(file);
                    efUploader._log("onSubmit(", id, ",", name, ",", file, ")");

                    // stop folder upload in SFU
                    if (!efUploader._validateFolderUploads(file)) {
                        return false;
                    }

                    // check already processed files here because file id is not available in the onValidate callback
                    if (!efUploader._validatePreviousUploads(file)) {
                        return false;
                    }

                    // validate mime type here because file type is not available in the onValidate callback
                    if (!efUploader._validateAcceptAttribute(file)) {
                        return false;
                    }

                    if (efUploader._uploadProtocol.session.caching === "hash") {
                        hash = efUploader._computeHash(file, function (hash) {
                            transaction = efUploader._uploadProtocol.ucTransaction(filePath, hash);
                            efUploader._tidFileMap[id] = transaction.tid;
                            if (transaction.cached) {
                                efUploader._cachedFileIds[id] = true;
                                efUploader._addCachedItemUI(id, name);
                                efUploader._log("File (", id, ") is cached.");
                                promise.failure();
                                efUploader._uploadProtocol.ucFinish(transaction.tid);
                            } else {
                                promise.success();
                            }
                        }, function (error) {
                            this._err("Computing hash for file (" + file + "): ", error);
                            // Upload the file anyway
                            promise.success();
                        });
                        return promise;
                    } else {
                        // No cache so no hash to compute
                        transaction = efUploader._uploadProtocol.ucTransaction(filePath);
                        efUploader._tidFileMap[id] = transaction.tid;
                        return true;
                    }
                },
                onSubmitted: function (id, name) {
                    efUploader._log("onSubmitted(" + id + ", " + name + ")");
                },
                onUpload: function (id, name) {
                    efUploader._log("onUpload(" + id + ", " + name + ")");
                    var tid = efUploader._tidFileMap[id];
                    this.setParams({"sid": efUploader._uploadProtocol.session.sid, "tid": tid}, id);
                },
                onStatusChange: function (id, oldStatus, newStatus) {
                    efUploader._log("onStatusChange(", id, ", ", oldStatus, ", ", newStatus, ")");
                    if (newStatus === qq.status.DELETED && id in efUploader._cachedFileIds) {
                        delete efUploader._cachedFileIds[id];
                    }
                    // replace onComplete - finish the transaction
                    if (newStatus === qq.status.UPLOAD_SUCCESSFUL || newStatus === qq.status.UPLOAD_FAILED) {
                        efUploader._uploadProtocol.ucFinish(efUploader._tidFileMap[id]);
                    }
                    efUploader._evaluateStatus();
                },
                onSubmitDelete: function (id) {
                    efUploader._log("onDelete(" + id + ")");
                    var tid = efUploader._tidFileMap[id];
                    this.setDeleteFileParams({"sid": efUploader._uploadProtocol.session.sid, "tid": tid}, id);
                },
                onDeleteComplete: function (id, xhr, isError) {
                    efUploader._log("onDeleteComplete(" + id + ", " + xhr + ", " + isError + ")");
                    return true;
                },
                onError: function (id, name, errorReason, xhrOrXdr) {
                    efUploader._err("onError(" + id + ", " + name + ") reason:", errorReason);
                    // TODO: manage error
                    // alert(qq.format("Error on file number {} - {}.  Reason: {}", id, name, errorReason));
                },
                onCancel: function (id, name) {
                    efUploader._log("onCancel(" + id + ", " + name + ")");
                    return true;
                },
                onComplete: function (id, name, resJson, xhrOrXdr) {
                    efUploader._log("onComplete(", id, ", ", name, ", ", resJson, ", ", xhrOrXdr, ")");
                    efUploader._addFilePathUI(id);
                },
                // NOTE: this event is not triggered if all the files are cancelled (scenario: cache and no effective upload)
                onAllComplete: function (successArray, failedArray) {
                    efUploader._log("onAllComplete(" + successArray + ", " + failedArray + ")");
                },
                onValidateBatch: function (fileOrBlobDataArray, buttonContainer) {
                    efUploader._log("onValidateBatch(", fileOrBlobDataArray, ",", buttonContainer,")");
                },
                onValidate: function (data, buttonContainer) {
                    var i,
                        isValid = true;

                    efUploader._log("onValidate(", data, ",", buttonContainer, ")");
                    efUploader._log("onValidate()  validators:",
                            " _acceptFiles(", efUploader._acceptFiles, ")",
                            " _allowedExtensions(", efUploader._allowedExtensions, ")",
                            " _nameValidators(", efUploader._nameValidators, ")",
                            " _mediaValidators(", efUploader._mediaValidators, ")");

                    // Check ef:file-filter
                    if (efUploader._nameValidators.length && !efUploader._mediaValidators.length) {
                        isValid = false;
                        for (i = 0; i < efUploader._nameValidators.length && !isValid; i += 1) {
                            if (efUploader._nameValidators[i].test(data.name)) {
                                isValid = true;
                            }
                        }
                        if (!isValid) {
                            alert(efEncodeHtml(data.name) + " is not valid.\n" +
                                    "File name must match one of the following glob pattern: " + efUploader._validatorMsg);
                        }
                    }
                    return isValid;
                }
            },
            validation: {
                allowEmpty: true,
                stopOnFirstInvalidFile: false,
                acceptFiles: this._acceptFiles,
                allowedExtensions: this._allowedExtensions
            }
        });
    },

    _validateAcceptAttribute: function (file) {
        var i,
            efUploader = this,
            isValid = true;

        // Check mime type and file extensions from accept attribute
        if (efUploader._mediaValidators.length) {
            isValid = false;
            for (i = 0; i < efUploader._nameValidators.length && !isValid; i += 1) {
                if (efUploader._nameValidators[i].test(file.name)) {
                    isValid = true;
                }
            }
            for (i = 0; i < efUploader._mediaValidators.length && !isValid; i += 1) {
                if (efUploader._mediaValidators[i].test(file.type)) {
                    isValid = true;
                }
            }
            if (!isValid) {
                alert(efEncodeHtml(file.name) + " is not valid.\n" +
                        "File name must match one of the following mime type or extensions: " + efUploader._validatorMsg);
                isValid = false;
            }
        }
        return isValid;
    },

    _validateFolderUploads: function (file) {
        var isValid = true,
            efUploader = this,
            filePath = efUploader._getFilePath(file);

        if (!efUploader.options.multiple && filePath.indexOf('/') !== -1) {
            alert("You cannot upload folders in a Single File Upload option.");
            isValid = false;
        }
        return isValid;
    },

    _validatePreviousUploads: function (file) {
        var i, submitting, uploads, uploadedFile, cachedFile,
            isValid = true,
            efUploader = this,
            filePath = efUploader._getFilePath(file);

        submitting = efUploader._fineUploader.getUploads({
            status: [ qq.status.SUBMITTING ]
        });
        uploads = efUploader._fineUploader.getUploads({
            // Discarded states: CANCELED, REJECTED, DELETED
            status: [ qq.status.SUBMITTED, qq.status.QUEUED, qq.status.PAUSED,
                      qq.status.UPLOADING, qq.status.UPLOAD_SUCCESSFUL, /*qq.status.UPLOAD_FAILED,*/
                      qq.status.UPLOAD_RETRYING, qq.status.DELETING, qq.status.DELETE_FAILED ]
        });

        // Check if a file has been already processed with SFU
        // NOTE: Current files are in submitting state
        if (!efUploader.options.multiple &&
                (uploads.length > 0 || Object.keys(efUploader._cachedFileIds).length > 0 || submitting.length > 1)) {
            alert("You cannot upload multiple files in a Single File Upload option.");
            isValid = false;
        }

        for (i = 0; i < uploads.length && isValid; i++) {
            uploadedFile = efUploader._fineUploader.getFile(uploads[i].id);
            if (efUploader._getFilePath(uploadedFile) === filePath) {
                alert(filePath + " already processed.");
                isValid = false;
            }
        }
        if (isValid) {
            for (cachedFileId in efUploader._cachedFileIds) {
                cachedFile = efUploader._fineUploader.getFile(cachedFileId);
                if (efUploader._getFilePath(cachedFile) === filePath) {
                    alert(filePath + " already processed.");
                    isValid = false;
                    return false;
                }
            }
        }
        return isValid;
    },

    returnUploadedFiles: function() {
	var efUploader = this;
        var uploads = efUploader._fineUploader.getUploads({
            status: [qq.status.UPLOAD_SUCCESSFUL, qq.status.DELETE_FAILED]
        });
        for (i = 0; i < uploads.length; i++) {
            var uploadItem = uploads[i];
	    var filePath = efUploader._getFilePath(efUploader._fineUploader.getFile(uploadItem.id))
	    uploads[i].filePath = filePath;
	    uploads[i].spooler = efUploader._uploadProtocol.session.spooler;
        }
	// clear list of files // https://docs.fineuploader.com/branch/master/api/methods.html#getUploads
	efUploader._fineUploader.clearStoredFiles();
	// for (cachedFileId in efUploader._cachedFileIds) {
        //     cachedFile = efUploader._fineUploader.getFile(cachedFileId);
        //     var filePath = efUploader._getFilePath(cachedFile);
	//     console.log(filePath);
	//     // filePathArr.push(filePath);
        // }
	return uploads;
    },
    
    _formatSize: function (bytes) {
        var i = -1;
        do {
            bytes = bytes / 1e3;
            i++;
        } while (bytes > 999);
        return Math.max(bytes, 0.1).toFixed(1) + this._fineUploader._options.text.sizeSymbols[i];
    },

    _getHasher: function () {
        var hasher = this._hashers.pop();
        if (hasher === undefined) {
            hasher = new SHA1({ heapSize: 0x10000 });
        }
        return hasher;
    },

    _saveHasher: function (hasher) {
        this._hashers.push(hasher.reset());
    },

    // Compute Hash function
    _computeHash: function(file, hashCallback, errorCallback) {
        var startTime;
        var hashAlgo; //  = new SHA1({ heapSize: 0x10000 });
        var hash;
        var efUploader = this;

        var // blobSlice = File.prototype.slice || File.prototype.mozSlice || File.prototype.webkitSlice,
            fileSize = file.size,
            chunkSize = 2097152,  // 1024 * 1024 * 2 - read in chunks of 2MB
            offset = 0,
            endByte,
            fileReader = new FileReader();

        function readNextChunk() {
            var endFromOffset = offset + chunkSize,
                endByte = endFromOffset >= fileSize ? fileSize : endFromOffset;
            //fileReader.readAsArrayBuffer(blobSlice.call(file, offset, endByte));
            fileReader.readAsArrayBuffer(file.slice(offset, endByte));
        }

        fileReader.onloadend = function (e) {
            var arrayBuffer = e.target.result;
            // Get the hasher on the callback let the system to reuse hashers more easily and frequently.
            if (hashAlgo === undefined) hashAlgo = efUploader._getHasher();
            hashAlgo.update(arrayBuffer);

            offset += arrayBuffer.byteLength;
            if (offset < fileSize) {
                readNextChunk();
            } else {
                hash = hashAlgo.finish();
                efUploader._saveHasher(hashAlgo);

                efUploader._log("Hash (sha1asm) - Time:", (new Date().getTime() - startTime), "(ms)", " file:", file.name, " size:", fileSize, " hash:", hash);

                if (!e.target.error) {
                    hashCallback(hash);
                } else if (typeof errorCallback === "function" ) {
                    errorCallback(e.target.error);
                }
            }
        };

        fileReader.onerror = function () {
        };

        startTime = new Date().getTime();
        readNextChunk();
    },

    _computeHashParse: function(file, hashCallback, errorCallback) {
        var hashAlgo, uint8Array;
        var efUploader = this;
        var startTime = new Date().getTime();

        function parseFile(file, chunkCallback, endCallback) {
            var // blobSlice = File.prototype.slice || File.prototype.mozSlice || File.prototype.webkitSlice,
                fileSize = file.size,
                chunkSize = 2097152,  // 1024 * 1024 * 2 - read in chunks of 2MB
                offset = 0,
                endByte,
                fileReader = new FileReader();

            function readNextChunk() {
                var endFromOffset = offset + chunkSize,
                    endByte = endFromOffset >= fileSize ? fileSize : endFromOffset;
                //fileReader.readAsArrayBuffer(blobSlice.call(file, offset, endByte));
                fileReader.readAsArrayBuffer(file.slice(offset, endByte));
            }

            fileReader.onloadend = function (e) {
                var arrayBuffer = e.target.result;

                chunkCallback(arrayBuffer);

                offset += arrayBuffer.byteLength;
                if (offset < fileSize) {
                    readNextChunk();
                } else {
                    endCallback(e.target.error);
                }
            };

            fileReader.onerror = function () {
                // running = false;
                // registerLog('<strong>Oops, something went wrong.</strong>', 'error');
            };

            readNextChunk();
        }

        parseFile(file,
            function (arrayBuffer) {
                if (hashAlgo === undefined) hashAlgo = efUploader._getHasher();
                hashAlgo.update(arrayBuffer);
            },
            function (err) {
                if (! err) {
                    var hash = hashAlgo.finish();
                    efUploader._saveHasher(hashAlgo);

                    efUploader._log("Hash (sha1asm) - Time:", (new Date().getTime() - startTime), "(ms)", " file:", file.name, " size:", file.size, " hash:", hash);

                    hashCallback(hash);
                } else if (typeof errorCallback === "function" ) {
                    errorCallback(err);
                }
            });
    },

    _initValidators: function () {
        var i, j, acceptValues, filters;

        this._hasComplexGlobFilters = false;
        this._acceptFiles = '';
        this._allowedExtensions = [];
        this._nameValidators = [];
        this._mediaValidators = [];
        this._validatorMsg = '';
        // check accept attribute and ef:file-filters to initialize acceptFiles, allowedExtensions and validator variables
        if (this.options.acceptAttribute) {
            this._acceptFiles = this.options.acceptAttribute;
            if (this.options.acceptAttribute.indexOf('/') === -1) {
                // no media type in the accept attribute
                this._allowedExtensions = this.options.acceptAttribute.replace(/\./g, '').split(',');
            }
            else {
                // media type in the accept attribute
                acceptValues = this.options.acceptAttribute.split(',');
                for (i = 0; i < acceptValues.length; i += 1) {
                    acceptValues[i] = acceptValues[i].replace(/[\/\+\.]/g, '\\$&').replace(/\*/g, '.*');
                    if (acceptValues[i].indexOf('/') === -1) {
                        this._nameValidators.push(new RegExp(acceptValues[i] + '$'));
                    }
                    else {
                        this._mediaValidators.push(new RegExp(acceptValues[i]));
                    }
                    this._validatorMsg += acceptValues[i] + " ";
                }
            }
        } else {
            // detect if mfu widget has complex glob file filters
            for (i = 0; i < this.options.fileFilters.length && !this._hasComplexGlobFilters; i += 1) {
                if (!/(^|\s)\*\.[a-z0-9]+$/gi.test(this.options.fileFilters[i])) {
                    this._hasComplexGlobFilters = true;
                }
            }

            if (!this._hasComplexGlobFilters) {
                // no complex glob patterns in the file filters, just simple file extensions pattern
                acceptValues = [];
                for (i = 0; i < this.options.fileFilters.length; i += 1) {
                    filters = this.options.fileFilters[i].match(/\.[a-z0-9]+/gi);
                    if (filters) {
                        if (filters.length) {
                            acceptValues = acceptValues.concat(filters);
                        } else {
                            acceptValues.push(filters);
                        }
                    }

                    filters = this.options.fileFilters[i].match(/[a-z0-9]+/gi);
                    if (filters) {
                        if (filters.length) {
                            this._allowedExtensions = this._allowedExtensions.concat(filters);
                        } else {
                            this._allowedExtensions.push(filters);
                        }
                    }
                }
                this._acceptFiles = acceptValues.join(',');
            } else {
                // convert glob to regex
                for (i = 0; i < this.options.fileFilters.length; i += 1) {
                    filters = this.options.fileFilters[i].split(/[\s]+/);
                    for (j = 0; j < filters.length; j += 1) {
                        this._nameValidators.push(this._globToRegex(filters[j]));
                        this._validatorMsg += filters[j] + " ";
                    }
                }
            }
        }
    },

    _globToRegex: function (gPat) {
        var rPat, i, inBrackets;

        rPat = "";
        inBrackets = false;
        for (i = 0; i < gPat.length; i += 1) {
            switch (gPat[i]) {
                case '*':
                    if (!inBrackets) {
                        rPat += '.';
                    }
                    rPat += '*';
                    break;
                case '?':
                    rPat += inBrackets ? '?' : '.';
                    break;
                case '[':
                    inBrackets = true;
                    rPat += gPat[i];
                    if (i < gPat.length - 1) {
                        switch (gPat[i + 1]) {
                            case '!':
                            case '^':
                                rPat += '^';
                                i += 1;
                                break;
                            case ']':
                                rPat += gPat[++i];
                                break;
                        }
                    }
                    break;
                case ']':
                    rPat += gPat[i];
                    inBrackets = false;
                    break;
                case '\\':
                    if (i === 0 && gPat.length > 1 && gPat[1] === '~') {
                        rPat += gPat[++i];
                    } else {
                        rPat += '\\';
                        if (i < gPat.length - 1 && "*?[]".indexOf(gPat[i + 1]) >= 0) {
                            rPat += gPat[++i];
                        } else {
                            rPat += '\\';
                        }
                    }
                    break;
                default:
                    if (!/^[a-zA-Z0-9]+$/g.test(gPat[i])) {
                        rPat += '\\';
                    }
                    rPat += gPat[i];
                    break;
            }
        }

        return new RegExp(rPat);
    },

    _deleteAll: function () {
        var uploads, i, cachedFileId;

        uploads = this._fineUploader.getUploads({
            status: [qq.status.UPLOAD_SUCCESSFUL, qq.status.DELETE_FAILED]
        });
        for (i = 0; i < uploads.length; i++) {
            this._fineUploader.deleteFile(uploads[i].id);
        }
        for (cachedFileId in this._cachedFileIds) {
            this._fineUploader.deleteFile(cachedFileId);
        }
    },

    _createUploadProtocol: function () {
        var efUploader = this;

        return {

            options: {
                client: "fineuploader",
                efRootContext: "enginframe",
                onInitSuccess: function (session) {},
                onInitFailure: function () {},
                debug: false
            },

            session: {
                sid: "",
                protocol: "",
                spooler: "",
                caching: ""
            },

            _init: function (options) {
                jQuery.extend(this.options, options);
            },

            //FIXME: what about reading params from this.options if argument is null?
            ucInit: function (optionId, serviceUri, sdfUrl, windowId) {
                jQuery.ajax({
                    context: this,
                    url: "/" + this.options.efRootContext + "/uc/init",
                    async: false,
                    type: 'post',
                    dataType: 'xml',
                    data: {
                        EF_OPTION_ID: optionId,
                        EF_SERVICE_URI: serviceUri,
                        EF_SDF_URL: sdfUrl,
                        windowId: windowId,
                        client: this.options.client
                    },
                    success: function (data, textStatus, jqXHR) {

                        efUploader._log("ucInit() success\n  textStatus:", textStatus, " \n  data:", data);

                        var mfuResultXml = jQuery(data);
                        this.session.sid = mfuResultXml.find('mfu\\:sid, sid').text();
                        this.session.protocol = mfuResultXml.find("mfu\\:protocol, protocol").attr("type");
                        this.session.spooler = mfuResultXml.find('mfu\\:spooler, spooler').text();
                        this.session.caching = mfuResultXml.find('mfu\\:caching, caching').text();

                        efUploader._log("ucInit()\n  session:", this.session);

                        if (typeof this.options.onInitSuccess === "function") {
                            this.options.onInitSuccess(this.session);
                        }
                    },
                    error: function (jqXHR, textStatus, errorThrown) {
                        // TODO: manage errors
                        efUploader._err("ucInit() \n  textStatus:", textStatus, " \n  error:", errorThrown, " \n  jqXHR:", jqXHR);
                    }
                });
            },

            ucTransaction: function (name, hash) {
                var tid, transaction;

                jQuery.ajax({
                    url: "/" + this.options.efRootContext + "/uc/transaction",
                    async: false,
                    type: 'post',
                    dataType: 'xml',
                    data: {
                        sid: this.session.sid,
                        name: name,
                        hash: hash
                    },
                    success: function (data, textStatus) {
                        efUploader._log("ucTransaction()  file:", name, " success\n  textStatus:", textStatus, " \n  data:", data);

                        var mfuResultXml = jQuery(data);
                        transaction = mfuResultXml.find('mfu\\:result, result').attr("transaction");
                        tid = mfuResultXml.find("mfu\\:tid, tid").text();

                        efUploader._log("ucTransaction()  file:", name, " transaction:",transaction, " tid:", tid);
                    },
                    error: function (jqXHR, textStatus, errorThrown) {
                        // TODO: manage errors
                        efUploader._err("ucTransaction() \n  textStatus:", textStatus, " \n  error:", errorThrown, " \n  jqXHR:", jqXHR);
                    }
                });
                return { tid: tid, cached: transaction === "cached" };
            },

            ucFinish: function (tid) {
                jQuery.ajax({
                    url: "/" + this.options.efRootContext + "/uc/finish",
                    async: false,
                    type: 'post',
                    dataType: 'xml',
                    data: {
                        sid: this.session.sid,
                        tid: tid
                    },
                    success: function (data, textStatus) {
                        efUploader._log("ucFinish()  tid:", tid, " success\n  textStatus:", textStatus, " \n  data:", data);

                        var mfuResultXml = jQuery(data),
                            finish = mfuResultXml.find('mfu\\:result, result').attr("finish");

                        efUploader._log("ucFinish()  tid:", tid, " finish:", finish);
                    },
                    error: function (jqXHR, textStatus, errorThrown) {
                        // TODO: manage errors
                        efUploader._err("ucFinish() \n  textStatus:", textStatus, " \n  error:", errorThrown, " \n  jqXHR:", jqXHR);
                    }
                });
            },

            ucClose: function (cleanup) {
                var self = this;
                jQuery.ajax({
                    url: "/" + this.options.efRootContext + "/uc/close",
                    async: false,
                    type: "post",
                    // dataType: 'xml',
                    data: {
                        sid: this.session.sid,
                        cleanup: cleanup
                    },
                    success: function (data, textStatus) {
                        efUploader._log("ucClose()  sid:", self.session.sid, " success\n  textStatus:", textStatus, " \n  data:", data);
                    },
                    error: function (jqXHR, textStatus, errorThrown) {
                        // TODO: manage errors
                        efUploader._err("ucClose() \n  textStatus:", textStatus, " \n  error:", errorThrown, " \n  jqXHR:", jqXHR);
                    }
                });
            }
        };
    }
});

var efFileupload = {

    form : '',
    isSubmitted : false,
    forceSubmit : false,

    activeUploads : {},

    uploadAndSubmit: function (form) {
        var firstSubmit = false;

        if (!efFileupload.isSubmitted) {
            efFileupload.isSubmitted = true;
            efFileupload.form = form;
            firstSubmit = true;
        }

        if (efFileupload.forceSubmit) {
            return true;
        }

        if (firstSubmit) {
            // disable uploads for all the fileupload widget
            jQuery(".ef-fileupload").each(function (index, element) {
                jQuery(this).fileupload("disableUploadUI");
                // jQuery(this).fileupload("rejectNewFiles");
            });
        }

        // If there are active fileupload widgets
        if (Object.keys(efFileupload.activeUploads).length > 0) {
            return false;
        }

        // Close all the fileupload widget
        jQuery(".ef-fileupload").each(function (index, element) {
            jQuery(this).fileupload("close");
        });

        let submitButton = jQuery("#" + form.id).find(':submit');

        submitButton.addClass('ui-state-disabled');
        submitButton.after('<i id=' + form.id + '_submit_spinner" class="fa fa-spinner fa-spin" style="margin:10px"></i><span>Loading</span>');

        return true;
    },

    activation : function (event, fileupload) {
        efFileupload.activeUploads[fileupload.options.optionId] = fileupload;
    },

    completion: function (event, fileupload) {
        if ((fileupload.status === "done" || fileupload.status === "init") && efFileupload.activeUploads[fileupload.options.optionId] !== undefined) {
            delete efFileupload.activeUploads[fileupload.options.optionId];

            if (efFileupload.isSubmitted) {
                if (typeof jQuery === "function") {
                    // Temporarily disable the "double submission prevention" mechanism
                    // See efPreventDoubleSubmission in com.enginframe.system.js
                    jQuery(efFileupload.form).data('EFLastSubmitTime', 0);
                }
                // targetActionId is set globally by com.enginframe.system.js
                document.getElementById(targetActionId).click();
            }
        } else {
            console.error("Weird condition for fileupload (" + fileupload.options.optionId + "), " +
                "status (" + fileupload.status + "), " +
                "activeUploads contains this fileupload: " + (efFileupload.activeUploads[fileupload.options.optionId] !== undefined));
        }
    }
};
// ex:ts=4:et:ai:ruler:
