#!/bin/bash

get_java_version () {
  if [ -n "${JAVA_HOME}" ]; then
    JAVA_VERSION=$("${JAVA_HOME}/bin/java" -version 2>&1)
  else
    JAVA_VERSION=$(java -version 2>&1)
  fi
  echo "${JAVA_VERSION}"
}

get_property_from_conf() {
  local _property_name="$1"
  local _config_file="$2"
  awk -f "${EF_ROOT}/plugins/admin/bin/get_property.awk" -v name="${_property_name}" \
    "$EF_ROOT/conf/${_config_file}" \
    "$EF_CONF_ROOT/enginframe/${_config_file}"
}

printtitle "EnginFrame self checks"
printsubtitle "Starting at `date`"

startsection "Checking EnginFrame plugins structure:"
# ---------------------------------------------------------

printsolovoice "directory tree:"

printsubvoice "${EF_ROOT}/plugins:"
[ -x "${EF_ROOT}/plugins" ] \
  && printok "Ok" \
  || printerror "Not accessible: ${EF_ROOT}/plugins"

for i in `ls ${EF_ROOT}/plugins`; do

  printsubvoice "${EF_ROOT}/plugins/${i}:"
  [ -x "${EF_ROOT}/plugins/${i}" ] \
    && printok "Ok" \
    || printerror "Not accessible: ${EF_ROOT}/plugins/${i}"

  if [ -d "${EF_ROOT}/plugins/${i}/bin" ]; then
    printsubvoice "${EF_ROOT}/plugins/${i}/bin:"
    [ -x "${EF_ROOT}/plugins/${i}/bin" ] \
      && printok "Ok" \
      || printerror "Not accessible: ${EF_ROOT}/plugins/${i}/bin"
  fi

  if [ -d "${EF_ROOT}/plugins/${i}/conf" ]; then
    printsubvoice "${EF_ROOT}/plugins/${i}/conf:"
    [ -x "${EF_ROOT}/plugins/${i}/conf" ] \
      && printok "Ok" \
      || printerror "Not accessible: ${EF_ROOT}/plugins/${i}/conf"
  fi

  if [ -d "${EF_CONF_ROOT}/plugins/${i}" ]; then
    printsubvoice "${EF_CONF_ROOT}/plugins/${i}:"
    [ -x "${EF_CONF_ROOT}/plugins/${i}" ] \
      && printok "Ok" \
      || printerror "Not accessible: ${EF_CONF_ROOT}/plugins/${i}"
  fi

done

closesection

#
# EnginFrame configuration
#

startsection "EnginFrame Configuration"
# ---------------------------------------------------------

printsolovoice "Configuration files:"

printsubvoice "EnginFrame root configuration:"

if [ -r "${EF_CONF_ROOT}/enginframe.conf" ]; then
  printok "Ok"
else
  printwarning "Not readable: ${EF_CONF_ROOT}/enginframe.conf"
fi

printsubvoice "EnginFrame agent configuration:"
if [ -r "${EF_CONF_ROOT}/enginframe/agent.conf" ]; then
  printok "Ok"
else
  printwarning "Not readable: ${EF_CONF_ROOT}/enginframe/agent.conf"
fi

printsubvoice "EnginFrame server configuration:"
if [ -r "${EF_CONF_ROOT}/enginframe/server.conf" ]; then
  printok "Ok"
else
  printwarning "Not readable: ${EF_CONF_ROOT}/enginframe/server.conf"
fi

printsolovoice "Checking server.conf: "

printsubvoice "Download Manager:"

_download_manager="$(get_property_from_conf ef.download.manager server.conf)"

case "${_download_manager}" in
    "")
      printok "Download manager is empty (fallback to REMOTE)"
      ;;
    [Ll][Oo][Cc][Aa][Ll])
      printok "Download manager is set to LOCAL"
      ;;
    [Rr][Ee][Mm][Oo][Tt][Ee])
      printok "Download manager is set to REMOTE"
      ;;
    *)
      printwarning "Download manager is set to invalid value \"${_download_manager}\""
esac


printsubvoice "Default Authority:"
_default_authority="$(get_property_from_conf EF_DEFAULT_AUTHORITY server.conf)"
case "${_default_authority}" in
    "")
      printwarning "Default Authority is empty"
      ;;
      pam|os|ldap|activedirectory|http|certificate)
      printok "${_default_authority}"
      ;;
    *)
      if [ -x "${EF_ROOT}/plugins/${_default_authority}/bin/ef.auth" ] ; then
        if [ -r "${EF_ROOT}/plugins/${_default_authority}/etc/${_default_authority}.login" ] ; then
          printok "Custom Authority specified: \"${_default_authority}\""
        else
          printwarning "${_default_authority}.login for \"${_default_authority}\" is not readable"
        fi
      else
        printwarning "ef.auth script for \"${_default_authority}\" is not executable"
      fi
      ;;
esac

printsubvoice "EF_ADMIN:"
_ef_admin="$(get_property_from_conf EF_ADMIN server.conf)"
printok "${_ef_admin}"

closesection

#
# Changes to system files
#

pushd "${EF_ROOT}/.." > /dev/null
_current_version="$(basename $(pwd))"
_versions_root="$(dirname $(pwd))"
_reference_report="${_versions_root}/install/${_current_version}/report.txt"
_directory_to_check="$(pwd)"
popd >/dev/null
_changes=$(mktemp /tmp/EF_REPORT.$(date +"%Y%d%m-%H%M%S").XXXXXX)

startsection "System files"
printvoice "Checking report \"${_reference_report}\""
if [ ! -r "${_reference_report}" ]; then
    printerror "Missing report!"
else
    "${EF_ROOT}/tools/reportdir.sh" check "${_directory_to_check}" "${_reference_report}" "${_changes}" > /dev/null
    _exit_code=$?
    if [ "$_exit_code" = "0" ]; then
        printok "Ok, no changes detected"
    else
      printwarning "Changes detected"
      cat "${_changes}" | format_changes
    fi
fi


closesection

#
# Java version
#
startsection "Java"
# ---------------------------------------------------------
if [ -n "${JAVA_HOME}" ]; then
  printsolovoice "JAVA_HOME points to: ${JAVA_HOME}"
else
  printsolowarning "JAVA_HOME variable not found"
fi

begincommandoutput
get_java_version
endcommandoutput

closesection

startsection "EnginFrame server info"
# ---------------------------------------------------------
begincommandoutput
"$EF_ROOT/plugins/admin/bin/webserverinfo.sh" | commandescape
endcommandoutput
closesection

startsection "User environment:"
# ---------------------------------------------------------
begincommandoutput
env | commandescape
endcommandoutput

closesection

printsubtitle "Ending at `date`"
