/*global jQuery */
/*
 * Copyright 2023-2025 by NI SP Software GmbH, All rights reserved.
 * Copyright 1999-2023 by Nice, srl., All rights reserved.
 *
 * This software includes confidential and proprietary information
 * of NI SP Software GmbH ("Confidential Information").
 * You shall not disclose such Confidential Information
 * and shall use it only in accordance with the terms of
 * the license agreement you entered into with NI SP Software. */

(function ($) {

    function isIOS() {
        return navigator.userAgent.match(/iPad|iPhone|iPod/i);
    }

    $.widget('ui.hyconnectionbar', {
        options: {
            conf: {},
            messageArea: null
        },

        _params: {
            SDF: '',
            maxWait: 20,
            clientDownloadUrl: ''
        },

        _connectionData: {
            params: {}
        },

        _create: function () {
            var self;

            self = this;

            $.extend(this._params, this.options.conf.params);

            this.element.addClass('hy-connectionbar');
            if (this.element.closest('.ui-widget').length <= 0) {
                this.element.addClass('ui-widget');
            }

            this.progressMsg = $('<div></div>').hymessage().appendTo(this.element);
        },

        _connectUri: function (newWindow) {
            var scheme, username, address, port, remote, remoteSessionId, password, authToken, uri, webUrlPath;

            scheme = this._connectionData.params.uriScheme;
            username = this._connectionData.params.username;
            address = this._connectionData.params.address;
            port = this._connectionData.params.port;
            remote = this._connectionData.params.sessionRemote;
            remoteSessionId = this._connectionData.params.remoteSessionId;

            if (scheme === undefined) {
                this.options.messageArea.hymessage('alert', 'Not supported.');
                return;
            }
            uri = scheme + '://';

            switch (remote) {
            case "dcv2":
            case "dcv2sm":
                password = this._connectionData.params.password;
                authToken = this._connectionData.params.authToken;
                webUrlPath = this._connectionData.params.webUrlPath;
                if (webUrlPath === undefined) {
                    webUrlPath = "/";
                    console.warn("Setting webUrlPath to / as it was undefined");
                }
                if (port === undefined) {
                    // FIXME; port can be undefined so we need a fallback
                    console.warn("Removing port which is undefined");
                    uri += address + webUrlPath;
                } else {
                    uri += address + ':' + port + webUrlPath;
                }
                if (authToken !== undefined) {
                    uri += '?authToken=' + authToken;
                } else if (username !== undefined  && password !== undefined) {
                    uri += '?username=' + username + '&password=' + password;
                }
                if (remoteSessionId !== undefined) {
                    uri += '#' + remoteSessionId;
                }
                break;
            default:
                password = "";
                break;
            }

            if (newWindow == null || typeof(newWindow) == 'undefined') {
                this.options.messageArea.hymessage('info', 'Session connection blocked by pop-up blocker. <a target="_blank" rel="noopener noreferrer" href="' + uri + '">Click here</a> to connect.', 0);
            } else {
                newWindow.location.href = uri;
                if (this._connectionData.params.uriScheme == "dcv") {
                    setTimeout(function () {
                        newWindow.close();
                    }, 10000) // adapt in case the DCV URI window closes too fast
                }
            }
        },

        _connectFileBasename: function () {
            var env, basename, os, remote, project, altname;
            env = this._connectionData.params;
            basename = null;
            // invoke the custom js function defined with <hy:set-param name="connectionFileBasename">
            if (typeof this._params.connectionFileBasename === 'string') {
                try {
                    basename = jQuery.hydrogen.executeFunctionByName(this._params.connectionFileBasename, window, env);
                } catch (ignore) {
                    // silently ignore any exception and use the default value defined below
                    // because we can't really foresee potential troubles of custom code
                }
            }
            // default value depends on EF_SPOOLER_NAME if it is set to something meaningful
            if (basename === null) {
                os = env.sessionOS || 'unknown_os';
                remote = env.sessionRemote || 'unknown_remote';
                project = env.sessionProject || 'session';
                altname = os + ' ' + remote + ' ' + project;
                basename = env.sessionName || altname;
                basename = basename.replace(/^tmp\d+\.ef$/, altname);
            }
            // replace Windows forbidden character for filenames (Linux is more flexible)
            basename = basename.replace(/[\\\/:*?"<>|]/g, '_');
            return basename;
        },

        _connectFile: function (name) {
            var loc, params, message, openNewTab;

            openNewTab = false;

            if (!this._connectionData.params.connectFileUri) {
                this.element.hide();
                this.options.messageArea.hymessage('alert', 'Unable to obtain the connection file.');
                return;
            }

            params = {
                _uri: this._connectionData.params.connectFileUri,
                basename: this._connectFileBasename()
            };

            $.each(this._connectionData.params, function (k, v) {
                params["remote_" + k] = v;
            });

            loc = window.location.pathname;

            $.download(loc, params, openNewTab);

            if (name) {
                message = '<div class="hy-connect-session-primary-message">Connecting to ' + efEncodeHtml(name) + '...</div>';
            } else {
                message = '<div class="hy-connect-session-primary-message">Connecting to your session...</div>';
            }
            message += '<div class="hy-connect-session-secondary-message">';
            message += 'Problems with the automatic connection? Please use this <a class="connect-file" href="#">direct link</a>.';
            if (this._params.clientDownloadUrl !== '') {
                message += ' To connect, you need a proper <a href="' + this._params.clientDownloadUrl + '">client</a> installed.';
            }
            message += '</div>';

            this.element.hide();
            this.options.messageArea.hymessage('info', message);
            $('a.connect-file', this.options.messageArea).click(function () {
                $.download(loc, params, openNewTab);
            });
        },

        _connect: function (sessionUri, name, remotePattern) {
            var viewerType, newWindow, remote, remoteTokens;
            var self = this;

            var remoteTokens = remotePattern.split("#");
            remote = remoteTokens[0];

            viewerType = self._params[remote + 'ViewerType'];
            var lUriScheme = "https"; // set default
            if (remoteTokens.length > 1) {
                switch (remoteTokens[1]) {
                case "hy_dcvuri":
                    lUriScheme = "dcv";
                    viewerType = "browser";
                    break;
                case "hy_dcvweb":
                    lUriScheme = "https";
                    viewerType = "browser";
                    break;
                case "hy_dcvfile":
                    lUriScheme = "";
                    viewerType = "file";
                    break;
                }
            } else if (viewerType === "dcvuri") {
                lUriScheme="dcv";
                viewerType="browser";
            }

            if (viewerType === 'browser') {
                // Do not open sessionUri directly...
                // newWindow = window.open('', sessionUri);
                // ...instead create a waiting window since we have more ways to connect now
                newWindow = window.open('', '_blank');
                if (newWindow) {
                    var content, dcvuriMessage = "";

                    if (lUriScheme === "dcv") {
                        dcvuriMessage = '<div class="smaller">This window will automatically close</div>';
                    }

                    content = '<head><style>' +
                        'body { color: #DDD; background-color:#0B2F40; }' +
                        '.loader { width: 100px; padding: 8px; aspect-ratio: 1; border-radius: 50%; background: #DDD;' +
                            '--_m: conic-gradient(#0000 10%,#000), linear-gradient(#000 0 0) content-box;' +
                            '-webkit-mask: var(--_m); mask: var(--_m);' +
                            '-webkit-mask-composite: source-out; mask-composite: subtract;' +
                            'animation: spinload 1s infinite linear;' +
                            'position: fixed; top: 40%; left: 50%; margin-left: -50px;' +
                        '}' +
                        '@keyframes spinload {to{transform: rotate(1turn)}}' +
                        '.message { position: fixed; width: 100%; top: 40%; margin-top: 130px; text-align: center; font: 1.7rem/1.7 "Verdana", sans-serif; }' +
                        '.message .smaller { font-size: 1.2rem; }' +
                    '</style></head>' +
                    '<body><div class="loader"></div><div class="message">Connecting' + dcvuriMessage + '</div></body>';
                    newWindow.document.write(content);
                }
            }

            var lViewerType = viewerType;
            $.hydrogen.invokeService({
                sdf: self._params.SDF,
                uri: '//ui.hydrogen/interactive.connection.data',
                data: {
                    sessionUri: sessionUri
                },
                success: function (xml) {
                    var viewer, callback, doc = $(xml);
                    if ($('connection', doc).length > 0) {
                        self._connectionData = {
                            params: {}
                        };
                        $('connection', doc).find('*').each(function (i, e) {
                            var el = $(e);
                            if (el.is("param")) {
                                self._connectionData.params[el.attr("name")] = el.text();
                            }
                        });

                        // Use saved viewer type instead of fetching from _connectionData
                        // since we get its value not only from general settings
                        // viewer = self._connectionData.params.viewer || 'client'; // get default from EF settings
                        viewer = lViewerType;
                        if (viewerType === "file") {
                            viewer = "file";
                        }

                        if (isIOS() || viewer === 'browser') {
                            self._connectionData.params.uriScheme=lUriScheme;
                            callback = function () {
                                self.element.hide();
                                self._connectUri(newWindow);
                            };
                        }
                        // Example for future support of different viewer type
                        /*
                        else if (viewer === 'embed') {
                            // E.g. embedded applet
                            callback = function () {
                                // replace the whole content rendering
                                var contentDiv = self.element.closest('.hy-content').hide();
                                self.appletDiv = $('<div></div>').insertAfter(contentDiv).append(self._getApplet());
                            };
                        }
                        */
                        else {
                            callback = function () {
                                self._connectFile(name);
                            };
                        }
                        setTimeout(callback, 250);
                    } else if ($('error', doc).length > 0) {
                        self._trigger('updated', null, {
                            sessionUri: sessionUri,
                            status: $('error', doc).attr('status'),
                            sessionRemote: remote
                        });
                        self.options.messageArea.hymessage('alert', $('error', doc).text());
                    }
                },
                messagebox: self.options.messageArea
            });
        },

        start: function (connectionParams) {
            var self, startWait;

            self = this;

            function checkStatus() {
                var maxWait, elapsed, percent;

                maxWait = self._params.maxWait * 1000; // count in ms
                elapsed = $.now() - startWait;
                if (elapsed > maxWait) {
                    self.element.hide();
                    self.options.messageArea.hymessage('info', 'Your session didn\'t start yet...');
                    self._trigger('expired');
                } else {
                    percent = Math.floor(elapsed * 100 / maxWait);
                    if (self.progress === undefined) {
                        self.progressMsg.hymessage('info', '<div class="hy-connect-session-primary-message">Your session will start soon, please wait...</div>');
                        self.progress = $('<div class="hy-connect-session-progressbar" ></div>').progressbar({ value: percent });
                    } else {
                        self.progress.progressbar('value', percent);
                    }

                    $.hydrogen.invokeService({
                        sdf: self._params.SDF,
                        uri: '//ui.hydrogen/interactive.session.status',
                        data: {
                            sessionUri: connectionParams.sessionUri
                        },
                        success: function (status) {
                            self._trigger('updated', null, {sessionUri: connectionParams.sessionUri, status: status, sessionRemote: connectionParams.sessionRemote});
                            if (status === 'Pending' || status === 'Starting') {
                                self.progressMsg.hymessage('info', $('<div>Status: <span class="hy-connect-session-status">' +
                                        status + '</span></div>').append(self.progress).html());
                                setTimeout(checkStatus, 1500);
                            } else if (status === 'Running') {
                                self._connect(connectionParams.sessionUri, connectionParams.sessionName, connectionParams.sessionRemote);
                            } else if (status === 'Failed' || status === 'Error') {
                                self.element.hide();
                                self.options.messageArea.hymessage('alert', 'The session failed to start.');
                            } else if (status === 'Closing' || status === 'Closed') {
                                self.element.hide();
                                self.options.messageArea.hymessage('info', 'The session was closed.');
                            }
                        },
                        error: function () {
                            self.element.hide();
                        },
                        messagebox: self.options.messageArea
                    });
                }
            }

            if (connectionParams.initialStatus === 'Running' || (connectionParams.initialStatus === 'Starting' && connectionParams.allowStarting)) {
                this._connect(connectionParams.sessionUri, connectionParams.sessionName, connectionParams.sessionRemote);
            } else if (connectionParams.initialStatus === 'Pending' || connectionParams.initialStatus === 'Starting') {
                startWait = $.now();
                checkStatus();
            } else {
                self.options.messageArea.hymessage('alert', 'Unable to connect to the session because it is ' + connectionParams.initialStatus + '.');
            }

            this._trigger('updated', null, {sessionUri: connectionParams.sessionUri, status: connectionParams.initialStatus, sessionRemote: connectionParams.sessionRemote});

            return this;
        },

        destroy: function () {
            this.element.removeClass('hy-connectionbar ui-widget');
            this.progressMsg.remove();

            $.Widget.prototype.destroy.apply(this, arguments);

            return this;
        }
    });

}(jQuery));

// ex:ts=4:et:
