/*global jQuery, efApplets.showAll, efApplets.hideAll */
/*
 * Copyright 2023-2025 by NI SP Software GmbH, All rights reserved.
 * Copyright 1999-2023 by Nice, srl., All rights reserved.
 *
 * This software includes confidential and proprietary information
 * of NI SP Software GmbH ("Confidential Information").
 * You shall not disclose such Confidential Information
 * and shall use it only in accordance with the terms of
 * the license agreement you entered into with NI SP Software. */

(function ($) {
    var rfbInstances = {};

    // Returns either the main path or the last selected path from the cookie.
    // If the cookie does not exist or no path was set, returns null.
    function getPathFromCookie(cookie) {
        var cookieContent, savedPaths;

        cookieContent = $.cookie(cookie);
        if (cookieContent) {
            try {
                savedPaths = JSON.parse(cookieContent);
                if (savedPaths && typeof savedPaths === 'object') {
                    if ('main' in savedPaths) {
                        return savedPaths.main;
                    } else if ('path' in savedPaths) {
                        return savedPaths.path;
                    }
                }
            } catch (_) { /* ignore */ }
        }

        return null;
    }

    $.extend($.hydrogen, {

        setRFBpath: function (id, path) {
            if (id in rfbInstances) {
                var params;
                params = rfbInstances[id].data();
                params.vroot.path = path;
                rfbInstances[id].data(params);
            } else {
                alert('invalid rfb instance: ' + id);
            }
        },

        initRFB: function (config) {
            var cookie, path, vroot, newPath;

            // Without config.id, the cookie will be shared among all RFBs in the service page.
            // This way we can get the latest visited folder from any RFB.
            // cookie = [config.serviceid, config.id, $.enginframe.user.name, 'last-dir'].join('-');
            cookie = [config.serviceid, $.enginframe.user.name, 'last-dir'].join('-');
            // Use base if defined; fallback to cookie; finally use HOME.
            path = config.base || $.cookie(cookie) || "${HOME}";

            vroot = {
                id: config.vrootId,
                label: '', // Not shown, but hyfilebrowser expects it
                path: path
            };

            rfbInstances[config.id] = $('<div class="hy-rfb"></div>').data({
                isActive: false,
                optionid: config.id,
                cookie: cookie,
                spooler: config.vrootSpooler,
                vroot: vroot,
                title: config.title,
                target: config.target,
                multiselect: config.multi === 'true' ? true : false,
                filter: config.filter,
                rowNum: config.rowNum,
                legacy: (config.plugin !== '' && config.target !== 'file' && config.target !== 'both' && config.target !== 'directory'),
                base: config.base,
                rfbClass: config['class']
            });

            $('#button_serviceopt_' + config.id + 'RFB').click(function () {
                var cookiePath;

                if (rfbInstances[config.id].data('isActive')) {
                    newPath = rfbInstances[config.id].hyrfb('prefillSelection');
                    cookie = rfbInstances[config.id].data('cookie');
                    if (!newPath) {
                        if (config.base) {
                            // use base path if configured
                            newPath = config.base;
                        } else {
                            // Get the path from the cookie.
                            // If it exists, no path was previously selected and no base path is configured, set as new path.
                            // Cookie is shared among all RFBs in the service.
                            cookiePath = getPathFromCookie(cookie);
                            if (cookiePath) {
                                newPath = cookiePath;
                            }
                        }
                    }

                    rfbInstances[config.id].dialog('open');
                    if (newPath != null) {
                        rfbInstances[config.id].hyrfb('changePath', newPath);
                    } else {
                        newPath = rfbInstances[config.id].data('vroot').path;
                        if (newPath != null) {
                            rfbInstances[config.id].hyrfb('changePath', newPath);
                        }
                    }
                } else {
                    rfbInstances[config.id].appendTo($('body')).hyrfb(rfbInstances[config.id].data());
                    rfbInstances[config.id].data('isActive', true);
                }
            });

            $('#button_clear_' + config.id + 'RFB').click(function () {
                $('#serviceopt_' + config.id).val('').change();
            });
        },

        dirname: function (path) {
            var res = path.replace(/\/[^\/]*$/, '');
            if (res === '') {
                res = '/';
            }
            return res;
        },

        basename: function (path) {
            var res = path.replace(/.*\//, '');
            if (res === '') {
                res = '/';
            }
            return res;
        }
    });

    $.widget('ui.hyrfb', {

        _create: function () {
            var self, colmodel, cookiePath, globalActions, firstLoad, newPath, vroot;

            self = this;
            newPath = null;

            // Note: selection spans across directories so
            // we need to keep a map of arrays
            this.selection = {
                d: {},
                clear: function () {
                    this.d = {};
                },
                add: function (dir, fnames) {
                    var arr, filtered;
                    arr = this.d[dir] ? this.d[dir].concat(fnames) : fnames;
                    // remove duplicates
                    filtered = $.grep(arr, function (e, i) {
                        return i === $.inArray(e, arr);
                    });
                    this.set(dir, filtered);
                },
                set: function (dir, fnames) {
                    this.d[dir] = fnames;
                },
                get: function (dir) {
                    return this.d[dir];
                },
                all: function () {
                    var s = [];
                    $.each(this.d, function (dir, a) {
                        $.each(a, function (i, fname) {
                            if (dir == "/") {
                                s.push(dir + (fname? fname : ''));
                            } else {
                                s.push(dir + (fname? ('/' + fname) : ''));
                            }
                        });
                    });
                    return s;
                }
            };

            vroot = this.options.vroot;
            newPath = this.prefillSelection();
            if (newPath != null) {
               vroot.path = newPath;
            }

           if (!newPath) {
                if (self.options.base) {
                    // use base path if configured
                    vroot.path = self.options.base;
                } else {
                    // Get the path from the cookie.
                    // If it exists, no path was previously selected and no base path is configured, set as new path.
                    // Cookie is shared among all RFBs in the service.
                    cookiePath = getPathFromCookie(self.options.cookie);
                    if (cookiePath) {
                        vroot.path = cookiePath;
                    }
                }
            }

            this.element.dialog({
                title: this.options.title,
                modal: true,
                classes: {
                    'ui-dialog': 'hy-rfb-dialog'
                },
                width: $(window).width() * 0.7,
                height: $(window).height() * 0.8,
                buttons: {
                    'Cancel': function () {
                        $(this).dialog('close');
                    },
                    'OK': function () {
                        var selectionAll;

                        selectionAll = self.selection.all();

                        // If we are selecting *only* dirs and *only* one, then return the selection will be empty,
                        // so we will return the current path as result.
                        if (self.options.target === 'directory' && !self.options.multiselect && selectionAll.length === 0) {
                            selectionAll = [self.hyfm.hyfilebrowser('currentPath')];
                        }

                        $('#serviceopt_' + self.options.optionid).val(selectionAll.join('\n')).change();

                        $(this).dialog('close');
                    }
                },
                resize: function () {
                    self._setGridSize();
                },
                open: function () {
                    if (typeof efApplets !== "undefined" && typeof efApplets.hideAll !== "undefined") {
                        efApplets.hideAll();
                    }
                },
                close: function () {
                    if (typeof efApplets !== "undefined" && typeof efApplets.showAll !== "undefined") {
                        efApplets.showAll();
                    }
                }
            });

            this.statusBar = $('<div class="hy-rfb-status">' +
                               '<span class="hy-rfb-status-label">Selected: </span><span class="hy-rfb-status-value"></span>' +
                               '</div>').appendTo($('.ui-dialog-buttonpane', this.element.parent()));

            this.msg = $('<div></div>').hymessage().appendTo(this.element);

            globalActions = [];
            if (self.options.target === 'directory' || self.options.target === 'both') {
                globalActions.push({
                    id: 'createFolder',
                    name: 'Create Folder',
                    action: function () {
                        return self.hyfm.hyfilebrowser('createNewFolder');
                    }
                });
                globalActions.push({
                    id: "sep1",
                    separator: true
                });
            }
            globalActions.push({
                id: 'refresh',
                name: 'Refresh',
                action: function () {
                    self.hyfm.hyfilebrowser('reload');
                }
            });
            this.toolbar = $('<div></div>').hytoolbar({
                conf: {
                    'global-action-items': globalActions
                },
                searchBox: true,
                searchBoxHint: 'Search by name',
                searchBoxTooltip: 'Glob expressions are allowed, e.g *.log'
            }).appendTo(this.element);

            this.toolbar.bind('hytoolbaraction', function (e, action) {
                action();
            });

            this.toolbar.bind('hytoolbarsearch', function (e, s) {
                var formula;

                formula = self.options.filter;

                // accept both "plain" string matches and glob expressions
                if (formula !== null && formula !== '') {
                    if (s.search !== '') {
                        formula += " and (name ?:= '" + s.search + "' or name *= '" + s.search + "')";
                    }
                } else {
                    if (s.search !== '') {
                        formula = "name ?:= '" + s.search + "' or name *= '" + s.search + "'";
                    } else {
                        formula = "";
                    }
                }

                self.hyfm.hyfilebrowser('filter', formula);
            });

            colmodel = [{
                hidden: true,
                name: 'selectable',
                xmlmap: function (row) {
                    if (self.options.target === 'directory') {
                        return $(row).attr('type') === 'folder' ? 'true' : 'false';
                    }
                    if (self.options.target !== 'both') {
                        return $(row).attr('type') === 'file' ? 'true' : 'false';
                    }
                    return 'true';
                }
            }, {
                index: 'name',
                name: 'name',
                xmlmap: 'name',
                label: 'Name',
                width: 10,
                sortable: true,
                resizable: false,
                formatter: $.hydrogen.fileNameFormatter,
                unformat: $.hydrogen.fileNameUnformatter,
                formatoptions: { linkFiles: false, disableFiles: self.options.target === 'directory' ? true : false }
            }];

            // Backward compat hack: show two string columns, first is the name, second
            // is the 'status' attribute, but it can contain anything, so we use an empty
            // title and a generic string formatter
            if (self.options.legacy) {
                colmodel.push({
                    index: 'status',
                    name: 'status',
                    xmlmap: 'status',
                    label: ' ',
                    width: 10,
                    sortable: true,
                    resizable: false
                });
            } else {
                colmodel.push({
                    index: 'size',
                    name: 'size',
                    xmlmap: 'size',
                    label: 'Size',
                    width: 4,
                    sortable: true,
                    resizable: false,
                    formatter: $.hydrogen.sizeFormatter
                });
                colmodel.push({
                    name: 'type',
                    xmlmap: '[type]',
                    hidden: true
                });
                colmodel.push({
                    index: 'modified',
                    name: 'modified',
                    label: 'Date Modified',
                    xmlmap: 'modified',
                    width: 4,
                    sortable: true,
                    resizable: false,
                    formatter: $.hydrogen.prettyDateFormatter,
                    formatoptions: {
                        showSeconds: false
                    }
                });
            }

            firstLoad = true;

            this.hyfm = $('<div></div>').appendTo(this.element);
            this.hyfm.hyfilebrowser({
                conf: {
                    vroots: [vroot],
                    colmodel: colmodel,
                    params: {
                        serviceParams: {
                            filter: this.options.filter
                        },
                        multiselect: this.options.multiselect,
                        rowMenu: false,
                        spoolerURI: this.options.spooler,
                        height: 'auto',
                        scroll: 1,
                        rowNum: this.options.rowNum,
                        SDF: $.hydrogen.SDF,
                        serviceUri: '//ui.hydrogen/fm.browse.data',
                        isRFB: true
                    }
                },
                messageArea: this.msg
            });

            this.hyfm.bind('hyfilebrowserloaderror', function (e, data) {
                if (firstLoad) {
                    setTimeout(function () {
                        self.changePath(self.options.base || "/");
                    }, 2000);
                }
                firstLoad = false;
            });

            this.hyfm.bind('hyfilebrowserloadcomplete', function (e, data) {
                var cookieContent, dir, savedPaths, $mainRFB;

                dir = $.hydrogen.normalizeFilepath($(this).hyfilebrowser('currentPath'));
                savedPaths = { path: dir };
                cookieContent = $.cookie(self.options.cookie);
                if (cookieContent) {
                    try {
                        savedPaths = JSON.parse(cookieContent);
                    } catch (_) { /* ignore */ }
                }

                // If the current RFB has been marked as main, set the main path into the cookie,
                // otherwise set the last visited path.
                if (typeof self.options.rfbClass === 'string' && self.options.rfbClass.indexOf('rfb-main-location') !== -1) {
                    savedPaths.main = dir;
                } else {
                    // Check if at least one RFB is marked as main.
                    // If so, we only need to update the last visited path preserving the main one.
                    // If not, we remove the main path, if any, from the cookie content.
                    $mainRFB = $('.rfb-main-location').first();
                    if ($mainRFB.length > 0) {
                        savedPaths.path = dir;
                    } else {
                        savedPaths = { path: dir };
                    }
                }

                $.cookie(self.options.cookie, JSON.stringify(savedPaths), {
                    expires: 30
                });

                $(this).hyfilebrowser('setSelection', self.selection.get(dir));
                firstLoad = false;
            });

            this.hyfm.bind('hyfilebrowserselectionchanged', function (e, data) {
                var dir, selectedIds,
                    selectedFiles = [];

                dir = $.hydrogen.normalizeFilepath(self.hyfm.hyfilebrowser('currentPath'));
                if (!self.options.multiselect) {
                    self.selection.clear();
                }

                selectedIds = self.hyfm.hyfilebrowser('selected');
                jQuery.each(selectedIds, function (index, value) {
                    var rowData = self.hyfm.hyfilebrowser('getRowData', value);
                    selectedFiles.push(rowData.name);
                });

                self.selection.set(dir, selectedFiles);
                self._updateStatus();
            });

            // Enable double-click only on single selection mode since toggling off
            // all the other previously selected items would be confusing.
            if (!this.options.multiselect) {
                this.hyfm.bind('hyfilebrowserdoubleclickrow', function (e, data) {
                    var row = self.hyfm.hyfilebrowser('getRowData', data.rowid);
                    if (row.selectable === 'true' && row.type === 'file') {
                        $('#serviceopt_' + self.options.optionid).val(self.selection.all().join('\n')).change();
                        self.element.dialog('close');
                    }
                });
            }

            this._updateStatus();

            this._setGridSize();
        },

        _setGridSize: function () {
            var h = this.element.height() - (this.toolbar.outerHeight(true) + $('.hy-file-browser-location-bar', this.htfm).outerHeight(true) + 30);
            this.hyfm.hyfilebrowser('setHeight', h);
        },

        _updateStatus: function () {
            var self, sel, label;

            self = this;

            sel = this.selection.all();
            label = $('span.hy-rfb-status-value', this.statusBar);

            if (sel.length === 0) {
                label.text('no items');
            } else {
                if (sel.length === 1) {
                    label.text(sel[0]);
                } else {
                    label.text(sel.length + ' items');
                }
            }
        },

        prefillSelection: function () {
            var self, current_path, last_selected;

            self = this;
            last_selected = null;

            // prepopulate the selection if there is something typed in the input form
            this.selection.clear();
            $.each($('#serviceopt_' + this.options.optionid).val().split('\n'), function (i, v) {
                var path, base, dir;
                path = $.hydrogen.normalizeFilepath(v);

                // Only for legacy case, if path does NOT contain '/', add it to the path.
                // E.g. RFB for select users or group (so, a plain list of items, not a tree)
                if (self.options.legacy && path && path.length > 0 && path.indexOf('/') < 0) {
                    path = '/' + path;
                }

                // We only accept absolute paths. Should we use "base" for relative paths?
                if (path && path.length > 0 && path.indexOf('/') === 0) {
                    base = $.hydrogen.basename(path);
                    dir = $.hydrogen.dirname(path);
                    self.selection.add(dir, [base]);
                    last_selected = dir;
                }
            });

            //  this runs only when showing again the dialog
            return last_selected;
        },

        changePath: function (path) {
            this._updateStatus();
            this.hyfm.hyfilebrowser('setSelection', this.selection.get(path));
            this.hyfm.hyfilebrowser('changePath', path);
        }

    });

}(jQuery));

// ex:ts=4:et:
