/*global jQuery, ace */
////////////////////////////////////////////////////////////////////////////////
// Copyright 2023-2025 by NI SP Software GmbH, All rights reserved.
// Copyright 1999-2023 by Nice, srl., All rights reserved.
//
// This software includes confidential and proprietary information
// of NI SP Software GmbH ("Confidential Information").
// You shall not disclose such Confidential Information
// and shall use it only in accordance with the terms of
// the license agreement you entered into with NI SP Software.
////////////////////////////////////////////////////////////////////////////////

(function ($) {

    $.widget('ui.texteditor', {

        options: {
            height : "300px",
            width : "100%",
            mode : "sh",
            data : "",
            autocompleter : "",
            toolbar: false,
            toolbarRowNumberCheckbox: false,
            rowNumberEnabled: false,
            toolbarModeSelector: false
        },

        _toolbarHeight: 35,

        _create: function () {
            this.element.addClass('text-editor');
            this.divId = this.element.attr("id");
            this.editorId = this.divId + "-text";

            $("#" + this.divId).append(
                    "<div class='editor-mask'>" +
                        "<div class='editor-container'>" +
                            "<div id='" + this.editorId + "' class='editor-content'>" +
                            "</div>" +
                        "</div>" +
                    "</div>");

            this._initTextEditor();

            if (this.options.toolbar) {
                this._initToolbar();
            }

            $("#" + this.divId + " .editor-content").css({ position: "absolute", top: "0", right: "0", bottom: "0", left: "0" });
            $("#" + this.divId + " .editor-container").css({ position: "relative", height: this.options.height, width: this.options.width});
            $("#" + this.divId + " .editor-mask").css({ height: this.options.height, "overflow-y": "hidden" });
        },

        setContent: function (data) {
            /* ACE editor specific code */
            ace.edit(this.editorId).getSession().setValue(data, -1);
        },

        getContent: function () {
            /* ACE editor specific code */
            return ace.edit(this.editorId).getSession().getValue();
        },

        changeMode: function (mode) {
            /* ACE editor specific code */
            this.editor.session.setMode("ace/mode/" + mode);
            this.editor.session.modeName = "ace/mode/" + mode;
        },

        enableAutocompletion: function (value) {
            /* ACE editor specific code */
            this.editor.setOptions({
                enableBasicAutocompletion: value
            });
        },

        cleanup: function () {
            /* ACE editor specific code */
            $("#" + this.divId + " .ace_autocomplete").hide();
        },

        resize: function (size) {
            if (size && size.height) {
                this.options.height = size.height;
                if (this.options.toolbar) {
                    this.options.height = this.options.height - this._toolbarHeight;
                }
            } else if ($('#' + this.divId).innerHeight()) {
                this.options.height = $('#' + this.divId).innerHeight();
                if (this.options.toolbar) {
                    this.options.height = this.options.height - this._toolbarHeight;
                }
            }
            $("#" + this.divId + " .editor-container").height(this.options.height);
            $("#" + this.divId + " .editor-mask").height(this.options.height);
            /* ACE editor specific code */
            ace.edit(this.editorId).resize();
        },

        _initTextEditor: function () {
            /* ACE editor specific code */
            this.editorLangTools = ace.require("ace/ext/language_tools");

            this.editor = ace.edit(this.editorId);
            this.editor.setTheme("ace/theme/eclipse");
            this.editor.session.setMode("ace/mode/" + this.options.mode);
            this.editor.session.setUseWrapMode(true);
            this.editor.session.setUseSoftTabs(true);
            this.editor.session.setTabSize(2);
            this.editor.setShowInvisibles(false);
            this.editor.setHighlightSelectedWord(true);
            this.editor.setHighlightActiveLine(true);
            this.editor.setBehavioursEnabled(true);
            this.editor.renderer.setShowGutter(this.options.rowNumberEnabled);
            this.editor.setShowPrintMargin(false);
            this.editor.clearSelection();
            this.editor.$blockScrolling=Infinity;
            this.editor.setValue(this.options.data);

            if (this.options.autocompleter !== "") {
                this._initAutocompletion();
            }
        },

        _initAutocompletion: function () {
            var self, addCompleter;

            self = this;
            this.editor.commands.on("afterExec", function (e) {
                if (e.command.name == "insertstring" && /\$/.test(e.args)) {
                    self.editor.execCommand("startAutocomplete");
                }
            });

            this.editor.setOptions({
                enableBasicAutocompletion: true
            });

            addCompleter = true;
            jQuery.each(this.editor.completers, function(index, val) {
                 if (jQuery.isFunction(val.getName)) {
                    if (val.getName() === "efcompleter") {
                        addCompleter = false;
                    }
                 }
            });

            if (addCompleter) {
                this.editorLangTools.addCompleter(this.options.autocompleter);
            }
        },

        _initToolbar: function () {
            var self;

            self = this;

            this.editorToolbar = $("<div>", {"id": self.editorId + "-toolbar", "class": "editor-toolbar"});
            this.editorToolbar.css("height", self._toolbarHeight);
            $(self.element).append(this.editorToolbar);
            jQuery("#" + this.divId + " .editor-toolbar").width("100%").css("background", "none repeat scroll 0 0 #E3E3E3").css("font-size", "0.9em");

            if (this.options.toolbarRowNumberCheckbox) {
                this._initRowNumberCheckbox();
            }
            if (this.options.toolbarModeSelector) {
                this._initModeSelector();
            }
            this._initCursorPosition();

            // decrease editor container size
            this.options.height = this.options.height - this._toolbarHeight;
            $("#" + this.divId + " .editor-container").height(this.options.height);
        },

        _initRowNumberCheckbox: function () {
            var self, rowNumberCheckbox;

            self = this;

            rowNumberCheckbox = $("<input>", {
                "id": self.editorId + "-rownumber-checkbox",
                "class": "editor-toolbar-checkbox",
                "type": "checkbox"
            });

            rowNumberCheckbox.change(function () {
                self.editor.renderer.setShowGutter($(this).is(':checked'));
            });
            rowNumberCheckbox.after("<label for='checkbox'>Show Row Number</label>");
            $(this.editorToolbar).append(rowNumberCheckbox);
            rowNumberCheckbox.prop('checked', self.options.rowNumberEnabled);
            jQuery("#" + this.divId + " .editor-toolbar-checkbox").css("margin", "0.5em");
        },

        _initModeSelector: function() {
            var self, supportedModes, modeSelector;

            self = this;

            supportedModes = {
                "Bash/SH": "sh",
                /*"Batch": "batchfile",*/
                "Python": "python",
                "Perl": "perl"
            };
            modeSelector = $("<select>", {"id": self.editorId + "-mode-selector", "class": "editor-toolbar-selector"});
            $.each(supportedModes, function (key, value) {
                modeSelector.append($("<option />").val(value).text(key));
            });
            $(this.editorToolbar).append(modeSelector);

            modeSelector.change(function (key, value) {
                self.editor.session.setMode("ace/mode/" + this.value);
                self.editor.session.modeName = "ace/mode/" + this.value;
            });
            jQuery("#" + this.divId + " .editor-toolbar-selector").css("margin", "0.5em 0.5em 0.5em 1.5em");
        },

        _initCursorPosition: function () {
            var self, cursorPosition;

            self = this;

            cursorPosition = $("<span>", {"id": self.editorId + "-cursor-position", "class": "editor-cursor-position"});
            $(cursorPosition).append($("<span>", {"id": self.editorId + "-cursor-position-value", "class": "editor-cursor-position-value", "text": "Ln 0, Col 0"}));
            $(this.editorToolbar).append(cursorPosition);
            
            this.editor.session.selection.on('changeCursor', function () {
                var position, rowN, colN;
                
                position = self.editor.selection.getCursor();
                rowN = parseInt(position.row, 10) + 1;
                colN = parseInt(position.column, 10) + 1;
                jQuery("#" + self.editorId + "-cursor-position-value").text("Ln " + rowN + ", Col " + colN);
            });
            jQuery("#" + this.divId + " .editor-cursor-position").css("padding", "0.5em").css("float", "right");
        },

        _destroy: function() {
            this.editor.destroy();
            this.element.removeClass("text-editor");
            this.element.children().remove();
            return this;
        }
    });
}(jQuery));
