#!/bin/bash

. "${EF_ROOT}/plugins/ef/lib/utils"
. "${EF_ROOT}/plugins/ef/conf/ef.xmlns.conf"

# Emit the specified interactive remote setting as hydrogen params.
# A different param will be emitted for every remote.
# The interactive remote settings read from the environment are in the form
# INTERACTIVE_DEFAULT_<REMOTE>_$1 and INTERACTIVE_PROFILE_<REMOTE>_$1.
# If present, the value of INTERACTIVE_PROFILE_<REMOTE>_$1 win over the value of INTERACTIVE_DEFAULT_<REMOTE>_$1
# The input parameter is expressed as uppercase and underscore, e.g. SETTING_NAME, while the emitted
# hydrogen parameter will be expressed as camel case, following the syntax <remote>SettingName
#
# E.g.
# calling emit_hy_param_for_interactive_remote_setting with the parameter "VIEWER_TYPE"
# will search for the variables
# INTERACTIVE_DEFAULT_<REMOTE>_VIEWER_TYPE and INTERACTIVE_PROFILE_<REMOTE>_VIEWER_TYPE
# for every remote
# The PROFILE setting, if present, will win over the DEFAULT one
# if
# INTERACTIVE_DEFAULT_DCV2_VIEWER_TYPE=browser
# INTERACTIVE_PROFILE_DCV2_VIEWER_TYPE=client
# are found, then the outpout will be
# <hy:set-param name="dcv2ViewerType">client</hy:set-param>
emit_hy_param_for_interactive_remote_setting() {

    local -r _param_suffix=$(echo "$1" | tr '[:lower:]' '[:upper:]')

    if [ -n "${_param_suffix}" ]; then

        while read -r line; do
            # Extract "remote" value from the variable line, e.g. take from "INTERACTIVE_DEFAULT_DCV2_VIEWER_TYPE=browser" the value "DCV2"
            _remote="$(_tmp=${line#INTERACTIVE_DEFAULT_*}; _tmp=${_tmp%*_${_param_suffix}=*}; echo ${_tmp} | tr '[:upper:]' '[:lower:]' )"
            # Extract the value from the variable line, e.g. take from "INTERACTIVE_DEFAULT_DCV2_VIEWER_TYPE=browser" the value "browser"
            _value="${line#*${_param_suffix}=}"

            _profile_var_name="INTERACTIVE_PROFILE_$(echo ${_remote} | tr '[:lower:]' '[:upper:]')_${_param_suffix}"
            _profile_value="${!_profile_var_name}"

            if [ -n "${_profile_value}" ]; then
                # override default value with profile value
                _value="${_profile_value}"
            fi

            if [ -n "${_remote}" -a -n "${_value}" ]; then
                echo "<hy:set-param name=\"${_remote}$(_underscore_to_camelcase "${_param_suffix}")\" type=\"text\" ${EF_XMLNS_hy}>$(ef_xml_escape_content -i "${_value}")</hy:set-param>"
            fi

        done < <(env | grep "INTERACTIVE_DEFAULT_.*_${_param_suffix}")
    fi
}

# Convert to uppercase the first letter following the underscore and remove all underscores e.g.
# dcv2_VIEWER_TYPE is converted to dcv2ViewerType
# Used to convert
_underscore_to_camelcase() {
    echo "$1" | tr '[:upper:]' '[:lower:]' | sed -r 's/(^|_)([a-z])/\U\2/g'
}
