#!/bin/bash

################################################################################
# Copyright 2023-2025 by NI SP Software GmbH, All rights reserved.
# Copyright 1999-2023 by Nice, srl., All rights reserved.
#
# This software includes confidential and proprietary information
# of NI SP Software GmbH ("Confidential Information").
# You shall not disclose such Confidential Information
# and shall use it only in accordance with the terms of
# the license agreement you entered into with NI SP Software.
################################################################################

# This script launches an AWS WorkSpaces Core Instance and sets the SESSION_INTERACTIVE_ADDITIONAL_SUBMITOPTS variable
# to propagate additional requirements to DCV Session Manager and run the Interactive DCV Session in the new instance.

# This script is meant to be used with DCV 2017 (and later) interactive sessions only.
# EF Portal must have the DCV Session Manager configured as remote visualization technology.
# The AMI used for the EC2 instance must have the DCV server installed and configured.

# The script requires a set of configuration parameters to be set, see following section.

# === CONFIGURATION START ===

# AWS configuration. AWS Cli must be installed and AWS credentials must be configured for EF Portal server user.
REGION=""
IMAGE_ID="ami-123"
INSTANCE_TYPE="t3.medium"
SUBNET_ID="subnet-123"
SECURITY_GROUP_ID="sg-123"

# The following value must be set as prefix for the Session Class in the Service Settings to enable the execution of this script.
REQUIRED_SESSION_CLASS_PREFIX="workspaces_hook"

# At Session Submission time the script will try to search and reuse the same WorkSpace and EC2 instance
# for the next Interactive Session of the same user with the same Service.
# Set the following parameter to "delete" if you want to deallocate the WorkSpace instance and terminate the EC2 instance
# at Interactive Session closing time.
# Set it to "stop" if you want to stop the EC2 instance while keeping the WorkSpace instance allocated.
# Unset it if you want to keep the EC2 instance running and the WorkSpace instance allocated.
INSTANCE_ACTION_AT_SESSION_CLOSE="stop"

# === CONFIGURATION END ===


# The following variables are used to tag the instances and will be queried to reuse existing instances, do not modify them.
INSTANCE_USER_TAG_NAME="efp-session-user"
INSTANCE_USER_TAG_VALUE="${EF_USER}"
INSTANCE_SPOOLER_NAME_TAG_NAME="efp-spooler-name"
INSTANCE_SPOOLER_NAME_TAG_VALUE="${EF_SPOOLER_NAME}"
INSTANCE_NAME_TAG_VALUE="${EF_SPOOLER_NAME} - ${EF_USER}"


# Check for required tools
check_dependencies() {
  # Check for AWS CLI
  if ! command -v aws >/dev/null 2>&1; then
    echo "Error: AWS CLI not found."
    exit 1
  else
    # Try to retrieve the list of regions
    local _regions
    _regions=$(aws ec2 describe-regions --query "Regions[].RegionName" --output text 2>/dev/null)

    if [ -z "${_regions}" ]; then
        echo "AWS CLI is installed but failed to retrieve regions. Check your credentials or network."
        exit 1
    fi
  fi

  # Check for jq
  if ! command -v jq >/dev/null 2>&1; then
      echo "Error: jq not found."
      exit 1
  fi
}


check_configuration_parameters() {
    local _missing_conf=0

    # Check each variable
    if [ -z "$REGION" ]; then
        echo "Error: REGION is not set."
        _missing_conf=1
    fi
    if [ -z "$IMAGE_ID" ]; then
        echo "Error: IMAGE_ID is not set.  Must be in the form ami-12345678901234567"
        _missing_conf=1
    fi
    if [ -z "$INSTANCE_TYPE" ]; then
        echo "Error: INSTANCE_TYPE is not set. Must be a valid AWS Instance Type (e.g. t3.medium)"
        _missing_conf=1
    fi
    if [ -z "$SUBNET_ID" ]; then
        echo "Error: SUBNET_ID is not set.  Must be in the form subnet-12345678901234567"
        _missing_conf=1
    fi
    if [ -z "$SECURITY_GROUP_ID" ]; then
        echo "Error: SECURITY_GROUP_ID is not set. Must be in the form sg-12345678901234567"
        _missing_conf=1
    fi

    # Final verdict
    if [ "${_missing_conf}" -eq 1 ]; then
        echo "Error: One or more configuration parameters are missing. Please review them."
        exit 1
    else
        echo "All AWS configuration parameters are set correctly."
    fi
}


# Propagate configuration parameters to be available for the closing hook.
# Variables prefixed with SESSION_ and exported in the environment will be automatically set as metadata.
propagate_configuration_parameters() {
  export SESSION_WORKSPACES_AWS_REGION="$REGION"
  export SESSION_WORKSPACES_INSTANCE_ACTION_AT_SESSION_CLOSE="$INSTANCE_ACTION_AT_SESSION_CLOSE"
  export SESSION_WORKSPACES_REQUIRED_SESSION_CLASS_PREFIX="$REQUIRED_SESSION_CLASS_PREFIX"
}


# Create Workspace Core Instance
# Set WORKSPACE_INSTANCE_ID in the environment
create_workspace_instance() {
  local _create_output

  echo "Creating WorkSpaces Core instance..."
  # We also pass the tags to the EC2 instance because the tags set via the create-workspace-instance API
  # cannot be used to filter workspace instances directly.
  # They are only accessible through the list-tags-for-resource API.
  _create_output=$(aws workspaces-instances create-workspace-instance \
    --region "$REGION" \
    --managed-instance "{
      \"ImageId\": \"$IMAGE_ID\",
      \"InstanceType\": \"$INSTANCE_TYPE\",
      \"SubnetId\": \"$SUBNET_ID\",
      \"SecurityGroupIds\": [\"$SECURITY_GROUP_ID\"],
      \"TagSpecifications\": [
        {
          \"ResourceType\": \"instance\",
          \"Tags\": [
             {\"Key\": \"$INSTANCE_USER_TAG_NAME\", \"Value\": \"$INSTANCE_USER_TAG_VALUE\"},
             {\"Key\": \"$INSTANCE_SPOOLER_NAME_TAG_NAME\", \"Value\": \"$INSTANCE_SPOOLER_NAME_TAG_VALUE\"},
             {\"Key\": \"Name\", \"Value\": \"$INSTANCE_NAME_TAG_VALUE\"}
          ]
       }
     ]
    }" \
    --tags "[
      {\"Key\": \"$INSTANCE_USER_TAG_NAME\", \"Value\": \"$INSTANCE_USER_TAG_VALUE\"},
      {\"Key\": \"$INSTANCE_SPOOLER_NAME_TAG_NAME\", \"Value\": \"$INSTANCE_SPOOLER_NAME_TAG_VALUE\"},
      {\"Key\": \"Name\", \"Value\": \"$INSTANCE_NAME_TAG_VALUE\"}
    ]")

  echo "Output of create-workspace-instances API call:"
  echo "${_create_output}"

  echo "Tagged EC2 instance with Name='$INSTANCE_NAME_TAG_VALUE',$INSTANCE_USER_TAG_NAME='$INSTANCE_USER_TAG_VALUE',$INSTANCE_SPOOLER_NAME_TAG_NAME='$INSTANCE_SPOOLER_NAME_TAG_VALUE'"

  WORKSPACE_INSTANCE_ID="$(echo "${_create_output}" | jq -r '.WorkspaceInstanceId')"
}


# Poll for WorkSpaces Instance Provision State
# Set WORKSPACE_EC2_INSTANCE_ID in the environment when the corresponding WorkSpace Instance is allocated.
wait_for_workspace_instance() {
  local _workspace_instance_id="$1"
  local _region="$2"

  local _status_output
  local _provision_state

  if [ -z "${_workspace_instance_id}" ]; then
    echo "Error: WorkSpace Instance Id not available."
    exit 1
  fi

  printf "\nWaiting for provisioning to complete...\n"
  while true; do
    _status_output=$(aws workspaces-instances get-workspace-instance \
      --region "${_region}" \
      --workspace-instance-id "${_workspace_instance_id}")
    # Output is something like:
    # { "ProvisionState": "ALLOCATED", "WorkspaceInstanceId": "wsinst-123", "EC2ManagedInstance": { "InstanceId": "i-123" } }
    _provision_state=$(echo "${_status_output}" | jq -r '.ProvisionState')
    echo "Current state: ${_provision_state}"

    if [ "${_provision_state}" == "ALLOCATED" ]; then
      echo "Workspace instance is ready: ${_workspace_instance_id}"
      WORKSPACE_EC2_INSTANCE_ID=$(echo "${_status_output}" | jq -r '.EC2ManagedInstance.InstanceId')
      echo "Corresponding EC2 instance is: $WORKSPACE_EC2_INSTANCE_ID"
      break
    fi

    if [ "${_provision_state}" != "ALLOCATING" ]; then
      echo "Error: Workspace instance is in a wrong state: ${_provision_state}, cannot execute session on it"
      exit 1
    fi

    # Wait before polling again to avoid hitting API rate limits
    sleep 5
  done
}


# Return current state of an EC2 instance
get_ec2_instance_state() {
    local _ec2_instance_id="$1"
    local _region="$2"
    local _ec2_status

    _ec2_status=$(aws ec2 describe-instances \
      --instance-ids "${_ec2_instance_id}" \
      --region "${_region}" \
      --query "Reservations[0].Instances[0].State.Name" \
      --output text 2>/dev/null)

    echo "${_ec2_status}"
}


# Start the EC2 instance associated to the WorkSpace instance
start_ec2_instance() {
  local _ec2_instance_id="$1"
  local _region="$2"

  if [ -z "${_ec2_instance_id}" ]; then
    echo "Error: EC2 Instance Id not available. Unable to start it."
    exit 1
  fi

  # Instances in stopping or pending state cannot be started
  echo "Waiting for EC2 instance ${_ec2_instance_id} to be in a stable state before starting it..."
  while true; do
    _ec2_status=$(get_ec2_instance_state "${_ec2_instance_id}" "${_region}")

    if [ "${_ec2_status}" == "stopped" ] || [ "${_ec2_status}" == "running" ]; then
      echo "EC2 instance is in ${_ec2_status} state."
      break
    elif [ "${_ec2_status}" == "stopping" ] || [ "${_ec2_status}" == "pending" ]; then
      echo "EC2 instance is in ${_ec2_status} state. Waiting..."
    else
      echo "Error: EC2 instance is in ${_ec2_status} state. It cannot be started"
      exit 1
    fi

    # Wait before polling again to avoid hitting API rate limits
    sleep 5
  done

  echo "Starting EC2 instance ${_ec2_instance_id}..."
  aws ec2 start-instances --region "${_region}" --instance-ids "${_ec2_instance_id}" >/dev/null
}


# Query EC2 instances to find one associated to the given Service and user and start it.
# Set WORKSPACE_INSTANCE_ID if there is a running EC2 instance with appropriate tags
search_workspace_instance_using_ec2_tags() {
  local _ec2_instance_id
  local _workspace_instance_id_tag_value

  echo "Searching for EC2 instances with tags: $INSTANCE_USER_TAG_NAME='$INSTANCE_USER_TAG_VALUE', $INSTANCE_SPOOLER_NAME_TAG_NAME='$INSTANCE_SPOOLER_NAME_TAG_VALUE' ..."

  # We can re-use only instances in stated different from shutting-down and terminated
  # https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-lifecycle.html
  _ec2_instance_id=$(aws ec2 describe-instances \
    --region "$REGION" \
    --filters \
      "Name=tag:$INSTANCE_USER_TAG_NAME,Values=$INSTANCE_USER_TAG_VALUE" \
      "Name=tag:$INSTANCE_SPOOLER_NAME_TAG_NAME,Values=$INSTANCE_SPOOLER_NAME_TAG_VALUE" \
      "Name=instance-state-name,Values=pending,running,stopping,stopped" \
    --query "Reservations[*].Instances[*].InstanceId" \
    --output text)

  if [ -n "${_ec2_instance_id}" ]; then
    echo "Found EC2 instance: ${_ec2_instance_id}, looking for corresponding Workspace Instance Id..."

    # Retrieve the workspace-instance-id tag value from the EC2 instance
    _workspace_instance_id_tag_value=$(aws ec2 describe-tags \
      --region "$REGION" \
      --filters \
        "Name=resource-id,Values=${_ec2_instance_id}" \
        "Name=key,Values=aws:workspaces-instances:workspace-instance-id" \
      --query "Tags[0].Value" \
      --output text)

    if [ "${_workspace_instance_id_tag_value}" != "None" ]; then
      echo "Workspace Instance ID tag found."
      WORKSPACE_INSTANCE_ID="${_workspace_instance_id_tag_value}"
    else
      echo "Error: Tag aws:workspaces-instances:workspace-instance-id not found on instance ${_ec2_instance_id}"
      exit 1
    fi

    start_ec2_instance "${_ec2_instance_id}" "$REGION"
  else
    echo "No EC2 instance found with the specified tags."
  fi
}


# Poll for EC2 Instance State
wait_for_ec2_instance() {
  local _ec2_instance_id="$1"
  local _region="$2"

  local _ec2_status
  local _instance_status_info
  local _ec2_system_check
  local _ec2_system_status

  if [ -z "${_ec2_instance_id}" ]; then
    echo "Error: EC2 Instance Id not available."
    exit 1
  fi

  echo "Waiting for EC2 instance ${_ec2_instance_id} to be running and ready..."
  while true; do
    _ec2_status=$(get_ec2_instance_state "${_ec2_instance_id}" "${_region}")
    echo "EC2 instance state: ${_ec2_status}"

    if [ "${_ec2_status}" == "running" ]; then
      # Check instance status (system and instance checks)
      _instance_status_info=$(aws ec2 describe-instance-status \
        --region "${_region}" \
        --instance-ids "${_ec2_instance_id}" \
        --query 'InstanceStatuses[0].{SystemStatus:SystemStatus.Status,InstanceStatus:InstanceStatus.Status}' \
        --output json)

      _ec2_system_status=$(echo "${_instance_status_info}" | jq -r '.SystemStatus')
      _ec2_system_check=$(echo "${_instance_status_info}" | jq -r '.InstanceStatus')

      echo "System status: ${_ec2_system_status}, Instance status: ${_ec2_system_check}"

      if [ "${_ec2_system_status}" == "ok" ] && [ "${_ec2_system_check}" == "ok" ]; then
        echo "EC2 instance is fully ready"
        break
      fi
    elif [ "${_ec2_status}" == "terminated" ] || [ "${_ec2_status}" == "shutting-down" ]; then
      echo "EC2 instance is in ${_ec2_status} state. Exiting"
      exit 1
    fi
    # Wait before polling again to avoid hitting API rate limits
    sleep 5
  done
}


# Set Environment Variables required to submit the session in the just launched instance.
# Variables prefixed with SESSION_ and exported in the environment will be automatically set as Session metadata
# and will be available for the closing hook as well.
export_runtime_session_variables() {
  local _workspace_instance_id="$1"
  local _ec2_instance_id="$2"

  # The SESSION_INTERACTIVE_ADDITIONAL_SUBMITOPTS name is a reserved name and it is used to propagate additional
  # requirements to DCV Session Manager
  export SESSION_INTERACTIVE_ADDITIONAL_SUBMITOPTS="server:Host.Aws.Ec2InstanceId=${_ec2_instance_id}\""

  # Set additional environment variables to be used by the closing hook
  export SESSION_WORKSPACES_EC2_INSTANCE_ID="${_ec2_instance_id}"
  export SESSION_WORKSPACES_INSTANCE_ID="${_workspace_instance_id}"

  printf "\nEnvironment variables with SESSION_ prefix:\n\n"
  env | grep '^SESSION_' | sort
}


main() {
  if [[ $INTERACTIVE_SESSION_CLASS == $REQUIRED_SESSION_CLASS_PREFIX* ]]; then
    check_dependencies
    printf "\nSession %s for OS %s is going to be submitted on cluster %s\n\n" "${EF_SPOOLER_NAME}" "${INTERACTIVE_SESSION_OS}" "${INTERACTIVE_SESSION_CLUSTER}"
    check_configuration_parameters
    propagate_configuration_parameters

    if [ "$INSTANCE_ACTION_AT_SESSION_CLOSE" = "delete" ]; then
      echo "INSTANCE_ACTION_AT_SESSION_CLOSE is set to 'delete' — WorkSpaces instance will be deallocated and EC2 instance terminated"
    elif [ "$INSTANCE_ACTION_AT_SESSION_CLOSE" = "stop" ]; then
      echo "INSTANCE_ACTION_AT_SESSION_CLOSE is set to 'stop' — WorkSpaces instance will be preserved but EC2 instance will be stopped"
    else
      printf "\nINSTANCE_ACTION_AT_SESSION_CLOSE is set to '%s'. Both WorkSpace and EC2 instances will be preserved.\n" "$SESSION_WORKSPACES_INSTANCE_ACTION_AT_SESSION_CLOSE"
    fi
    # Set WORKSPACE_INSTANCE_ID in the environment if found a matching instance in a reusable state
    search_workspace_instance_using_ec2_tags

    # If WORKSPACE_INSTANCE_ID has not be found and is not set in the environment, create a new one
    if [ -z "$WORKSPACE_INSTANCE_ID" ]; then
      # Create instance and set WORKSPACE_INSTANCE_ID in the environment
      create_workspace_instance
    fi
    echo "WorkspaceInstanceId: $WORKSPACE_INSTANCE_ID"

    # Wait for WorkSpaces instance and set WORKSPACE_EC2_INSTANCE_ID in the environment
    wait_for_workspace_instance "$WORKSPACE_INSTANCE_ID" "$REGION"

    wait_for_ec2_instance "$WORKSPACE_EC2_INSTANCE_ID" "$REGION"
    export_runtime_session_variables "$WORKSPACE_INSTANCE_ID" "$WORKSPACE_EC2_INSTANCE_ID"

    printf "\nInteractive Session pre submit hook completed.\n\n"
  else
    # Skip hook for sessions where Session Class does not have "workspaces_hook" as prefix
    echo "INTERACTIVE_SESSION_CLASS is '$INTERACTIVE_SESSION_CLASS'. It does not have '$REQUIRED_SESSION_CLASS_PREFIX' has prefix. Skipping the hook."
  fi
}


main
