#!/bin/bash

################################################################################
# Copyright 2023-2025 by NI SP Software GmbH, All rights reserved.
# Copyright 1999-2023 by Nice, srl., All rights reserved.
#
# This software includes confidential and proprietary information
# of NI SP Software GmbH ("Confidential Information").
# You shall not disclose such Confidential Information
# and shall use it only in accordance with the terms of
# the license agreement you entered into with NI SP Software.
################################################################################
#################################################################################
#
# Script to add an rspooler entry in current spooler
# support@ni-sp-software.com
#
#



# Print help
print_usage() {
  cat << EOF

  Remote spooler file creator script
  Copyright 2023-2025 by NI SP Software GmbH, All rights reserved.
  Copyright 1999-2023 by Nice, srl., All rights reserved.

  Usage: ef.rspooler.add.sh <options>
    required options:
      <-h host>        - Remote host (name or ip address)
      <-p path>        - Remote full path
    optional:
      [-t type]        - Remote spooler type. Default: local
      [-u user]        - Remote user. Default: user logged into EnginFrame
      [-i rspooler_id] - Remote spooler id. Must be unique and integer
      [-l label]       - Label in file manager (do not use ':' character)
      [-s spooler]     - EnginFrame spooler. Default: EF_SPOOLER env var
      [-e]             - Emulate. Doesn't actually write .rspooler
      [-q]             - Quiet mode. Do not print anything in std output
      [-h]             - Print this message
      [-r]             - EnginFrame root. Defaults to EF_ROOT env var
      [-v]             - Verbose output

EOF
}


# Setting defaults
RSPOOLER_HOST=localhost
RSPOOLER_TYPE=local
RSPOOLER_USER=${USER}
RSPOOLER_SPOOLER=${EF_SPOOLER}
QUIET_MODE=""


# Parse options
if [ -z "$*" ]; then
  print_usage
  exit 0
fi

while [ $# -gt 0 ]; do
  case "$1" in
    -h)   shift
          if [ -n "$1" ]; then
            RSPOOLER_HOST="$1" 
          else 
            print_usage
            exit 0
          fi
          ;;
    -p)   shift
          RSPOOLER_PATH="$1"
          ;;
    -t)   shift
          RSPOOLER_TYPE="$1"
          ;;
    -u)   shift
          RSPOOLER_USER="$1"
          ;;
    -i)   shift
          RSPOOLER_ID="$1"
          ;;
    -l)   shift
          RSPOOLER_FM_ANCHOR_LABEL="$1"
          ;;
    -s)   shift
          RSPOOLER_SPOOLER="$1"
          ;;
    -q)   QUIET_MODE=true
          ;;
    -r)   shift
          EF_ROOT="$1"
          ;;
    -v)   VERBOSE_MODE=true
          ;;
    -e)   EMULATE_MODE=true
          ;;
    [?])  print_usage
          exit 0
          ;;
  esac
  shift
done


# Checks
if [ -z "${EF_ROOT}" ]; then
  [ -n "${QUIET_MODE}" ] || echo "Error: EF_ROOT not found nor specified"
  exit 1
fi

if [ "${RSPOOLER_HOST}" = "" -o "${RSPOOLER_PATH}" = "" ]; then
  [ -n "${QUIET_MODE}" ] || echo "Error: Missing host and/or path, required"
  exit 1
fi

if [ -z "${RSPOOLER_SPOOLER}" ]; then
  [ -n "${QUIET_MODE}" ] || echo "Error: no spooler found. Please specify spooler or export variable EF_SPOOLER"
  exit 1
fi

if [ ! -w "${RSPOOLER_SPOOLER}" ]; then
  [ -n "${QUIET_MODE}" ] || echo "Error: cant write .rspooler file under ${RSPOOLER_SPOOLER}"
  exit 1
fi

# If label is empty, use path as label
[ -n "${RSPOOLER_FM_ANCHOR_LABEL}" ] || RSPOOLER_FM_ANCHOR_LABEL=`echo "${RSPOOLER_PATH}" | sed 's/:/ /g'`

# If not specified, find a correct rspooler id
if [ -z "${RSPOOLER_ID}" ]; then
  [ -r "${RSPOOLER_SPOOLER}/.rspooler" ] && RSPOOLER_ENTRIES=`cat "${RSPOOLER_SPOOLER}/.rspooler" | sed -n "$="`  
  if [ -z "${RSPOOLER_ENTRIES}" -o "${RSPOOLER_ENTRIES}" = "0" ]; then
    # Base case: no entries in .rspooler
    [ ! -n "${RSPOOLER_ID}" ] && RSPOOLER_ID="1"
  else
    # Take the last rspooler_id
    LAST_RSPOOLER_ID=`cat "${RSPOOLER_SPOOLER}/.rspooler" | sed '$!d' | sed 's/^\([^:]*\):*\([^:]*\):*\([^:]*\):*\([^:]*\):*\(.*\)$/\4/'`
    RSPOOLER_ID=`expr ${LAST_RSPOOLER_ID} + 1 2>/dev/null`
    if [ ! "$?" = "0" ]; then
      [ -n "${QUIET_MODE}" ] || echo "Error: found non-numeric rspooler id in file ${RSPOOLER_SPOOLER}/.rspooler. Please correct. Operation canceled."
      exit 1
    fi
  fi
fi

# Check if found id is already present
if [ -r "${RSPOOLER_SPOOLER}/.rspooler" ]; then
  USED_RSPOOLER_IDS=`cat "${RSPOOLER_SPOOLER}/.rspooler" | sed 's/^\([^:]*\):*\([^:]*\):*\([^:]*\):*\([^:]*\):*\(.*\)$/\4/'`
  MATCHED=`echo ${USED_RSPOOLER_IDS} | grep ${RSPOOLER_ID}`
  if [ -n "${MATCHED}" ]; then
    [ -n "${QUIET_MODE}" ] || echo "Error: guessed rspooler id already present. Please correct id in file ${RSPOOLER_SPOOLER}/.rspooler"
    exit 1
  fi 
fi

# Verbose output
if [ "${VERBOSE_MODE}" = "true" ]; then
  cat << EOF
  Results:
    remote host       = ${RSPOOLER_HOST}
    rspooler type     = ${RSPOOLER_TYPE}
    rspooler id       = ${RSPOOLER_ID}
    remote user       = ${RSPOOLER_USER}
    remote path       = ${RSPOOLER_PATH}
    spooler           = ${RSPOOLER_SPOOLER}
    label             = ${RSPOOLER_FM_ANCHOR_LABEL}
EOF
fi

# Emulate output
if [ "${EMULATE_MODE}" = "true" ]; then
  echo "Would write:"
  if [ -n "${RSPOOLER_USER}" ]; then  
    echo "${RSPOOLER_TYPE}:${RSPOOLER_USER}@${RSPOOLER_HOST}:${RSPOOLER_PATH}:${RSPOOLER_ID}:${RSPOOLER_FM_ANCHOR_LABEL} >> ${RSPOOLER_SPOOLER}/.rspooler"
  else
    echo "${RSPOOLER_TYPE}:${RSPOOLER_HOST}:${RSPOOLER_PATH}:${RSPOOLER_ID}:${RSPOOLER_FM_ANCHOR_LABEL} >> ${RSPOOLER_SPOOLER}/.rspooler"
  fi
else
  if [ -n "${RSPOOLER_USER}" ]; then  
    echo "${RSPOOLER_TYPE}:${RSPOOLER_USER}@${RSPOOLER_HOST}:${RSPOOLER_PATH}:${RSPOOLER_ID}:${RSPOOLER_FM_ANCHOR_LABEL}" >> "${RSPOOLER_SPOOLER}/.rspooler"
  else
    echo "${RSPOOLER_TYPE}:${RSPOOLER_HOST}:${RSPOOLER_PATH}:${RSPOOLER_ID}:${RSPOOLER_FM_ANCHOR_LABEL}" >> "${RSPOOLER_SPOOLER}/.rspooler"
  fi
fi
