#!/bin/bash

fail() {
    echo "ERROR: $1" >&2
    exit 1
}

log() {
    echo "INFO : $1"
}

create_cert() {
    # Variables that must be set in the environment:
    # _validity

    local _keystore="$1"
    local _alias="$2"
    local _dname="$3"
    local _owner="$4"
    local _password="$5"

    [ -z "${_validity}" ] && fail "Certificate validity must be set into environment."
    [ -z "${_keystore}" ] && fail "Keystore parameter cannot be empty."
    [ -z "${_alias}" ] && fail "Alias parameter cannot be empty."
    [ -z "${_dname}" ] && fail "Distinguished Name parameter cannot be empty."
    [ -z "${_owner}" ] && fail "Owner parameter cannot be empty."
    [ -z "${_password}" ] && fail "Password parameter cannot be empty."

    [ -f "${_keystore}" ] && fail "Keystore file '${_keystore}' already exists. Please remove it to continue."
    log "Generating key pair in keystore '${_keystore}'..."
    "${JAVA_HOME}/bin/keytool" -v \
    -genkey \
    -keyalg RSA \
    -alias "${_alias}" \
    -dname "${_dname}" \
    -keypass "${_password}" \
    -keystore "${_keystore}" \
    -validity "${_validity}" \
    -storepass "${_password}" || exit 1

    log "Setting ownership and permissions on keystore '${_keystore}'..."
    _chown "${_owner}" "${_keystore}" || exit 1
    chmod  "600" "${_keystore}" || exit 1
}

export_cert() {
    # Variables that must be set in the environment:

    local _keystore="$1"
    local _alias="$2"
    local _cert="$3"
    local _owner="$4"
    local _password="$5"

    [ -z "${_keystore}" ] && fail "Keystore parameter cannot be empty."
    [ -z "${_alias}" ] && fail "Alias parameter cannot be empty."
    [ -z "${_cert}" ] && fail "Certificate parameter cannot be empty."
    [ -z "${_owner}" ] && fail "Owner parameter cannot be empty."
    [ -z "${_password}" ] && fail "Password parameter cannot be empty."

    if [ -f "${_cert}" ] ; then
        log "Removing existing '${_cert}' file..."
        rm -f "${_cert}" || exit 1
    fi

    log "Exporting certificate from '${_keystore}' to '${_cert}'..."
    "${JAVA_HOME}/bin/keytool" -v \
              -export \
              -rfc \
              -alias "${_alias}" \
              -file "${_cert}" \
              -keystore "${_keystore}" \
              -storepass "${_password}" || exit 1

    log "Setting ownership and permissions on certificate '${_cert}'..."
    _chown "${_owner}" "${_cert}" || exit 1
    chmod  "600" "${_cert}" || exit 1
}

export_p12() {
    # Variables that must be set in the environment:

    local _keystore="$1"
    local _alias="$2"
    local _cert="$3"
    local _owner="$4"
    local _password="$5"

    [ -z "${_keystore}" ] && fail "Keystore parameter cannot be empty."
    [ -z "${_alias}" ] && fail "Alias parameter cannot be empty."
    [ -z "${_cert}" ] && fail "Certificate parameter cannot be empty."
    [ -z "${_owner}" ] && fail "Owner parameter cannot be empty."
    [ -z "${_password}" ] && fail "Password parameter cannot be empty."

    log "Exporting certificate from '${_keystore}' to '${_cert}'..."
    "${JAVA_HOME}/bin/keytool" -v \
    -importkeystore \
    -noprompt \
    -srckeystore "${_keystore}" \
    -srcstoretype "JKS" \
    -srcalias "${_alias}" \
    -destkeystore "${_cert}" \
    -deststoretype "PKCS12" \
    -destalias "${_alias}" \
    -srcstorepass "${_password}" \
    -srckeypass "${_password}" \
    -deststorepass "${_password}" \
    -destkeypass "${_password}" || exit 1

    log "Setting ownership and permissions on certificate '${_cert}'..."
    _chown "${_owner}" "${_cert}" || exit 1
    chmod  "600" "${_cert}" || exit 1
}

list_cert() {
    # Variables that must be set in the environment:

    local _keystore="$1"
    local _password="$2"

    [ -z "${_keystore}" ] && fail "Keystore parameter cannot be empty."
    [ -z "${_password}" ] && fail "Password parameter cannot be empty."

    log "Listing keystore '${_keystore}' contents..."
    "${JAVA_HOME}/bin/keytool" -v -list -keystore "${_keystore}" -storepass "${_password}" || exit 1
    log ""
    log "------------------------------------------------------------------------------------------"
}

trust_cert() {
    # Variables that must be set in the environment:

    local _cert_in="$1"
    local _alias_in="$2"
    local _truststore_out="$3"
    local _owner_out="$4"
    local _password="$5"

    [ -z "${_cert_in}" ] && fail "Input certificate parameter cannot be empty."
    [ -z "${_alias_in}" ] && fail "Input alias parameter cannot be empty."
    [ -z "${_truststore_out}" ] && fail "Output truststore parameter cannot be empty."
    [ -z "${_owner_out}" ] && fail "Output owner parameter cannot be empty."
    [ -z "${_password}" ] && fail "Password parameter cannot be empty."

    log "Importing certificate from '${_cert_in}' to '${_truststore_out}' truststore..."
    "${JAVA_HOME}/bin/keytool" -v \
                -import \
                -alias "${_alias_in}" \
                -file "${_cert_in}" \
                -keystore "${_truststore_out}" \
                -storepass "${_password}" \
                -noprompt || exit 1

    log "Setting ownership and permissions on truststore '${_truststore_out}'..."
    _chown "${_owner_out}" "${_truststore_out}" || exit 1
    chmod  "600" "${_truststore_out}" || exit 1

}

cross_trust() {
    # Variables that must be set in the environment:

    local _truststore1="$1"
    local _alias1="$2"
    local _cert1="$3"
    local _owner1="$4"
    local _password1="$5"
    local _truststore2="$6"
    local _alias2="$7"
    local _cert2="$8"
    local _owner2="$9"
    local _password2="$10"

    [ -z "${_truststore1}" ] && fail "First Truststore parameter cannot be empty."
    [ -z "${_alias1}" ] && fail "First Alias parameter cannot be empty."
    [ -z "${_cert1}" ] && fail "First Certificate parameter cannot be empty."
    [ -z "${_owner1}" ] && fail "First Owner parameter cannot be empty."
    [ -z "${_password1}" ] && fail "First Password parameter cannot be empty."
    [ -z "${_truststore2}" ] && fail "Second Truststore parameter cannot be empty."
    [ -z "${_alias2}" ] && fail "Second Alias parameter cannot be empty."
    [ -z "${_cert2}" ] && fail "Second Certificate parameter cannot be empty."
    [ -z "${_owner2}" ] && fail "Second Owner parameter cannot be empty."
    [ -z "${_password2}" ] && fail "Second Password parameter cannot be empty."

    trust_cert "${_cert1}" "${_alias1}" "${_truststore2}" "${_owner2}" "${_password2}"
    trust_cert "${_cert2}" "${_alias2}" "${_truststore1}" "${_owner1}" "${_password1}"
}

_chown() {
    if [ "$_cuid" == "0" ]; then
        chown "$@"
    fi
}

