#!/bin/bash

##################################################################################
# Copyright 2023-2025 by NI SP Software GmbH, All rights reserved.
# Copyright 1999-2023 by Nice, srl., All rights reserved.
#
# This software includes confidential and proprietary information
# of NI SP Software GmbH ("Confidential Information").
# You shall not disclose such Confidential Information
# and shall use it only in accordance with the terms of
# the license agreement you entered into with NI SP Software.
################################################################################

[ -z "$1" ] && exit 2

main() {
  _use_systemd="$(which systemctl >/dev/null 2>&1 && echo true || echo false)"
  case "$1" in
    check)
      check
      ;;
    install)
      install
      ;;
    uninstall)
      uninstall
      ;;
    *)
      echo "Unrecognized option: $1"
      exit 1
      ;;
  esac
}

check() {
  if [ "${_use_systemd}" == "true" ]; then
    check_systemd
  else
    check_chkconfig
  fi
}

check_systemd() {
  if [ -f "/etc/os-release" ]; then
    # SLE 12 sp1 and also SLES15 (https://www.suse.com/releasenotes/x86_64/SUSE-SLES/15/#fate-324409)
    systemctl is-enabled enginframe >/dev/null 2>&1
    _check=$?
  elif [ -f "/etc/redhat-release" ]; then
    systemctl is-enabled enginframe >/dev/null 2>&1
    _check=$?
  else
    echo "Linux Distribution not identified."
    exit 2
  fi   

  if [ ${_check} = 0 ]; then
      echo "EnginFrame is set to startup at boot."
  else
      echo "EnginFrame is NOT set to startup at boot."
  fi
  exit ${_check}
}

check_chkconfig() {
  if [ -f "/etc/SuSE-release" -o -f "/etc/os-release" ]; then
    # In SLES15 we are missing /etc/SuSE-release (https://www.suse.com/releasenotes/x86_64/SUSE-SLES/15/#fate-324409)
    chkconfig --check enginframe 35
    _check=$?
  elif [ -f "/etc/redhat-release" -o -f "/etc/system-release" ]; then
    chkconfig --list enginframe
    _check=$?
  else
    echo "Linux Distribution not identified."
    exit 2
  fi

  if [ ${_check} = 0 ]; then
      echo "EnginFrame is set to startup at boot."
  else
      echo "EnginFrame is NOT set to startup at boot."
  fi
  exit ${_check}
}

install() {
  if [ "${_use_systemd}" == "true" ]; then
    install_systemd
  else
    install_chkconfig
  fi
}

install_systemd() {
  _temp_script=$(mktemp '/tmp/enginframe.service.sh.XXXXXXXXXX')
  cat > "${_temp_script}" << EOF
##################################################################################
# Copyright 2023-2025 by NI SP Software GmbH, All rights reserved.
# Copyright 1999-2023 by Nice, srl., All rights reserved.
#
# This software includes confidential and proprietary information
# of NI SP Software GmbH ("Confidential Information").
# You shall not disclose such Confidential Information
# and shall use it only in accordance with the terms of
# the license agreement you entered into with NI SP Software.
################################################################################

[Unit]
Description=NI SP EF Portal (http://www.ni-sp-software.com)
After=local-fs.target network.target remote-fs.target 

# Uncomment following requirement in case needed, setting custom mount point to be checked
#RequiresMountsFor=<mount point for the EF filesystem>

[Service]
Type=forking
TimeoutStartSec=0
TimeoutStopSec=0
ExecStart=${EF_TOP}/bin/enginframe --conf ${EF_CONF} start
ExecStop=${EF_TOP}/bin/enginframe --conf ${EF_CONF} stop
Restart=on-failure
RestartSec=4s
SuccessExitStatus=143

[Install]
WantedBy=multi-user.target

EOF

  rm -f /usr/lib/systemd/system/enginframe.service
  cp -f "${_temp_script}" /usr/lib/systemd/system/enginframe.service
  rm -f "${_temp_script}"
  systemctl enable enginframe >/dev/null 2>&1
  systemctl daemon-reload
  check
}

install_chkconfig() {
  _temp_script=$(mktemp '/tmp/enginframe.service.sh.XXXXXXXXXX')
  cat > "${_temp_script}" << EOF
#!/bin/bash

##################################################################################
# Copyright 2023-2025 by NI SP Software GmbH, All rights reserved.
# Copyright 1999-2023 by Nice, srl., All rights reserved.
#
# This software includes confidential and proprietary information
# of NI SP Software GmbH ("Confidential Information").
# You shall not disclose such Confidential Information
# and shall use it only in accordance with the terms of
# the license agreement you entered into with NI SP Software.
################################################################################

################################################################################
# The following is for the Linux chkconfig utility
# chkconfig: 35 99 01 
# description: NI SP EF Portal
################################################################################


#
# http://refspecs.linuxbase.org/LSB_3.1.0/LSB-Core-generic/LSB-Core-generic/tocsysinit.html
#
### BEGIN INIT INFO
# Provides:          enginframe
# Default-Start:     3 5
# Default-Stop:      0 1 2 4 6
# Required-Start:    \$local_fs \$network
# Required-Stop:     \$local_fs \$network
# Short-Description: NI SP EF Portal
# Description:       NI SP EF Portal (http://www.ni-sp-software.com)
### END INIT INFO

'${EF_TOP}/bin/enginframe' --conf '${EF_CONF}' "\$@"

EOF

  rm -f /etc/init.d/enginframe
  cp -f "${_temp_script}" /etc/init.d/enginframe
  chmod +rx /etc/init.d/enginframe
  rm -f "${_temp_script}"
  chkconfig --del enginframe
  chkconfig --add enginframe
  check
}

uninstall() {
  if [ "${_use_systemd}" == "true" ]; then
    uninstall_systemd
  else
    uninstall_chkconfig
  fi
}

uninstall_systemd() {
  systemctl disable enginframe
  rm -f /usr/lib/systemd/system/enginframe.service
  systemctl daemon-reload
  check
}

uninstall_chkconfig() {
  chkconfig --del enginframe
  rm -f /etc/init.d/enginframe
  check
}

main "$@"
