#!/bin/bash

# Variables that must be set in the environment:
# JAVA_HOME

. "$(dirname $0)/certificates-utils.sh"

# Files name
_tomcat_keystore="tomcat.keystore"
_tomcat_cert="tomcat.cert"
_tomcat_truststore="tomcat.truststore"
_agent_keystore="agent.keystore"
_agent_cert="agent.cert"
_agent_truststore="agent.truststore"
_client_keystore="client.keystore"
_client_cert="client.cert"
_client_cert_p12="client.cert.p12"
_client_truststore="client.truststore"
_ef_default_alias="ef"
_client_default_alias="my"
_ef_keystore_password="ef.keystore.password"
_client_keystore_password="client.keystore.password"


# Constants
_OU="EF Portal"
_O="NI SP Software GmbH"
_L="Tuebingen"
_S="Tuebingen"
_C="DE"

# 10 year validity
_validity=3650

fail() {
    echo "ERROR: $1" >&2
    exit 1
}

log() {
    echo "INFO : $1"
}

syntax() {
    _cmd=$(basename "$0")
    echo "Syntax"
    echo "  $_cmd generate"
    echo "      Generates self-signed certificate for EF Tomcat Server into the EF_CONF_ROOT/tomcat/conf/certs/ folder."
    echo "      The EF certificate password is read from the EF_CONF_ROOT/tomcat/conf/certs/${_ef_keystore_password} file."
    echo "      The Client certificate password is read from the EF_CONF_ROOT/tomcat/conf/certs/${_client_keystore_password} file."
    echo ""
    echo "      -"
    echo "          The EF certificate password is provided as standard input,"
    echo "          and will be stored into a ${_ef_keystore_password} file in the same location as the certificate file."
    echo ""
    echo "      --tomcatuser user, --tomcatuser=user"
    echo "          Specify the user running Tomcat"
    echo "          (needed to set right permission on the certificates)."
    echo ""
    echo "      --ef-alias alias, --ef-alias=alias"
    echo "          Permit to create a EF Tomcat server certificate with the specified alias."
    echo "          If not specified, \"${_ef_default_alias}\" value will be used."
    echo ""
    echo "      --ef-common-name hostname, --ef-common-name=hostname"
    echo "          Permit to create a EF Tomcat server certificate with the specified common name."
    echo "          The value will be set into the CN field of the DN server certificate string."
    echo "          If not specified, --ef-alias parameter value will be used."
    echo ""
    echo "      --client-auth"
    echo "          Generates self-signed certificate (in JKS and PKCS12 format) for a client into the EF_CONF_ROOT/tomcat/conf/certs/ folder."
    echo "          This certificate will be imported into the EF Tomcat Server truststore file."
    echo "          Useful for the EnginFrame certificate authority, when Tomcat connector require client authentication."
    echo ""
    echo "      --client-aliases 'clienthost1 clienthost2 ...', --client-aliases='clienthost1 clienthost2 ...'"
    echo "          Used together with the --client-auth option, permit to create a certificate for every specified client alias."
    echo "          If not specified, one certificate with alias \"${_client_default_alias}\" will be used."
    echo "          The alias will be also set into the CN field of the DN client certificate string."
    echo "          Useful for the EnginFrame certificate authority with authorization.certificate.userCertificate option set to true."
    echo ""
    echo "  $_cmd list"
    echo "      List certificates in the EF_CONF_ROOT/tomcat/conf/certs/ folder."
    echo ""
    echo "  $_cmd clean"
    echo "      Removes all certificates except the ${_ef_keystore_password} and ${_client_keystore_password} files. Use with care!!!"
    echo ""
    echo ""
    echo "Examples"
    echo "  echo \"myPassword\" | $_cmd - generate --tomcatuser efnobody"
    echo "  echo \"myPassword\" | $_cmd - generate --tomcatuser efnobody --client-auth --client-aliases efadmin"
    echo "  $_cmd generate --tomcatuser=efnobody"
    echo "  $_cmd generate --tomcatuser=efnobody --client-auth"
    echo "  $_cmd generate --tomcatuser=efnobody --client-auth --ef-alias=enginframe --client-aliases='client1 client2'"
    echo "  $_cmd list"
    echo "  $_cmd clean"
    echo ""
}

generate() {
    # Variables that must be set in the environment:
    # _tomcatuser
    # _ef_cert_folder
    # _ef_keystore_password_file
    # _client_keystore_password_file
    # _ef_password
    # _validity
    # _tomcat_keystore

    [ -z "${_tomcatuser}" ] && fail "TOMCAT_USER must be set into environment."
    [ -z "${_ef_cert_folder}" ] && fail "EF certificate folder must be set into environment."
    [ -z "${_ef_keystore_password_file}" ] && fail "EF keystore password file must be set into environment."
    [ -z "${_client_keystore_password_file}" ] && fail "Client keystore password file must be set into environment."
    [ -z "${_ef_password}" ] && fail "Password must be set into environment."
    [ -z "${_validity}" ] && fail "Certificate validity must be set into environment."
    [ -z "${_tomcat_keystore}" ] && fail "EF tomcat keystore name must be set into environment."

    mkdir -p "${_ef_cert_folder}" || fail "Unable to create ${_ef_cert_folder}."

    echo "${_ef_password}" > "${_ef_keystore_password_file}" || fail "Unable to write to \"${_ef_keystore_password_file}\"."
    log "EF keystore password file created: ${_ef_keystore_password_file}"
    chmod 600 "${_ef_keystore_password_file}" || fail "Unable to chmod \"${_ef_keystore_password_file}\"."

    # Create the EF Tomcat Private Keys and Public Certificates
    #for _ef_host in ${_ef_hosts}; do
        _ef_tomcat_keystore="${_ef_cert_folder}/${_ef_alias}.${_tomcat_keystore}"
        _ef_tomcat_dname="CN=${_ef_common_name},OU=${_OU},O=${_O},L=${_L},S=${_S},C=${_C}"
        create_cert "${_ef_tomcat_keystore}" "${_ef_alias}" "${_ef_tomcat_dname}" "${_tomcatuser}" "${_ef_password}"
    #done

    if [ "${_client_auth}" = "true" ]; then
        # Variables that must be set in the environment:
        # _tomcat_cert
        # _tomcat_truststore
        # _agent_cert
        # _agent_truststore
        # _client_keystore
        # _client_cert
        # _client_truststore
        # _client_password

        [ -z "${_tomcat_cert}" ] && fail "EF tomcat certificate name must be set into environment."
        [ -z "${_tomcat_truststore}" ] && fail "EF tomcat truststore name must be set into environment."
        [ -z "${_agent_cert}" ] && fail "EF agent certificate name must be set into environment."
        [ -z "${_agent_truststore}" ] && fail "EF agent truststore name must be set into environment."
        [ -z "${_client_keystore}" ] && fail "Client keystore name must be set into environment."
        [ -z "${_client_cert}" ] && fail "Client certificate name must be set into environment."
        [ -z "${_client_truststore}" ] && fail "Client truststore name must be set into environment."
        [ -z "${_client_password}" ] && fail "Password must be set into environment."

        echo "${_client_password}" > "${_client_keystore_password_file}" || fail "Unable to write to \"${_client_keystore_password_file}\"."
        log "Client keystore password file created: ${_client_keystore_password_file}"
        chmod 600 "${_client_keystore_password_file}" || fail "Unable to chmod \"${_client_keystore_password_file}\"."

        _ef_tomcat_keystore="${_ef_cert_folder}/${_ef_alias}.${_tomcat_keystore}"
        _ef_tomcat_cert="${_ef_cert_folder}/${_ef_alias}.${_tomcat_cert}"
        export_cert "${_ef_tomcat_keystore}" "${_ef_alias}" "${_ef_tomcat_cert}" "${_tomcatuser}" "${_ef_password}"

        # Create the EF Agent Private Keys and Public Certificates
        _ef_agent_keystore="${_ef_cert_folder}/${_ef_alias}.${_agent_keystore}"
        _ef_agent_dname="CN=${_ef_common_name},OU=${_OU},O=${_O},L=${_L},S=${_S},C=${_C}"
        create_cert "${_ef_agent_keystore}" "${_ef_alias}" "${_ef_agent_dname}" "root" "${_ef_password}"
        _ef_agent_cert="${_ef_cert_folder}/${_ef_alias}.${_agent_cert}"
        export_cert "${_ef_agent_keystore}" "${_ef_alias}" "${_ef_agent_cert}" "root" "${_ef_password}"

        # Create the Client Private Keys and Public Certificates
        for _client_alias in ${_client_aliases}; do
            _ef_client_keystore="${_ef_cert_folder}/${_client_alias}.${_client_keystore}"
            _ef_client_dname="CN=${_client_alias},OU=${_OU},O=${_O},L=${_L},S=${_S},C=${_C}"
            _ef_client_cert="${_ef_cert_folder}/${_client_alias}.${_client_cert}"
            create_cert "${_ef_client_keystore}" "${_client_alias}" "${_ef_client_dname}" "root" "${_client_password}"
            export_cert "${_ef_client_keystore}" "${_client_alias}" "${_ef_client_cert}" "root" "${_client_password}"

            # Export client Certificate into p12 format (to be used from browser)
            _ef_client_cert_p12="${_ef_cert_folder}/${_client_alias}.${_client_cert_p12}"
            export_p12 "${_ef_client_keystore}" "${_client_alias}" "${_ef_client_cert_p12}" "root" "${_client_password}"
        done

        # Build a cross trust between EF tomcat/agent and EF tomcat/client
        _ef_tomcat_truststore="${_ef_cert_folder}/${_ef_alias}.${_tomcat_truststore}"
        _ef_tomcat_cert="${_ef_cert_folder}/${_ef_alias}.${_tomcat_cert}"

        _ef_agent_truststore="${_ef_cert_folder}/${_ef_alias}.${_agent_truststore}"
        _ef_agent_cert="${_ef_cert_folder}/${_ef_alias}.${_agent_cert}"

        # cross_trust "${_ef_agent_truststore}" "${_ef_alias}" "${_ef_agent_cert}" "root" "${_ef_tomcat_truststore}" "${_ef_alias}" "${_ef_tomcat_cert}" "${_tomcatuser}"
        trust_cert "${_ef_agent_cert}" "${_ef_alias}" "${_ef_tomcat_truststore}" "${_tomcatuser}" "${_ef_password}"
        trust_cert "${_ef_tomcat_cert}" "${_ef_alias}" "${_ef_agent_truststore}" "root" "${_ef_password}"

        for _client_alias in ${_client_aliases}; do
            _ef_client_truststore="${_ef_cert_folder}/${_client_alias}.${_client_truststore}"
            _ef_client_cert="${_ef_cert_folder}/${_client_alias}.${_client_cert}"

            # cross_trust "${_ef_client_truststore}" "${_client_alias}" "${_ef_client_cert}" "root" "${_client_password}" "${_ef_tomcat_truststore}" "${_ef_alias}" "${_ef_tomcat_cert}" "${_tomcatuser}" "${_ef_password}"
            trust_cert "${_ef_client_cert}" "${_client_alias}" "${_ef_tomcat_truststore}" "${_tomcatuser}" "${_ef_password}"
            trust_cert "${_ef_tomcat_cert}" "${_ef_alias}" "${_ef_client_truststore}" "root" "${_client_password}"
        done
    fi
}

init() {
    _cuid="$(id -u)"

    # script location is EF_ROOT/tools
    local -x EF_CONF_ROOT="$(readlink -f "$(dirname "$0")/../../../conf")"
    [ -d "${EF_CONF_ROOT}/tomcat/conf" ] || fail "Could not find a valid EF_CONF_ROOT/tomcat/conf folder. Please check script location."

    [ -x "${JAVA_HOME}/bin/java" ] || fail "Please specify a valid JAVA_HOME environment variable: '${JAVA_HOME}/bin/java' not found."
    [ -x "${JAVA_HOME}/bin/keytool" ] || fail "Please specify a valid JAVA_HOME environment variable: '${JAVA_HOME}/bin/keytool' not found."

    _ef_cert_folder="${EF_CONF_ROOT}/tomcat/conf/certs"
    _ef_keystore_password_file="${_ef_cert_folder}/${_ef_keystore_password}"
    _client_keystore_password_file="${_ef_cert_folder}/${_client_keystore_password}"
}

parse_options() {
    _command=''
    _input_passwd=false
    _tomcatuser=''
    _ef_alias=''
    _ef_common_name=''
    _client_aliases=''
    _client_auth=false
    _client_password=''

    if [ $# -eq 0 ]; then
        syntax
        exit 0
    fi

    while [ $# -gt 0 ] ; do
        case "$1" in
            generate|list|clean)
               [ -n "${_command}" ] && fail "Only one command at a time can be used."
                _command="$1"
            ;;
            -)
                _input_passwd=true
            ;;
            --ef-alias)
                _ef_alias="$2"
                shift
            ;;
            --ef-alias=*)
                _ef_alias="${1#*=}"
            ;;
            --ef-common-name)
                _ef_common_name="$2"
                shift
            ;;
            --ef-common-name=*)
                _ef_common_name="${1#*=}"
            ;;
            --client-aliases)
                _client_aliases="$2"
                shift
            ;;
            --client-aliases=*)
                _client_aliases="${1#*=}"
            ;;
            --client-auth)
                _client_auth=true
            ;;
            --client-password)
                _client_password="$2"
                shift
            ;;
            --client-password=*)
                _client_password="${1#*=}"
            ;;
            --tomcatuser)
                _tomcatuser="$2"
                shift
            ;;
            --tomcatuser=*)
                _tomcatuser="${1#*=}"
            ;;
            -h|--help|help)
                syntax
                exit 0
            ;;
            *)
                syntax
                fail "Unrecognized option '$1'"
            ;;
        esac
        shift
    done
}

check_options() {
    if [ "${_command}" = "generate"  -o "${_command}" = "list" ]; then
        if [ "${_input_passwd}" = "true" ]; then
            _ef_password=$(cat -)
        else
            if [ -f "${_ef_keystore_password_file}" ]; then
                _ef_password=$(cat "${_ef_keystore_password_file}" | head -n 1)
            else
                fail "${_ef_keystore_password_file} file does not exist, and EF certificates password cannot be read. Use - to specify the password from STDIN."
            fi
        fi
        [ "${#_ef_password}" -lt 6 ] && fail "EF key password must be at least 6 characters"
    fi

    if [ "${_command}" = "list" ]; then
        if ls "${_ef_cert_folder}/"*"${_client_keystore}" "${_ef_cert_folder}/"*"${_client_truststore}" 1> /dev/null 2>&1; then
            if [ -z "${_client_password}" ]; then
                if [ -f "${_client_keystore_password_file}" ]; then
                    _client_password=$(cat "${_client_keystore_password_file}" | head -n 1)
                else
                    fail "${_client_keystore_password_file} file does not exist, and Client certificates password cannot be read. Use --client-password option to specify the password."
                fi
            fi
        fi
    fi

    if [ "${_command}" = "generate" ]; then
        [ -z "${_tomcatuser}" ] && fail "Please specify --tomcatuser option with valid TOMCAT_USER user as value."
        id "${_tomcatuser}"  > /dev/null 2>&1 || fail "Please specify a valid TOMCAT_USER user: '${_tomcatuser}' not found."

        [ -n "${_client_aliases}" -a "${_client_auth}" = "false" ] && fail "Client hosts can be specified only together with the --client-auth option."
        [ -n "${_client_password}" -a "${_client_auth}" = "false" ] && fail "Client password can be specified only together with the --client-auth option."

        if [ "${_client_auth}" = "true" ]; then
            if [ -z "${_client_password}" ]; then
                if [ -f "${_client_keystore_password_file}" ]; then
                    _client_password=$(cat "${_client_keystore_password_file}" | head -n 1)
                else
                    fail "${_client_keystore_password_file} file does not exist, and Client certificates password cannot be read. Use --client-password option to specify the password."
                fi
            fi
            [ "${#_client_password}" -lt 6 ] && fail "Client key password must be at least 6 characters"
        fi

        [ -z "${_ef_alias}" ] && _ef_alias="${_ef_default_alias}" # Setting default ef alias
        [ -z "${_client_aliases}" ] && _client_aliases="${_client_default_alias}" # Setting default client alias
        [ -z "${_ef_common_name}" ] && _ef_common_name="${_ef_alias}" # Setting default certificate common name
    fi
}

do_command() {
    case "${_command}" in
        clean)
            read -p "All certificates, keystore and trustore files are going to be removed. The ${_ef_keystore_password} and ${_client_keystore_password} files will NOT be removed. Do you want to continue? [y/N] " -n 1 -r
            echo
            if [[ $REPLY =~ ^[Yy]$ ]]
            then
                for _cert in $(ls "${_ef_cert_folder}/"*"${_tomcat_keystore}" \
                             "${_ef_cert_folder}/"*"${_tomcat_truststore}" \
                             "${_ef_cert_folder}/"*"${_tomcat_cert}" \
                             "${_ef_cert_folder}/"*"${_agent_keystore}" \
                             "${_ef_cert_folder}/"*"${_agent_truststore}" \
                             "${_ef_cert_folder}/"*"${_agent_cert}" \
                             "${_ef_cert_folder}/"*"${_client_keystore}" \
                             "${_ef_cert_folder}/"*"${_client_truststore}" \
                             "${_ef_cert_folder}/"*"${_client_cert_p12}" \
                             "${_ef_cert_folder}/"*"${_client_cert}" 2>/dev/null); do
                    [ -f "${_cert}" ] && rm "${_cert}" > /dev/null 2>&1
                done
            fi
        ;;
        generate)
            generate
        ;;
        list)
            for _cert in $(ls "${_ef_cert_folder}/"*"${_tomcat_keystore}" \
                         "${_ef_cert_folder}/"*"${_tomcat_truststore}" 2>/dev/null); do
                [ -f "${_cert}" ] && list_cert "${_cert}" "${_ef_password}"
            done
            for _cert in $(ls "${_ef_cert_folder}/"*"${_client_keystore}" \
                         "${_ef_cert_folder}/"*"${_client_truststore}" 2>/dev/null); do
                [ -f "${_cert}" ] && list_cert "${_cert}" "${_client_password}"
            done
        ;;
    esac
}

main() {
    init
    parse_options "$@"
    check_options
    do_command
}

main "$@"
