#! /bin/bash

#
################################################################################
# Copyright 2023-2025 by NI SP Software GmbH, All rights reserved.
# Copyright 1999-2023 by Nice, srl., All rights reserved.
#
# This software includes confidential and proprietary information
# of NI SP Software GmbH ("Confidential Information").
# You shall not disclose such Confidential Information
# and shall use it only in accordance with the terms of
# the license agreement you entered into with NI SP Software.
################################################################################
#################################################################################



# ---[ CONFIGURATION ]------------------------------------------------------- #


_derby_start_wait_time_seconds=5
_derby_stop_wait_time_seconds=30
_is_pid_mandatory=true
_default_derby_host='127.0.0.1'
IFS=$'\n'



# ---[ PRIVATE METHODS ]----------------------------------------------------- #


check_requirements() {
    if [ -z "${BASH_VERSINFO[0]}" ]; then
        die 'Wrong environment: missing variable BASH_VERSINFO. Bash too old?'
    fi

    if [ "${BASH_VERSINFO[0]}" -lt 3 ]; then
        die "Bash too old: minimum version required is 3"
    fi

    compgen 2>/dev/null || die 'Missing mandatory program: compgen'

    AWK="$(find_awk)"

    if [ -z "${AWK}" ]; then
        die 'Unable to find a suitable version of AWK'
    fi

    return 0
}


find_awk() {
    local _awk_test_code_1='function fl(s){gsub(/f/,"F",s);gensub(/o/,"F","g",s);return tolower(s)} BEGIN {print fl("foo")}'
    local _awk_test_code_2='BEGIN{print ENVIRON["FOO"]}'
    local _awk=''
    local _awk_output=''

    for _awk in \
        "$(which gawk 2>/dev/null)" \
        "$which nawk 2>/dev/null)" \
        '/usr/toolbox/nawk' \
        '/usr/bin/awk' \
        ;
        do

        if [ -z "${_awk}" ]; then
            continue
        fi

        if [ ! -x "${_awk}" ]; then
            continue
        fi

        _awk_output=`"${_awk}" "${_awk_test_code_1}" 2>/dev/null`
        if [ "${_awk_output}" != 'foo' ]; then
            continue
        fi
        _awk_output=`env FOO=foo "${_awk}" "${_awk_test_code_2}" 2>/dev/null`
        if [ "${_awk_output}" != 'foo' ]; then
            continue
        fi

        echo "${_awk}"
        return 0
    done

    return 1
}


die() {
    echo "[ERROR] $1" >&2
    exit 1
}


verbose() {
    if [ "${_verbose}" = true ]; then
        echo "$1"
    fi
}


get_pid() {
    if [ -z "${_pid_file}" ]; then
        return 0
    fi

    if [ ! -e "${_pid_file}" ]; then
        return 0
    fi

    if [ ! -r "${_pid_file}" ]; then
        die "Unable to read pid file ${_pid_file}"
    fi

    cat "${_pid_file}"
    return 0
}

parse_properties() {

    "${AWK}" '
        function trim(s) {
            sub(/^ +/, "", s)
            sub(/ +$/, "", s)
            return s
        }

        BEGIN {
            FS="="
        }

        /^ *$/ {
            next
        }

        /^ *#/ { 
            next
        }

        {
            property = $0

            eq = index(property, "=")

            if (eq <= 0) {
                next
            }

            name  = trim(substr(property, 1, eq-1))
            value = trim(substr(property, eq+1))

            print "-D" name "=" value
        }
        '
}


# Let's start Derby DB
#
do_start() {
    if [ "${DERBY_ENABLED}" != 'true' ]; then
        return 0
    fi

    _pid=$(get_pid)
    _exit_code="$?"

    if [ "${_exit_code}" != 0 ]; then
        return "${_exit_code}"
    fi

    if [ -n "${_pid}" ]; then
        verbose "Checking existing PID ${_pid}..."
        check_pid "${_pid}"
        _exit_code="$?"

        # Derby running!
        if [ "${_exit_code}" = 0 ]; then
            die "Derby seems already started (process with PID: ${_pid})"
        fi

        verbose "Process already killed, cleaning PID file..."

        # Remote the pid FILE
        rm -f "${_pid_file}"
    fi

    local -a _cmd=()

    _cmd+=("${JAVA_HOME}/bin/java")
    _cmd+=("-Dderby.system.home=${DERBY_SYSTEM_HOME}")
    _cmd+=("-Dderby.stream.error.file=${_log_file}")

    if [ -n "${DERBY_JAVA_PROPERTIES}" ]; then
        _old_IFS="${IFS}"
        IFS=$'\n'
        for _property in $(echo "${DERBY_JAVA_PROPERTIES}" | parse_properties); do
            if [ -n "${_property}" ]; then
                _cmd+=("${_property}")
            fi
        done
        IFS="${_old_IFS}"
    fi

    _cmd+=('-jar' "${_derby_jar}")
    _cmd+=('server')
    _cmd+=('start')
    _cmd+=('-p' "${DERBY_PORT}")
    _cmd+=('-h' "${DERBY_HOST}")

    "${_cmd[@]}" >"${_out_file}" 2>"${_err_file}" &
    _exit_code="$?"

    _pid="$!"
    echo "${_pid}" > "${_pid_file}"

    _success=false

    if [ "$_exit_code" = "0" ] ; then
        _derby_pid=$(cat "${_pid_file}")
        echo "Starting Derby DB [PID: ${_derby_pid}]..."
    else
        die 'Unable to start Derby Database.'
    fi
 
    echo "Checking Derby is starting..."

    ## Wait some time the background process to start
    #sleep "${_derby_start_wait_time_seconds}"
    _i=0

    while [ "${_i}" -lt "${_derby_start_wait_time_seconds}" ]; do
        sleep 1

        check_pid "${_pid}"
        _exit_code="$?"

        if [ "${_exit_code}" = "0" ] ; then
            _success=true
            break
        fi
        let _i='_i + 1'
    done

    case "${_success}" in
        true)
            echo "[OK] Derby Database started"
            return 0
            ;;
        false)
            echo "[ERROR] Unable to start Derby Database. Check port ${DERBY_PORT} is available."
            echo "derby.out:"
            tail "${_out_file}"
            echo "derby.err:"
            tail "${_err_file}"
            return 1
            ;;
    esac

    return 0
}


do_restart() {
    do_stop
    do_start
}


get_derby_info() {
    _userpass_pattern='^derby\.user\.\([^=]\{1,\}\)\ *=\ *\(.*\)\ *$'
    _derby_config_file="${DERBY_SYSTEM_HOME}/derby.properties"

    case "$1" in
        user|username)
            sed -n -e "s/${_userpass_pattern}/\1/p" "${_derby_config_file}" | head -1
            ;;
        pass|password)
            sed -n -e "s/${_userpass_pattern}/\2/p" "${_derby_config_file}" | head -1
            ;;
    esac
}



do_ping() {
    local _user=''
    local _password=''
    local _exit_code=0

    while [ "$#" -gt 0 ]; do
        case "$1" in
            -u|--user)
                _user="$2"
                shift
                ;;
            -p|--password)
                _password="$2"
                shift
                ;;
        esac
        shift
    done

    if [ -z "${_user}" ]; then
        _user=$(get_derby_info user)
    fi

    if [ -z "${_password}" ]; then
        _password=$(get_derby_info password)
    fi


    "${JAVA_HOME}/bin/java" \
        -Dderby.system.home="${DERBY_SYSTEM_HOME}" \
        -Dderby.stream.error.file="${_log_file}" \
        -jar "${_derby_jar}" \
        server \
        ping \
        -user "${_user}" \
        -password "${_password}" \
        -p "${DERBY_PORT}" \
        -h "${DERBY_HOST}"

    _exit_code=$?

    return ${_exit_code}
}


get_process_info() {
    local _pid="$1"
    local _exit_code=0

    if [ -z "${_pid}" ]; then
        return 1
    fi

    ps -o "user,pid,ppid,pcpu,pmem,stime,time,args" "${_pid}"
    _exit_code=$?

    return ${_exit_code}
}


# Reports the status of Derby DB
#
do_status() {
    if [ "${DERBY_ENABLED}" != 'true' ]; then
        return 0
    fi

    local _user=$(get_derby_info user)
    local _password=$(get_derby_info password)
    local _pid=$(get_pid)

    check_pid "${_pid}" && _is_running=true || _is_running=false

    # If Derby is running, let us retrieve some better informations
    if [ "${_is_running}" = true ]; then

        # Full PS output
        _ps_out=$(get_process_info "${_pid}" 2>/dev/null)

        # Ping output
        _ping_out=$(do_ping -u "${_user}" -p "${_password}" 2>&1)
        _exit_code=$?

        # First level, check by exit code
        if [ "${_exit_code}" = 0 ]; then
            _is_running=true
        else
            _is_running=false
        fi
 
        if [ -n "${_ping_out}" -a "${_is_running}" = true ]; then
            # Check by parsing output
            echo "${_ping_out}" | fgrep -q 'Connection refused' >/dev/null 2>/dev/null && _is_running=false
        fi
    fi

    echo ''
    echo '---- Derby process Information ----'
    echo "Pid: ${_pid}"
    echo "${_ps_out}"
    echo ''
    echo '---- Derby network Information ----'
    echo "Port: ${DERBY_PORT}"
    echo ''
    echo 'Output of ping command:'
    echo "${_ping_out}" | sed -e 's/^/    /'
    echo ''

    case "${_is_running}" in
        true)
            echo "Derby is running."
            ;;
        false)
            echo "Derby is not responding. Check the above message."
            ;;
    esac
}


check_pid() {
    if [ -z "${1}" ]; then
        return 1
    fi

    kill -0 "${1}" >/dev/null 2>/dev/null

    return $?
}


# Stop Derby DB
#
do_stop() {
    if [ "${DERBY_ENABLED}" != 'true' ]; then
        return 0
    fi

    if [ -r "${_pid_file}" ]; then
        _pid=$(cat "${_pid_file}")
        check_pid "${_pid}" && _alive=true || _alive=false

        case "${_alive}" in
            true)
                verbose "Trying to stop Derby with PID ${_pid} on port ${DERBY_PORT}..."
                ;;
            false)
                if [ "${_is_pid_mandatory}" = true ]; then
                    echo '[OK] Derby already stopped (accordingly to PID file)'
                    return 0
                else
                    # Try to kill anyway
                    verbose "Trying to stop Derby on port ${DERBY_PORT}"
                fi
                ;;
        esac
    else
        if [ "${_is_pid_mandatory}" = true ]; then
            echo '[OK] Derby already stopped'
            return 0
        fi
        verbose "Trying to stop Derby on port ${DERBY_PORT}"
    fi

    _user=$(get_derby_info user)
    _password=$(get_derby_info password)

    "${JAVA_HOME}/bin/java" \
        -Dderby.system.home="${DERBY_SYSTEM_HOME}" \
        -Dderby.stream.error.file="${_log_file}" \
        -jar "${_derby_jar}" \
        server \
        shutdown \
        -user "${_user}" \
        -password "${_password}" \
        -p "${DERBY_PORT}" \
        -h "${DERBY_HOST}" \
    </dev/null
    _exit_code=$?

    if [ -n "${_pid}" ]; then
        _i=0
        _success=false
        while [ "${_i}" -lt "${_derby_stop_wait_time_seconds}" ]; do
            let _i='_i + 1'

            check_pid "${_pid}"
            _exit_code="$?"

            if [ "${_exit_code}" = "0" ] ; then
                # Still alive, check again later
                echo -n '.'
                sleep 1
            else
                echo ''
                _success=true
                break
            fi
        done
    else
        # Assuming kill success...
        _success=true
    fi

    case "${_success}" in
        true)
            echo 'Derby terminated'

            if [ -n "${_pid}" ]; then
                rm -f "${_pid_file}"
            fi
            ;;
        false)
            echo 'Unable to kill Derby.'
            ;;
    esac
}


get_var_value() {
    echo "${1#*=}"
}


env_var() {
    if [ -z "$1" ]; then
        return 0
    fi

    # Be sure the variable is defined (and valid)
    if [ "$(compgen -v "$1")" != "$1" ]; then
        return 1
    fi

    eval echo "\"\$${1}\""
}


align_to_title() {
    "${AWK}" -v title="$1" '
        BEGIN {
            space = title
            gsub(/./, " ", space)
        }
        {
            if (NR == 1) {
                print title $0
            } else {
                print space $0
            }
        }
        '
}


do_dump_config() {
    echo 'Derby configuration:'
    echo "  Java HOME:       ${JAVA_HOME}"
    echo "  Dist home:       ${DERBY_DIST_HOME}"
    echo "  System home:     ${DERBY_SYSTEM_HOME}"
    echo "  OS user:         ${DERBY_OS_USER}"
    echo "  Log dir:         ${DERBY_LOGDIR}"
    echo "  HTTP port:       ${DERBY_PORT}"
    echo "  Is enabled:      ${DERBY_ENABLED}"

    if [ -n "${DERBY_JAVA_PROPERTIES}" ]; then
        echo "${DERBY_JAVA_PROPERTIES}" \
        | parse_properties \
        | "${AWK}" -v title='  Java properties: ' '
            BEGIN {
                space = title
                gsub(/./, " ", space)
            }
            {
                if (NR == 1) {
                    print title $0
                } else {
                    print space $0
                }
            }
            '
    fi
    echo ''

}

# Check configuration or exit in case of problems
#
check_config() {

    # Validate mandatory parameters
    for _v_name in 'JAVA_HOME' 'DERBY_DIST_HOME' 'DERBY_SYSTEM_HOME' 'DERBY_OS_USER' 'DERBY_LOGDIR' 'DERBY_PORT'; do
        _v_value=$(env_var "${_v_name}")

        if [ -z "${_v_value}" ]; then
            die "Missing configuration variable: ${_v_name}"
        fi
    done

    # Check valid directories.
    for _v_name in 'DERBY_DIST_HOME' 'DERBY_SYSTEM_HOME' 'DERBY_LOGDIR'; do
        _v_value=$(env_var "${_v_name}")
        if [ ! -d "${_v_value}" ]; then
            die "Value for config variable ${_v_name} is not a valid directory: ${_v_value}"
        fi
    done

    # Validate users
    id "${DERBY_OS_USER}" >/dev/null 2>/dev/null \
    || die "Specified Derby OS user (DERBY_OS_USER) is not a valid user: ${DERBY_OS_USER}"

    # Validate TCP port
    printf '%s' "${DERBY_PORT}" \
    | egrep -q '^[0-9]+$' \
    2>/dev/null \
    || die "Specified Derby TCP port (DERBY_PORT) does not seems to be valid: ${DERBY_PORT}"
}


usage() {
    echo '###'
    echo '##  EnginFrame Derby Control Script'
    echo '#'
    echo ''
    echo 'Usage:'
    echo "  $(basename "$0") [ -h | --help ]"
    echo "  $(basename "$0") [ OPTIONS ] ACTION [ ARGS ]"
    echo ''
    echo 'Options:'
    #    0         1         2         3         4         5         6         7
    #    0123456789012345678901234567890123456789012345678901234567890123456789012345678
    #    |                      |                                                      |
    echo '  -c CONF_FILE | --conf CONF_FILE'
    echo '                      Config file to source.'
    echo '  -h | --help         Show this help.'
    echo '  -v | --verbose      Print more informations.'
    echo ''
    echo 'Actions:'
    echo '  get-pid | get-process-id'
    echo '                      Retrieve the server PID.'
    echo '  get-process-info    Get process informations.'
    echo '  ping                Ping server.'
    echo '  start               Start the server.'
    echo '  status              Print server status informations.'
    echo '  stop                Stop the server.'
    echo ''
}



# ---[ MAIN ]---------------------------------------------------------------- #


check_requirements || exit 1

umask 0022

declare -a _args=()
_conf_file=''
_verbose=false

while [ "$#" -gt 0 ]; do
    case "$1" in
        -c|--conf)
            _conf_file="$2"
            shift
            ;;
        -c=*|--conf=*)
            _conf_file=$(get_var_value "$1")
            ;;
        -v|--verbose)
            _verbose=true
            ;;
        --force)
            _is_pid_mandatory=false
            ;;
        -h|--help)
            usage
            exit 0
            ;;
        --)
            shift
            _args+=("$@")
            break
            ;;
        -*)
            die "Unrecognized option: $1"
            ;;
        *)
            _args+=("$1")
            ;;
    esac
    shift
done

if [ -n "${_conf_file}" ]; then
    verbose "[INFO] Conf file: ${_conf_file}"
    verbose ''

    if [ ! -r "${_conf_file}" ]; then
        die "Unable to load the specified config file: ${_conf_file}"
    fi
    . "${_conf_file}"
fi

check_config || exit 1

# Configure locale
if [ -n "${EF_LOCALE}" ] ; then
    export LC_ALL="${EF_LOCALE}"
    export LANG="${EF_LOCALE}"

    export LANGUAGE="${EF_LOCALE}"

    export LC_CTYPE="${EF_LOCALE}"
    export LC_COLLATE="${EF_LOCALE}"
    export LC_TIME="${EF_LOCALE}"
    export LC_NUMERIC="${EF_LOCALE}"
    export LC_MONETARY="${EF_LOCALE}"
    export LC_MESSAGES="${EF_LOCALE}"
fi


if [ -z "${DERBY_HOST}" ]; then
    DERBY_HOST="${_default_derby_host}"
fi

_pid_file="${DERBY_LOGDIR}/derby.pid"
_out_file="${DERBY_LOGDIR}/derby.out"
_err_file="${DERBY_LOGDIR}/derby.err"
_log_file="${DERBY_LOGDIR}/derby.log"

_derby_jar="${DERBY_DIST_HOME}/lib/derbyrun.jar"


_action="${_args[0]}"


case "${_action}" in
    start)
        do_start
        _exit_code=$?
        ;;
    stop)
        do_stop
        _exit_code=$?
        ;;
    restart)
        do_restart
        _exit_code=$?
        ;;
    status)
        do_status
        _exit_code=$?
        ;;
    get-pid|get-process-id)
        get_pid
        _exit_code=$?
        ;;
    ping)
        do_ping
        _exit_code=$?
        ;;
    get-process-info)
        get_process_info "$(get_pid)"
        _exit_code=$?
        ;;
    # Undocumented action
    dump-config)
        do_dump_config
        _exit_code=$?
        ;;
    *)
        exit 1
        ;;
esac

exit ${_exit_code}


#
# vi: ts=4 sw=4 et syntax=sh :
#

