// Object that defines a terminal element
function Shell(element, url, profile) {
    this.element = element;
    this.url     = url;
    this.profile   = profile || "default";
    this.socket  = null;
    this.term    = null;
}

Shell.prototype.createTerminal = function () {
    this.socket = new WebSocket(this.url);
    this.socket.onopen    = this.runTerminal.bind(this);
    this.socket.onmessage = this.getMessage.bind(this);
    this.socket.onclose   = this.closeTerminal.bind(this);
};


Shell.prototype.runTerminal = function () {
    var that = this;

    // Create an instance of hterm.Terminal
    this.term = new hterm.Terminal({ profileId: this.profile });

    // Handler that fires when terminal is initialized and ready for use
    this.term.onTerminalReady = function () {
        // Create a new terminal IO object and give it the foreground.
        // (The default IO object just prints warning messages about unhandled
        // things to the JS console.)
        var io = this.io.push();

        // Set up event handlers for io
        io.onVTKeystroke    = that.onVTKeystroke.bind(that);
        io.sendString       = that.sendString.bind(that);
        io.onTerminalResize = that.onTerminalResize.bind(that);

        // Capture all keyboard input
        this.installKeyboard();
    };

    // Patch cursor setting
    this.term.options_.cursorVisible = true;

    // Connect terminal to sacrificial DOM node
    this.term.decorate(this.element);

    // Warn user if he/she unloads page
    window.onbeforeunload = function() {
        return 'Leaving this page will terminate your terminal session.';
    };
};

Shell.prototype.getMessage = function (ev) {
    this.term.io.print(ev.data);
}

Shell.prototype.closeTerminal = function (ev) {
    var errorDiv;

    // Do not need to warn user if he/she unloads page
    window.onbeforeunload = null;

    // Inform user they lost connection
    if (this.term === null) {
        errorDiv = document.createElement('div');
        errorDiv.className = 'error';
        errorDiv.innerHTML = 'Failed to establish a websocket connection. Be sure you are using a browser that supports websocket connections.';
        this.element.appendChild(errorDiv);
    } else if (ev.code === 3146) {
        document.querySelector('iframe').remove();
        errorDiv = document.createElement('div');
        errorDiv.className = 'error';
        errorDiv.innerHTML = ev.reason;
        this.element.appendChild(errorDiv);
    } else {
        this.term.io.print('\r\nYour connection to the remote server has been terminated.');
    }
}

Shell.prototype.onVTKeystroke = function (str) {
    // Do something useful with str here.
    // For example, Secure Shell forwards the string onto the NaCl plugin.
    this.socket.send(JSON.stringify({
        input: str
    }));
};

Shell.prototype.sendString = function (str) {
    // Just like a keystroke, except str was generated by the
    // terminal itself.
    // Most likely you'll do the same this as onVTKeystroke.
    this.onVTKeystroke(str)
};

Shell.prototype.changeTheme = function (theme) {
    this.term.setProfile(theme);
}

Shell.prototype.onTerminalResize = function (columns, rows) {
    // React to size changes here.
    // Secure Shell pokes at NaCl, which eventually results in
    // some ioctls on the host.
    this.socket.send(JSON.stringify({
        resize: {
            cols: columns,
            rows: rows
        }
    }));
};
