#!/bin/bash

################################################################################
################################################################################
# Copyright 2023-2025 by NI SP Software GmbH, All rights reserved.
# Copyright 1999-2023 by Nice, srl., All rights reserved.
#
# This software includes confidential and proprietary information
# of NI SP Software GmbH ("Confidential Information").
# You shall not disclose such Confidential Information
# and shall use it only in accordance with the terms of
# the license agreement you entered into with NI SP Software.
################################################################################
################################################################################

################################################################################
# Author: Salvo Maccarone <support@ni-sp-software.com>
# Version: 1.0
################################################################################

#-------------------------------------------------------------------------------
# bplace.sh - displays hosts available to execute job
#-------------------------------------------------------------------------------

# default user is current logged user
myUser=$(id | sed -n 's/[^(]*(\([^)]*\)).*/\1/p')

myHost=''

myGroup=''

# default Resource Requirement String is empty
myResReq=""

# default queue is whatever LSF says
myQueue=$(bparams -l | sed -n 's/.*DEFAULT_QUEUE *= *//p')

# syntax string
mySyntax='Syntax: '$(basename "$0")' <options>
Options:
  -u {<user_name>|<user_group>|all}
  -m <hostname>
  -g <hostgroup>
  -R <Resource Requirement String>
  -q <queue> [-q <queue>] ... | -q all'

# parse output of standard LSF commands
parseLSF() {
    awk '
function parse() { gsub(/^[^:]+: +|\+[0-9]+ *|\//, " ", $0); return $0 }
$1 == "QUEUE:"   { q=$2; qlist[q]=1 }
$1 == "HOSTS:"   { hosts[q] = parse() }
$1 == "RES_REQ:" { resreq[q] = parse() }
$1 == "PRIO" { getline
   status[q] = $3
   qlimit[q] = $4   # QJOB_LIMIT is the job slot limit
   ulimit[q] = $5   # UJOB_LIMIT is the per-user job slot limit
   plimit[q] = $6   # PJOB_LIMIT is the per-processor job slot limit
   hlimit[q] = $7   # HJOB_LIMIT is the per-host job slot limit
  running[q] = $10  # The number of job slots used by running jobs
}
    
END {
  for (q in qlist) {
    print q
    print hosts[q]
    print resreq[q]
    print status[q]
    print qlimit[q]
    print ulimit[q]
    print plimit[q]
    print hlimit[q]
    print running[q]
  }
}
'
}

# merge two resource requirement strings
mergeResources() {
    printf "%s\n%s\n" "${1}" "${2}" | awk '
# find all sections
{ gsub(/[^ \[]+\[[^\]]+\]/,"\n&",$0)
  print $0
}' | awk '
# remove empty lines
/^ *$/ {next}
# if there is no section name, then it is a "select"
!/[^ \[]+\[/ {print "select[" $0 "]"; next}
# print other sections
{print }' | awk '
# merge all sections
{ split($0,fields,/\[|\]/)
  sections[fields[1]] = sections[fields[1]] \
      ? sprintf("%s && %s",sections[fields[1]],fields[2]) \
      : fields[2]
}
END {
  for (section in sections)
    printf ("%s[%s] ",section,sections[section])
  printf "\n"
}'
}

# read command-line parameters
declare -a myResReqList=()
while [[ $# -gt 0 ]] ; do
    _option="${1}"; shift
    case "${_option}" in
      -h|--help) echo "${mySyntax}"; exit 1 ;;
      -u) myUser="${1}"; shift ;;
      -m) myHost="${1}"; shift ;;
      -g) myGroup="${1}"; shift ;;
      -R) myResReqList[${#myResReqList[@]}]="${1}"; shift ;;
      -q) myQueue="${1}"; shift ;;
       *) ;; # ignore silently unknown parameters
    esac
done

# Merge all resource requirement strings
if [[ ${#myResReqList[@]} -eq 0 ]]; then
    myResReq=''
else
    myResReq="${myResReqList[0]}"
    if [[ -n ${myResReq} ]]; then
        for ((i=1; i<${#myResReqList[@]}; i++)); do
            if [[ -n "${myResReqList[i]}" ]]; then
                myResReq=$(mergeResources "${myResReq}" "${myResReqList[i]}")
            fi
        done
    fi
fi

# query LSF queues
while \
      read queue && \
      read hosts && \
      read resreq && \
      read status && \
      read qlimit && \
      read ulimit && \
      read plimit && \
      read hlimit && \
      read running
do

    # reset chosen host
    chosen=""

    # the queue is closed and/or inactive
    if [ "${status}" != "Open:Active" ] ; then
        echo "# queue ${queue} is closed/inactive" >&2
        continue
    fi

    # the queue forwards jobs to a remote cluster
    if [ "${hosts}" = "none" ] ; then
        echo "# queue ${queue} is remote-only" >&2
        continue
    fi

    # re-define hosts according to command-line parametes
    if [ -n "${myHost}" ] ; then
        hosts="${myHost}"
    elif [ -n "${myGroup}" ] ; then
        hosts="${myGroup}"
    fi

    # the queue has reached its job slot limit
    if [ "${qlimit}" != "-" -a "${running}" != "-" ] ; then
        if [ ${running} -ge ${qlimit} ] ; then
            echo "# queue ${queue} reached job limit" >&2
            continue
        fi
    fi

    # the queue has reached its per-user job slot limit
    if [ "${ulimit}" != "-" ] ; then
        myJobs=$(bjobs -r -u "${myUser}" -q "${myQueue}" 2>/dev/null | sed '1d' | wc -l)
        if [ ${myJobs} -ge ${ulimit} ] ; then
            echo "# queue ${queue} reached per-user limit" >&2
            continue
        fi
    fi

    # if the queue works on all nodes redefine $hosts
    [[ ${hosts} = all* ]] && hosts=""

    # get available hosts using hlimit if necessary
    hostListWithJobs=$(
        eval bhosts -w "${hosts}" | awk -v hlimit="${hlimit}" '
$2=="ok" && hlimit=="-" {printf "%s %s\n",$1,$6; next}
$2=="ok" && $6 < hlimit {printf "%s %s\n",$1,$6; next}
$2=="ok" {printf "# Per-host limit reached: %s\n",$1 >"/dev/stderr"}
#$2=="closed_Full" {printf "# host is full: %s\n",$1 >"/dev/stderr"}'
    )

    # get available hosts using plimit if necessary
    tmpList=$(echo "${hostListWithJobs}" | awk '{printf "%s ",$1}')
    if [ "${plimit}" != "-" ] ; then
        hostList=$(
          ( echo "${hostListWithJobs}" \
                | awk '{printf "[host] %s %s\n",$1,$2}'
            eval lshosts -w "${tmpList}" | sed '1d' \
                | awk '{printf "[lshosts] %s %s\n",$1,$5}'
          ) | awk -v plimit="${plimit}" '
/^\[host\] / {jobs[$2]=$3; next}
/^\[lshosts\] / && jobs[$2] < $3 * plimit {printf "%s ",$2; next}
{printf "# Per-CPU limit reached: %s\n",$2 >"/dev/stderr"}'
        )
    else
        hostList="${tmpList}"
    fi

    # trim empty lines
    hostList=$(echo "${hostList}" | sed '/^ *$/d')

    # continue only if there is at least one host
    if [ -n "${hostList}" ] ; then

        # merge two Resource Requirement Strings if necessary
        mergedResReq=""
        myResReq=$(mergeResources "type==any" "${myResReq}")
        if [ -n "${resreq}" -a -n "${myResReq}" ] ; then
            merged=$(mergeResources "${resreq}" "${myResReq}")
            mergedResReq="-R \"${merged}\""
        elif [ -n "${resreq}" ] ; then
            mergedResReq="-R \"${resreq}\""
        elif [ -n "${myResReq}" ] ; then
            mergedResReq="-R \"${myResReq}\""
        fi

        # ask LSF about available hosts
        chosen=$(eval lsplace "${mergedResReq}" "${hostList}" 2>/dev/null)
    fi

    # print result
    echo "${queue} ${chosen}"

done < <(
    if [ -n "${myHost}" ] ; then
        eval bqueues -l -u "${myUser}" -m "${myHost}" "${myQueue}"
    elif [ -n "${myGroup}" ] ; then
        eval bqueues -l -u "${myUser}" -m "${myGroup}" "${myQueue}"
    else
        eval bqueues -l -u "${myUser}" "${myQueue}"
    fi | parseLSF
)

# ex:ts=4:sw=4:et:ft=sh:
