
#
################################################################################
# Copyright 2023-2025 by NI SP Software GmbH, All rights reserved.
# Copyright 1999-2023 by Nice, srl., All rights reserved.
#
# This software includes confidential and proprietary information
# of NI SP Software GmbH ("Confidential Information").
# You shall not disclose such Confidential Information
# and shall use it only in accordance with the terms of
# the license agreement you entered into with NI SP Software.
################################################################################
#################################################################################



# Create a table containing the host's raw data.
#
# Table structure:
#
#   01: hostname
#   02: hostmodel
#   03: architecture
#   04: architecture family
#   05: number of cpus
#   06: status
#   07: total swap
#   08: free swap
#   09: total physical memory
#   10: free physical memory
#   11: % cpu utilization
#   12: r1m (short)
#   13: r5m (medium)
#   14: r15m (long)
#   15: running jobs
#

# qhost output sample:
# NOTE: the "global" line is already removed before this awk
#
#$ qhost -F -j
#HOSTNAME                ARCH         NCPU NSOC NCOR NTHR NLOAD  MEMTOT  MEMUSE  SWAPTO  SWAPUS
#----------------------------------------------------------------------------------------------
#global                  -               -    -    -    -     -       -       -       -       -
#ip-10-0-1-41            lx-amd64        1    1    1    1     -  991.2M       -     0.0       -
#   hl:arch=lx-amd64
#   hl:num_proc=1.000000
#   hl:mem_total=991.188M
#   hl:m_core=1.000000
#   hc:m_mem_free=991.000M
# ...

BEGIN {
    HOST_TO_PRINT = 0
    resetVars()

    # Read status mapping file
    while (getline < statusmap > 0) {
        split($0, fields, ":")
        s = split(fields[2], label, " ")
        for (i=1; i<=s; i++) {
            arch[label[i]] = fields[1]
        }
    }
    close(statusmap)
}


END {
    # Terminate last record, only if there is at least one (HOST_TO_PRINT == 1).
    # NOTE: the HOST_TO_PRINT variable is set in the "host row matching" function body
    if (HOST_TO_PRINT) {
        printOutput()
    }
}


# read architecture from configuration file
/^[^:]+:[^=0-9][^=]*$/ {
    split($0,fields,":")
    s = split(fields[2], label, " ")
    for (i=1; i<=s; i++) {
        arch[label[i]] = fields[1]
    }
    next
}

# Matching the header row
/^HOSTNAME/ {
    next
}

# Matching the header separator row
/^-------/ {
    next
}


# /^\ +[0-9]+\ / {
/^\s+[0-9]+\s/ {
    NJOBS++
}


/^\s+hl:cpu=/ {
    split($1, INFO, "=")
    UT = INFO[2]
}


/^\s+hl:np_load_long=/ {
    split($1, INFO, "=")
    R15M = INFO[2]
}

/^\s+hl:np_load_medium=/ {
    split($1, INFO, "=")
    R5M = INFO[2]
}

/^\s+hl:np_load_short=/ {
    split($1, INFO, "=")
    R1M = INFO[2]
}

/^\s+hl:swap_free=/ {
    split($1, INFO, "=")
    replaced_m = gsub("M", "", INFO[2])
    if (!replaced_m) {
        replaced_g = gsub("G", "", INFO[2])
        if (replaced_g) {
            INFO[2] = INFO[2] * 1024
        }
    }
    FREESWP = INFO[2] * 1048576
}


/^\s+hl:swap_total=/ {
    split($1, INFO, "=")
    replaced_m = gsub("M", "", INFO[2])
    if (!replaced_m) {
        replaced_g = gsub("G","", INFO[2])
        if (replaced_g) {
            INFO[2] = INFO[2] * 1024
        }
    }
    MAXSWP = INFO[2] * 1048576
}


/^\s+hl:mem_total=/ {
    split($1, INFO, "=")
    replaced_m = gsub("M", "", INFO[2])
    if (!replaced_m) {
        replaced_g = gsub("G", "", INFO[2])
        if (replaced_g) {
            INFO[2] = INFO[2] * 1024
        }
    }
    MAXMEM = INFO[2] * 1048576
}


/^\s+hl:mem_free=/ {
    split($1, INFO, "=")
    replaced_m = gsub("M", "", INFO[2])
    if (!replaced_m) {
        replaced_g = gsub("G", "", INFO[2])
        if (replaced_g) {
            INFO[2] = INFO[2] * 1024
        }
    }
    FREEMEM = INFO[2] * 1048576
}


# Matching the host row
#ip-10-0-1-41            lx-amd64        1    1    1    1     -  991.2M       -     0.0       -
# /^[^\ ]+/ {
/^\S/ {
    # print "\nROW: " $0
    if (HOST_TO_PRINT) {
        # Print the previous record and keep HOST_TO_PRINT=1 because we have the current host to print.
        # NOTE: latest record will be terminated in the END block.
        printOutput()
        resetVars()
    } else {
        HOST_TO_PRINT = 1
    }

    # Gridware format:
    # 1                       2            3    4    5    6     7     8       9       10      11
    # HOSTNAME                ARCH         NCPU NSOC NCOR NTHR  LOAD  MEMTOT  MEMUSE  SWAPTO  SWAPUS
    # static                  lx-amd64        4    1    4    4  0.29    7.6G    3.6G     0.0     0.0
    #  hl:load_avg=0.290000
    # 
    # UGE format:
    # 1                       2            3    4    5    6    7      8       9       10      11
    # HOSTNAME                ARCH         NCPU NSOC NCOR NTHR NLOAD  MEMTOT  MEMUSE  SWAPTO  SWAPUS
    #
    # SoGE format:
    # 1                       2            3    4    5    6     7     8       9       10      11
    # HOSTNAME                ARCH         NCPU NSOC NCOR NTHR  LOAD  MEMTOT  MEMUSE  SWAPTO  SWAPUS
    #
    # SGE old format:
    # 1                       2            3     4     5       6       7       8
    # HOSTNAME                ARCH         NCPU  LOAD  MEMTOT  MEMUSE  SWAPTO  SWAPUS
    #
    # OGS/GE format:
    # 1                       2            3     4     5       6       7       8
    # HOSTNAME                ARCH         NCPU  LOAD  MEMTOT  MEMUSE  SWAPTO  SWAPUS



    if ($2=="-") {
        # SGE, UGE: ARCH = -
        STATUS = "unavailable"
    } else if ($4=="-" && $6=="-" && $8=="-") {
        # SGE: LOAD, MEMUSE, SWAPUS = -
        STATUS = "unavailable"
    } else if ($7=="-" && $9=="-" && $11=="-") {
        # UGE: NLOAD, MEMUSE, SWAPUS = -
        STATUS = "unavailable"
    } else {
        STATUS = "ok"
    }

    myArch = arch[$2]
    if (myArch == "") {
        myArch = "unknown"
    }

    # print the record header
    printf $1 ";DEFAULT;" $2 ";" myArch ";" $3 ";" STATUS ";"
}


function printOutput() {
    if (STATUS == "ok") {
        printf "%d;%d;%d;%d;%d;%d;%d;%d;%d\n",MAXSWP,FREESWP,MAXMEM,FREEMEM,UT,R1M,R5M,R15M,NJOBS
    } else {
        print "-;-;-;-;-;-;-;-;" NJOBS
    }
}


function resetVars() {
    NJOBS   = 0
    UT      = 0
    R1M     = 0
    R15M    = 0
    MAXSWP  = 0
    FREESWP = 0
    MAXMEM  = 0
    FREEMEM = 0
}


#
# vi: ts=4 sw=4 et syntax=awk :
#

