#!/bin/bash
# ---------- Launch parameters ---------- #

# 128 is the max size accepted by AWS Batch
[ ${#name} -gt 128 ] && echo "Error: Job name length cannot exceed 128 characters" && exit 1
# The AWS Batch job name must be alphanumeric (uppercase and lowercase), numbers, hyphens, and underscores are allowed
_sanitized_jobname=$(printf "%s" "${name}" | tr -c 'a-zA-Z0-9_-' _)

# validate cpus
if [ ! "${cpus}" == ""  ] && ! [[ ${cpus} =~ ^[0-9]$ ]]; then
  echo "Error: Number of processors parameter (${cpus}) is wrong"
  exit 1
fi

_opts=""

# validate memory
if [ ! "${memory}" == "" ]; then
  if ! [[ ${memory} =~ ^[0-9]+$ ]]; then
    echo "Error: Memory parameter (${memory}) is not a number"
    exit 1
  else
    _opts="${_opts} --memory ${memory}"
  fi
fi

# validate retry_attempts
if ! [[ ${retry_attempts} =~ ^[0-9]$ ]]; then
  echo "Error: Retry attempts parameter (${retry_attempts}) is wrong"
  exit 1
else
  _opts="${_opts} --retry-attempts ${retry_attempts}"
fi

# validate timeout
if [ ! "${timeout}" == "" ]; then
  if ! [[ ${timeout} =~ ^[0-9]+$ ]]; then
    echo "Error: Timeout parameter (${timeout}) is wrong"
    exit 1
  else
    _opts="${_opts} --timeout ${timeout}"
  fi
fi

# ---------- Array parameters ---------- #

# validate size
if [ ! "${size}" == ""  ]; then
  if ! [[ ${size} =~ ^[0-9]$ ]]; then
    echo "Error: array size parameter (${size}) is wrong"
    exit 1
  else
    _opts="${_opts} --array-size ${size}"
  fi
fi

# validate dependencies
if [ ! "${size}" == ""  ]; then
  # array
  if [ ! "${depends_on_job}" == ""  ]; then
    case "${depends_on_type}" in
      SEQUENTIAL)   echo "Error: You cannot set Job ID dependency when Dependency Type is set to SEQUENTIAL"; exit 1;;
      N_TO_N)       depends_on="jobId=${depends_on_job},type=N_TO_N;";;
      *)            depends_on="jobId=${depends_on_job};"
    esac
  else
    case "${depends_on_type}" in
      SEQUENTIAL)   depends_on="type=SEQUENTIAL;";;
      N_TO_N)       echo "Error: You must set Job ID dependency when Dependency Type is set to N_TO_N"; exit 1;;
      *)            ;;
    esac
  fi
else
  # single job
  if [ ! "${depends_on_type}" == ""  ]; then
    echo "Error: Dependency Type can be set for Array jobs only."
    exit 1
  fi
  if [ ! "${depends_on_job}" == ""  ]; then
    depends_on="jobId=${depends_on_job}"
  fi
fi


# ---------- Application parameters ---------- #

# populate working dir
if [ ! "${working_dir}" == "" ]; then
  _opts="${_opts} --working-dir ${working_dir}"
fi

# ---------- Input files ---------- #

# validate input files
if [ ! "${rfb_1}" == "" ]; then
  if [ ! -f "${rfb_1}" ]; then
    echo "Remote file (${rfb_1}) not found" >&2
    exit 1
  else
    _opts="${_opts} --input-file \"${rfb_1}\""
  fi
fi

if [ ! "${mfu_1}" == "" ]; then
  while read -r file; do
    if [ ! -f "${file}" ]; then
      echo "Uploaded file (${file}) not found" >&2
      exit 1
    else
      _opts="${_opts} --input-file \"${file}\""
    fi
  done <<< "${mfu_1}"
fi

declare -a _args=(
  ${_sanitized_jobname:+  --jobname "${_sanitized_jobname}"}
  ${cpus:+                --processors "${cpus}"}
)

# ---------- Job submission ---------- #

# Parameters for the applications.submit supported by awsbatch scheduler
#  --jobname         <job name>
#  --jobmanager      awsbatch
#  --cluster         <cluster>
#  --submitopts      <quoted string of extra options passed verbatim to the jobmanager submission command>
#  --command         <command> <command parameters> - By Default is the Job Script

applications.submit "${_args[@]}" --jobmanager "awsbatch" --submitopts "${_opts}"

if [ -n "${EF_SPOOLER_URI}" ]; then
  # Set spooler visibility. Supported values:
  #  - "always" (always show data box)
  #  - "never" (do not show data box)
  #  - "smart" (show the data box only if the spooler contains visible files/directories)
  "${EF_ROOT}/plugins/ef/bin/ef.spooler.set.metadata" --spooler "${EF_SPOOLER_URI}" EF_SPOOLER_DATA_VISIBILITY "always"
fi
