#!/bin/bash

# ---------- Launch parameters ---------- #

# 128 is the max size accepted by AWS Batch
[ ${#name} -gt 128 ] && echo "Error: name length cannot exceed 128 characters" && exit 1
# The AWS Batch job name must be alphanumeric (uppercase and lowercase), numbers, hyphens, and underscores are allowed
_sanitized_jobname=$(printf "%s" "${name}" | tr -c 'a-zA-Z0-9_-' _)

# validate cpus parameter
if ! [[ ${cpus} =~ ^[0-9]*$ ]]; then
  echo "Error: cpus parameter (${cpus}) is wrong"
  exit 1
fi

# validate nodes parameter
if ! [[ ${nodes} =~ ^[0-9]*$ ]]; then
  echo "Error: nodes parameter (${nodes}) is wrong"
  exit 1
fi
_opts="--nodes ${nodes}"

# validate memory
if [ ! "${memory}" == "" ]; then
  if ! [[ ${memory} =~ ^[0-9]+$ ]]; then
    echo "Error: Memory parameter (${memory}) is not a number"
    exit 1
  else
    _opts="${_opts} --memory ${memory}"
  fi
fi

# validate timeout
if [ ! "${timeout}" == "" ]; then
  if ! [[ ${timeout} =~ ^[0-9]+$ ]]; then
    echo "Error: Timeout parameter (${timeout}) is wrong"
    exit 1
  else
    _opts="${_opts} --timeout ${timeout}"
  fi
fi

# ---------- Input files ---------- #

# validate input files
if [ ! "${rfb_1}" == "" ]; then
  if [ ! -f "${rfb_1}" ]; then
    echo "Remote file (${rfb_1}) not found" >&2
    exit 1
  else
    _opts="${_opts} --input-file \"${rfb_1}\""
  fi
fi

if [ ! "${mfu_1}" == "" ]; then
  while read -r file; do
    if [ ! -f "${file}" ]; then
      echo "Uploaded file (${file}) not found" >&2
      exit 1
    else
      _opts="${_opts} --input-file \"${file}\""
    fi
  done <<< "${mfu_1}"
fi

declare -a _args=(
  ${_sanitized_jobname:+  --jobname "${_sanitized_jobname}"}
  ${cpus:+                --processors "${cpus}"}
)

# ---------- Job submission ---------- #

# Parameters for the applications.submit supported by awsbatch scheduler

#  --jobname         <job name>
#  --jobmanager      <awsbatch>
#  --cluster         <cluster>
#  --submitopts      <quoted string of extra options passed verbatim to the jobmanager submission command>
#  --command         <command> <command parameters> - By Default is the Job Script

# submit job
applications.submit "${_args[@]}" --jobmanager "awsbatch" --submitopts "${_opts}"

if [ -n "${EF_SPOOLER_URI}" ]; then
  # Set spooler visibility. Supported values:
  #  - "always" (always show data box)
  #  - "never" (do not show data box)
  #  - "smart" (show the data box only if the spooler contains visible files/directories)
  "${EF_ROOT}/plugins/ef/bin/ef.spooler.set.metadata" --spooler "${EF_SPOOLER_URI}" EF_SPOOLER_DATA_VISIBILITY "always"
fi

