/*
 * EnginFrame Service JS
 * vim: ts=4 sw=4 et
 * ---------------------------------------------------------------------------
 * These functions are designed to be executed inside a Dynamic Service
 * Update.
 */



/* ---[ CONFIGURATION ]----------------------------------------------------- */
var EF_service_debug = false;
var EF_service_debug_private = false;



/* ---[ DATA STRUCTURES ]--------------------------------------------------- */

// 1. Selection restore
var ef_opt_selection_data = {};



/* ---[ PUBLIC FUNCTIONS ]-------------------------------------------------- */


/*
 * EF_OPT_get_id
 * -------------
 * Get the ID associated to option name /opt_name/.
 *
 * Arguments:
 * @opt_name: the name of the ef:option
 */

function ef_opt_get_id(opt_name) {
    return ef_optname2id(opt_name);
}


/*
 * EF_OPT_get_value
 * ----------------
 * Get the option value for ef:option /opt_name/.
 *
 * Arguments:
 * @opt_name: the name of the ef:option
 */

function ef_opt_get_value(opt_name) {
    var el = ef_optname2element(opt_name);

    if (!el)
        return false;

    return $(el).value;
}


/*
 * EF_OPT_set
 * ----------
 * Set the value of ef:option /opt_name/ to /value/.
 *
 * Arguments:
 * @opt_name: the name of the ef:option
 * @value: the value to set
 */

function ef_opt_set_value(opt_name, value) {
    var el = ef_optname2element(opt_name);

    if (!el)
        return false;

    $(el).value = value;

    return true;
}


/*
 * EF_OPT_enable
 * -------------
 * Enable (disabled) ef:option /opt_name/.
 *
 * Arguments:
 * @opt_name: the name of the ef:option
 */
 
function ef_opt_enable(opt_name) {
    var el = ef_optname2element(opt_name);

    if (! el)
        return false;

    el.disabled = false;

    return true;
}


/*
 * EF_OPT_disable
 * --------------
 * Disable option /opt_name/. A disabled option is still visible, but will not
 * submitted.
 *
 * Arguments:
 * @opt_name: the name of the ef:option
 */
 
function ef_opt_disable(opt_name) {
    var el = ef_optname2element(opt_name);

    if (!el)
        return false;

    el.disabled = true;

    return true;
}


/*
 * EF_OPT_show
 * -----------
 * Show (hidden) option /opt_name/. This will display an hidden element, it
 * will not convert convert an ef:option type=hidden into another type of
 * option !!!
 *
 * Arguments: 
 * @opt_name: the name of the ef:option
 */

function ef_opt_show(opt_name) {
    var el = ef_optname2element(opt_name);
    if (!el)
        return false;

    Element.show(el);

    return true;
}


/*
 * EF_OPT_hide
 * -----------
 * Hide option /opt_name/. An hidden option is still enabled.
 *
 * Arguments: 
 * @opt_name: the name of the ef:option
 */
 
function ef_opt_hide(opt_name) {
    var el = ef_optname2element(opt_name);

    if (!el)
        return false;

    Element.hide(el);

    return true;
}


/*
 * EF_OPT_change
 * -------------
 * Change the value of option /opt_name/ to /value/.
 *
 * Arguments: 
 * @opt_name: the name of the ef:option
 */
 
function ef_opt_change(opt_name, value) {
    var el = ef_optname2element(opt_name);
    var old_value;

    if (!el) 
        return false;

    old_value = el.value;
    el.value = value;

    if (EF_service_debug)
        window.alert('[EF_OPT_change]: changed value for ' + opt_name + ' from "' + old_value + '" to "' + value + '"... now value is ' + el.value + '\n');

    return true;
}


/*
 * EF_OPT_change_label
 * -------------------
 * Change the label for option /opt_name/ to /label/.
 *
 * Arguments: 
 * @opt_name: the name of the ef:option
 */

function ef_opt_change_label(opt_name, label) {

    var el = ef_opt_get_label_for(opt_name);

    if (!el)
        return false;

    el.innerHTML = label;

    return true;
}


/*
 * EF_OPT_show_label
 * -----------------
 * Show the (hidden) label for /opt_name/.
 *
 * Arguments:
 * @opt_name: the name of the ef:option
 */

function ef_opt_show_label(opt_name) {
    var el = ef_opt_get_label_for(opt_name);

    if (!el)
        return false;

    Element.show(el);

    return true;
}


/*
 * EF_OPT_hide_label
 * -----------------
 * Hide the (visible) label for /opt_name/.
 *
 * Arguments:
 * @opt_name: the name of the ef:option
 */

function ef_opt_hide_label(opt_name) {
    var el = ef_opt_get_label_for(opt_name);

    if (!el)
        return false;

    Element.hide(el);

    return true;
}


/*
 * EF_OPT_select_radio
 * -------------------
 * Select the item /sid/ of the option /opt_name/ of type radio.
 *
 * Arguments: 
 * @opt_name: the name of the ef:option
 */

function ef_opt_select_radio(opt_name, sid) {
    var el = ef_optname2element(ef_opt_sid(opt_name, sid));

    if (!el)
        return false;

    el.checked = true;

    return true;
}


/*
 * EF_OPT_add_list_item
 * -------------
 * Add an item to /list_name/ at the position /idx/ or
 * at the end if /idx/ is not specified.
 *
 * @list_name: name of the list option
 * @id: ID of the item to add
 * @text: content of the item to add
 * @idx: optional index where to add the item
 */
function ef_opt_add_list_item(list_name, id, text, idx) {
    var list = ef_optname2element(list_name);

    if (!list) {
        return false;
    }

    if (idx === undefined || idx>list.options.length) {
        idx = list.options.length;
    }

    list.options.add(new Option(text, id), idx);

    return true;
}


/*
 * EF_OPT_remove_list_item
 * -------------
 * Remove an item from /list_name/.
 *
 * @list_name: name of the list option
 * @id: ID of the item to remove
 */
function ef_opt_remove_list_item(list_name, id) {
    var list = ef_optname2element(list_name);
    var i;

    if (!list) {
        return false;
    }

    for (i=0; i<list.options.length; i++) {
        if (id == list.options[i].value) {
            list.remove(i);
            return true;
        }
    }

    return false;
}


/***
 **  SELECTION
 */


/*
 * EF_OPT_set_selection
 * --------------------
 * Change the selection (index) for /opt_name/ to item in position /idx/.
 *
 * Arguments: 
 * @opt_name: the name of the ef:option
 * @idx: the target index
 */
 
function ef_opt_set_selection(opt_name, idx) {
    var el = ef_optname2element(opt_name);

    if (!el) {
        return false;
    }

    if (!ef_opt_selection_check(el, idx))
        return false;

    el.selectedIndex = idx;

    return true;
}



/*
 * EF_OPT_set_selection_by_value
 * -----------------------------
 * Change the selection (index) for /opt_name/ to the position of the item with
 * value /value/.
 *
 * Arguments:
 * @opt_name: the name of the ef:option
 * @value: the value
 */

function ef_opt_set_selection_by_value(opt_name, value) {
    var el = ef_optname2element(opt_name);

    if (!el) {
        return false;
    }

    for (i=0; i<el.options.length; i++) {
        if (el.options[i].value == value) {
            return ef_opt_set_selection(opt_name,i);
        }
    }

    return false;
}



/*
 * EF_OPT_get_selection
 * --------------------
 * Get current selection index for option /opt_name/
 *
 * Arguments: 
 * @opt_name: the name of the ef:option
 */
 
function ef_opt_get_selection(opt_name) {
    var el = ef_optname2element(opt_name);

    if (!el) {
        return null;
    }

    return el.selectedIndex;
}


/*
 * EF_OPT_enable_selection
 * -----------------------
 * Enable selection /idx/ for option /opt_name/
 *
 * Arguments: 
 * @opt_name: the name of the ef:option
 */

function ef_opt_enable_selection(opt_name, idx) {
    var el = ef_optname2element(opt_name);

    if ( !el || !ef_opt_selection_check(el, idx) )
        return false;

    el.options[idx].disabled = false;

    return true;
}


/*
 * EF_OPT_disable_selection
 * ------------------------
 *
 * Arguments: 
 * @opt_name: the name of the ef:option
 */

function ef_opt_disable_selection(opt_name, idx) {
    var el = ef_optname2element(opt_name);

    if ( !el || !ef_opt_selection_check(el, idx) )
        return false;
    
    el.options[idx].disabled = true;

    return true;
}


/*
 * EF_OPT_show_selection
 * ---------------------
 *
 * Arguments: 
 * @opt_name: the name of the ef:option
 */
 
function ef_opt_show_selection(opt_name, idx) {
    var el = ef_optname2element(opt_name);

    if ( !el || !ef_opt_selection_check(el, idx) )
        return false;

    Element.show(el.options[idx]);

    return true;
}


/*
 * EF_OPT_hide_selection
 * ---------------------
 * Hide the option with index /idx/ inside selection /opt_name/
 *
 * Arguments:
 * @opt_name: the name of the ef:option
 * @idx: the index of the ef:option inside main ef:option
 */

function ef_opt_hide_selection(opt_name, idx) {
    var el = ef_optname2element(opt_name);

    if ( !el || !ef_opt_selection_check(el, idx) )
        return false;

    Element.hide(el.options[idx]);

    return true;
}


/*
 * EF_OPT_save_selection
 * ---------------------
 * Save the current value (by default the original value) of the selection
 * named /name/ in the main storage.
 *
 * @name: the selection name
 */

function ef_opt_save_selection(name) {
    var sel = ef_optname2element(name);
    var n,v;
    var i, msg = '';

    if (EF_service_debug) {
        msg = '[EF_OPT_save_selection] on selection ' + name + '\n';
    }

    if (!ef_opt_selection_check(sel)) {
        if (EF_service_debug) {
            msg += 'Element ' + name + ' is valid selection: EXITING !!!';
            window.alert(msg);
        }
        return false;
    }

    // Overwrite existing values
    ef_opt_selection_data[name] = [];

    for (i=0; i<sel.options.length; i++) {
        ef_opt_selection_data[name][i] = [sel.options[i].value, sel.options[i].text];
    }

    if (EF_service_debug)
        msg += 'Saved data:\n';
    for (i=0; i<ef_opt_selection_data[name].length; i++) {
        v = ef_opt_selection_data[name][i][0];
        n = ef_opt_selection_data[name][i][1];
        if (EF_service_debug)
            msg += ' - ' + i + '.  Name = ' + n + ' | Value = ' + v +'\n';
    }

    if (EF_service_debug) {
        window.alert(msg);
    }
    
    return true;
}


function ef_opt_restore_selection(name) {
    var i, msg = '';

    if (EF_service_debug) {
        msg = '[EF_OPT_restore_selection] name = ' + name + '\n';
    }

    var sel = ef_optname2element(name);
    if (!ef_opt_selection_check(sel)) {
        if (EF_servide_debug) {
            msg += 'Element doesn\'t exist: EXITING !!!\n';
            window.alert(msg);
        }
        return false;
    }

    if (!ef_opt_selection_data[name]) {
        if (EF_service_debug) {
            msg += 'Select named ' + name + ' is not initted, so you could not restore it\n';
            window.alert(msg);
        }
        return true;
    }


    // Preparation:
    //     remove all existing element of the list before repopulating,
    //     starting from the end of the list.
    for (i=sel.options.length - 1; i >=0 ; i--) {
        sel.remove(i);
    }
    
    var orig = ef_opt_selection_data[name];
    for (i=0; i < orig.length; i++) {
        v = orig[i][0];
        n = orig[i][1];
        //option = new Option(hash['name'], hash['value']); REMOVED
        //itemList.options.add(option, i); REMOVED
        sel.options[i] = new Option(n, v, false, false);
    }

    return true;
}


/*
 * EF_OPT_remove
 * -------------
 * Remove from selection /name/ the item (listed by name) in /id_list/.
 *
 * @name: name of the selection
 * @id_list: list of ID to remove
 */

function ef_opt_remove_from_selection(name, ids) {
    var sel = ef_optname2element(name);
    var i, j, msg = '';

    if (EF_service_debug) {
        msg = '[EF_OPT_remove_from_selection] name=' + name + '\n';
    }

    if (!ef_opt_selection_check(sel)) {
        if (EF_service_debug) {
            msg += 'Select ' + name + ' is not a valid element: EXITING !!!\n';
            window.alert(msg);
        }
        return false;
    }

    if (!ids.length || ids.length <= 0) {
        if (EF_service_debug) {
            msg += 'This is not a valid list of ID: EXITING !!!\n';
            window.alert(msg);
        }
        return false;
    }
    
    if (!ef_opt_selection_data[name]) {
        /*
        if (EF_service_debug) {
            msg += 'You could not remove elements from from an unsaved selection: EXITING !!!\n';
            window.alert(msg);
        }
        return false;
        */
        if (!ef_opt_save_selection(name))
            return false;
    }

    for (i=0; i<ids.length; i++) {
        for (j=0; j<sel.options.length; j++) {
            if (EF_service_debug)
                msg += 'Step (i='+i+',j='+j+'): ids('+(ids[i])+') sel('+(sel.options[j].value)+')\n';
            if (ids[i] == sel.options[j].value) {
                if (EF_service_debug)
                    msg += 'Found ids [' + i + ']:' + ids[i] + ' in position ' + j + '\n';
                sel.remove(j);
                break;
            }
        }
    }
    if (EF_service_debug)
        window.alert(msg);

    return true;
}


/*
 * TODO: EF_OPT_filter_selection
 * -----------------------------
 * Set the selection options to those elements for which function /filter/
 * returns true.
 *
 * @name: the selection name
 * @filter: the filter function
 */



/* ---[ PRIVATE FUNCTIONS ]------------------------------------------------- */


/*
 * EF_optname2id (PRIVATE)
 * -----------------------
 * Used by all functions, directly or indirectly.
 *
 * Obtain option ID from /opt_name/.
 *
 * Arguments: 
 * @opt_name: the name of the ef:option
 */

function ef_optname2id(name) {
    var tmp_name = 'serviceopt_' + name;

    return tmp_name;
}


/*
 * EF_optname2element (PRIVATE)
 * ----------------------------
 * Used by all functions.
 *
 * Try to find the DOM element for ef:option named /name/ (return null if
 * fails)
 *
 * Arguments:
 * @name: ef:option name
 */

function ef_optname2element(opt_name) {
    var id = ef_optname2id(opt_name);
    var el = $(id);
    var msg = '';

    if (EF_service_debug_private) {
        msg = '[EF_optname2element] for id ' + id;
    }

    if (!el) {
        if (EF_service_debug_private) {
            msg += ' not found !!!';
            window.alert(msg);
        }
        return null;
    }

    if (EF_service_debug_private) {
        msg += ' found in element ' + el;
        window.alert(msg);
    }

    return el;
}


/*
 * EF_OPT_selection_check (PRIVATE)
 * --------------------------------------
 * Used by all funcions like EF_opt_*_selection.
 *
 * Check if /idx/ is a valid index for element /el/.
 *
 * Arguments:
 * @el: the element
 * @idx: the index
 */

function ef_opt_selection_check(el, idx) {
    var i_idx;

    if (!el)
        return false;

    if (el.tagName.toLowerCase() != 'select')
        return false;

    if (! (el.options && el.options.length) )
        return false;
    
    if (idx != undefined) {
        i_idx = parseInt(idx);

        if (i_idx == NaN)
            return false;

        if (i_idx >= el.options.length)
            return false;
    }

    return true;
}


/*
 * EF_OPT_get_label (PRIVATE)
 * --------------------------
 * Get the label for /opt_name/.
 *
 * Arguments:
 * @opt_name: the name of the ef:option
 */

function ef_opt_get_label_for(opt_name) {
    var id = ef_optname2id(opt_name);

    var lbls = document.getElementsByTagName('label');

    for (var i=0; i<=lbls.length; i++) {
        if (!lbls[i])
            continue;
        if (lbls[i].htmlFor == id) {
            return lbls[i];
        }
    }

    return null;
}


/*
 * EF_OPT_sid (PRIVATE)
 * --------------------
 *
 * Arguments:
 * @opt_name: option name (base)
 * @sid: sub item ID
 */

function ef_opt_sid(opt_name, sid) {
    var tmp_name = opt_name + '_sid_' + sid;

    return tmp_name;
}

