<?xml version="1.0" encoding="UTF-8"?>
<!--
  * Copyright 2023-2025 by NI SP Software GmbH, All rights reserved.
 * Copyright 1999-2023 by Nice, srl., All rights reserved.
 *
 * This software includes confidential and proprietary information
 * of NI SP Software GmbH ("Confidential Information").
 * You shall not disclose such Confidential Information
 * and shall use it only in accordance with the terms of
 * the license agreement you entered into with NI SP Software.-->
<xsl:stylesheet version="1.0"
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns:ch="http://www.enginframe.com/2010/chart"
  xmlns:ef="http://www.enginframe.com/2000/EnginFrame"
  xmlns:exsl="http://exslt.org/common"
  xmlns:func="http://exslt.org/functions"
  xmlns:math="http://exslt.org/math"
  xmlns:str="http://exslt.org/strings"
  xmlns:xalan="http://xml.apache.org/xalan"
  xmlns:efutils="http://www.enginframe.com/2010/EF/Utils"
  xmlns:encode="org.owasp.encoder.Encode"
  extension-element-prefixes="exsl func math str math efutils"
  exclude-result-prefixes="ch ef func xalan encode">

  <!--
    We need to have this also here (besides com.enginframe.system.xsl) since
    this xsl can also be used standalone for rest services producing charts.
   -->
  <xsl:param name="_root_context">enginframe</xsl:param>

  <!-- extension function for xalan -->
  <xalan:component prefix="efutils" functions="getProperty">
    <xalan:script lang="javaclass" src="xalan://com.enginframe.common.utils.Utils"/>
  </xalan:component>

  <xsl:strip-space elements="ch:*" />

  <!-- This variable is global and used also to init the corresponding js -->
  <xsl:variable name="_charts_base_url">
    <xsl:value-of select="efutils:getProperty('ef.charts.base.url', concat('/', $_root_context, '/efcharts'))"/>
  </xsl:variable>

  <!-- low-priority template that matches everything -->
  <xsl:template match="node()|@*" priority="-999">
    <xsl:copy>
      <xsl:apply-templates select="node()|@*" />
    </xsl:copy>
  </xsl:template>

  <!-- Make the base url also available in js -->
  <xsl:template match="/ef:agent" mode="charts.head">
    <script type="text/javascript">
      jQuery.enginframe.chartsBaseUrl="<xsl:value-of select="encode:forJavaScriptBlock($_charts_base_url)" />";
    </script>
  </xsl:template>

  <xsl:template match="ch:chart">
    <img>
      <xsl:if test="@id">
        <xsl:attribute name="id">
          <xsl:value-of select="@id"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:if test="@class">
        <xsl:attribute name="class">
          <xsl:value-of select="@class"/>
        </xsl:attribute>
      </xsl:if>
      <xsl:attribute name="alt">
        <xsl:call-template name="chart.alt"/>
      </xsl:attribute>
      <xsl:attribute name="height">
        <xsl:value-of select="floor(ch:view/@height)"/>
      </xsl:attribute>
      <xsl:attribute name="width">
        <xsl:value-of select="floor(ch:view/@width)"/>
      </xsl:attribute>
      <xsl:attribute name="src">
        <xsl:call-template name="chart.target"/>
      </xsl:attribute>
    </img>
  </xsl:template>

  <xsl:template name="chart.alt">
    <xsl:choose>
      <xsl:when test="ch:view/ch:option[@name='title']/text()!=''">
        <xsl:value-of select="ch:view/ch:option[@name='title']/text()"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:variable name="chartType" select="ch:view/@type"/>
        <xsl:choose>
          <xsl:when test="$chartType='pie'">Pie Chart</xsl:when>
          <xsl:when test="$chartType='line'">Line Chart</xsl:when>
          <xsl:when test="$chartType='bar'">Bar Chart</xsl:when>
        </xsl:choose>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template name="chart.target">
    <xsl:variable name="base.url">
      <xsl:choose>
        <xsl:when test="@base!=''">
          <xsl:value-of select="@base"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="$_charts_base_url"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>

    <xsl:value-of select="concat($base.url, '?')"/>

    <!--
      First of all figure out max and min since we need them in
      many places (axis labels, scaling, ...).
      In particular the scale factor used for the data encoding is:
        scale = max - min
      with:
        min = MIN(min_specified_as_option, min_value_of_any_serie)
      and
        max = MAX(max_specified_as_option, max_value_of_any_serie)
      or in case of a stacked chart
        max = MAX(max_specified_as_option, max_stacked_value)
    -->
    <xsl:variable name="minval">
      <xsl:call-template name="chart.data.min"/>
    </xsl:variable>
    <xsl:variable name="min">
      <xsl:choose>
        <xsl:when test="ch:view[@type='pie']">
           <xsl:value-of select="0"/>
         </xsl:when>
        <xsl:when test="ch:view/ch:option[@name='min'] and string-length(normalize-space(ch:view/ch:option[@name='min'])) &gt; 0 and number(ch:view/ch:option[@name='min']) &lt; number($minval)">
           <xsl:value-of select="normalize-space(ch:view/ch:option[@name='min'])"/>
         </xsl:when>
         <xsl:otherwise>
           <xsl:choose>
             <xsl:when test="ch:view[@type='bar']">
               <xsl:choose>
                 <xsl:when test="ch:view/ch:option[@name='isStacked']/text()='false'">
                   <xsl:value-of select="$minval"/>
                 </xsl:when>
                 <xsl:otherwise>
                   <xsl:value-of select="0"/>
                 </xsl:otherwise>
               </xsl:choose>
             </xsl:when>
             <xsl:otherwise>
               <xsl:value-of select="$minval"/>
             </xsl:otherwise>
           </xsl:choose>
         </xsl:otherwise>
       </xsl:choose>
    </xsl:variable>

    <xsl:variable name="maxval">
      <xsl:choose>
        <xsl:when test="ch:view[@type='bar']">
          <xsl:choose>
            <xsl:when test="ch:view/ch:option[@name='isStacked']/text()='false'">
              <xsl:call-template name="chart.data.max"/>
            </xsl:when>
            <xsl:otherwise>
              <xsl:call-template name="chart.data.max.stacked"/>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:when>
        <xsl:otherwise>
          <xsl:call-template name="chart.data.max"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <xsl:variable name="max">
      <xsl:choose>
        <xsl:when test="ch:view/ch:option[@name='max'] and string-length(normalize-space(ch:view/ch:option[@name='max'])) &gt; 0 and number(ch:view/ch:option[@name='max']) &gt; number($maxval)">
          <xsl:value-of select="normalize-space(ch:view/ch:option[@name='max'])"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="$maxval"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>

    <!-- If both max and min are equal to 0, we set max to 1 to create a valid chart range -->
    <xsl:variable name="adjustedMax">
      <xsl:choose>
        <xsl:when test="number($min) = number($max) and number($max) = 0">
          <xsl:value-of select="1"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="$max"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>

    <xsl:apply-templates select="ch:view" mode="chart.type"/>

    <xsl:apply-templates select="ch:view" mode="chart.options">
      <xsl:with-param name="min" select="$min"/>
      <xsl:with-param name="max" select="$adjustedMax"/>
    </xsl:apply-templates>

    <xsl:apply-templates select="." mode="chart.legend"/>

    <xsl:apply-templates select="ch:data" mode="chart.data">
      <xsl:with-param name="min" select="$min"/>
      <xsl:with-param name="max" select="$adjustedMax"/>
    </xsl:apply-templates>
  </xsl:template>


  <!-- Chart types -->

  <!-- Pie chart -->
  <xsl:template match="ch:view[@type='pie']" mode="chart.type">
    <xsl:text>cht=</xsl:text>
    <xsl:choose>
      <xsl:when test="ch:option[@name='is3D']='true'">
        <xsl:text>p3</xsl:text>
      </xsl:when>
      <xsl:otherwise>
        <xsl:text>p</xsl:text>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <!-- Line chart -->
  <xsl:template match="ch:view[@type='line']" mode="chart.type">
    <xsl:text>cht=</xsl:text>
    <xsl:choose>
      <xsl:when test="ch:option[@name='showAxisLines']='false'">
        <xsl:text>ls</xsl:text>
      </xsl:when>
      <xsl:otherwise>
        <xsl:text>lc</xsl:text>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <!-- Bar chart -->
  <xsl:template match="ch:view[@type='bar']" mode="chart.type">
    <xsl:text>cht=</xsl:text>
    <xsl:text>b</xsl:text>
    <xsl:choose>
      <xsl:when test="ch:option[@name='isVertical']='true'">
        <xsl:text>v</xsl:text>
      </xsl:when>
      <!-- default isVertical false -->
      <xsl:otherwise>
        <xsl:text>h</xsl:text>
      </xsl:otherwise>
    </xsl:choose>
    <xsl:choose>
      <xsl:when test="ch:option[@name='isStacked']='false'">
        <xsl:text>g</xsl:text>
      </xsl:when>
      <!-- default isStacked true -->
      <xsl:otherwise>
        <xsl:text>s</xsl:text>
      </xsl:otherwise>
    </xsl:choose>
    <xsl:text>&amp;chbh=a</xsl:text>
  </xsl:template>

  <xsl:template match="ch:view" mode="chart.type">
    <xsl:message>Chart of type <xsl:value-of select="@type"/> not supported</xsl:message>
  </xsl:template>


  <!-- Chart Options -->

  <!-- options common to all chart types -->
  <xsl:template match="ch:view" mode="chart.options">
    <xsl:param name="min"/>
    <xsl:param name="max"/>

    <xsl:if test="@width and @height">
      <xsl:text>&amp;chs=</xsl:text>
      <xsl:value-of select="floor(@width)"/>
      <xsl:text>x</xsl:text>
      <xsl:value-of select="floor(@height)"/>
    </xsl:if>

    <xsl:if test="ch:option[@name='title']/text()!=''">
      <xsl:text>&amp;chtt=</xsl:text>
      <xsl:value-of select="encode:forUriComponent(ch:option[@name='title']/text())"/>
    </xsl:if>

    <xsl:if test="ch:option[@name='titleColor']/text()!=''">
      <xsl:text>&amp;chts=</xsl:text>
      <xsl:value-of select="encode:forUriComponent(ch:option[@name='titleColor']/text())"/>
      <xsl:if test="ch:option[@name='titleSize']/text()!=''">
        <xsl:text>,</xsl:text>
        <xsl:value-of select="encode:forUriComponent(ch:option[@name='titleSize']/text())"/>
      </xsl:if>
    </xsl:if>

    <xsl:choose>
      <xsl:when test="ch:option[@name='colors']/text()!=''">
        <xsl:text>&amp;chco=</xsl:text>
        <xsl:value-of select="encode:forUriComponent(ch:option[@name='colors']/text())"/>
      </xsl:when>
      <xsl:when test="ch:option[@name='color']/text()!=''">
        <xsl:variable name="firstColor">
          <xsl:value-of select="str:tokenize(ch:option[@name='color']/text(), ',')"/>
        </xsl:variable>
        <xsl:text>&amp;chco=</xsl:text>
        <xsl:value-of select="encode:forUriComponent($firstColor)"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:text>&amp;chco=</xsl:text>
        <xsl:choose>
          <xsl:when test="@type='pie'">
            <xsl:for-each select="str:tokenize(../ch:data/ch:series, ',')">
              <xsl:value-of select="ch:color()"/>
                <xsl:if test="position() != last()">
                  <xsl:text>,</xsl:text>
                </xsl:if>
              </xsl:for-each>
          </xsl:when>
          <xsl:otherwise>
            <xsl:for-each select="../ch:data/ch:series">
              <xsl:value-of select="ch:color()"/>
                <xsl:if test="position() != last()">
                  <xsl:text>,</xsl:text>
              </xsl:if>
            </xsl:for-each>
          </xsl:otherwise>

        </xsl:choose>
      </xsl:otherwise>
    </xsl:choose>

    <!-- default backgroundColor white #FFFFFF-->
    <xsl:choose>
      <xsl:when test="ch:option[@name='backgroundColor']/text()!=''">
        <xsl:text>&amp;chf=bg,s,</xsl:text>
        <xsl:value-of select="encode:forUriComponent(ch:option[@name='backgroundColor']/text())"/>
      </xsl:when>
        <!-- omit -->
    </xsl:choose>

    <xsl:choose>
      <xsl:when test="ch:option[@name='legend']/text()!=''">
        <xsl:variable name="legendPosition" select="ch:option[@name='legend']/text()"/>
        <xsl:variable name="legendOrientation" select="ch:option[@name='legendOrientation']/text()"/>
        <xsl:text>&amp;chdlp=</xsl:text>
        <xsl:choose>
          <xsl:when test="$legendPosition='left'">
            <xsl:text>l</xsl:text>
          </xsl:when>
          <xsl:when test="$legendPosition='right'">
            <xsl:text>r</xsl:text>
          </xsl:when>
          <xsl:when test="$legendPosition='bottom'">
            <xsl:text>b</xsl:text>
            <xsl:if test="$legendOrientation='vertical'">
              <xsl:text>v</xsl:text>
            </xsl:if>
          </xsl:when>
          <xsl:when test="$legendPosition='top'">
            <xsl:text>t</xsl:text>
            <xsl:if test="$legendOrientation='vertical'">
              <xsl:text>v</xsl:text>
            </xsl:if>
          </xsl:when>
        </xsl:choose>
      </xsl:when>
      <xsl:otherwise>
        <xsl:text>&amp;chdlp=r</xsl:text>
      </xsl:otherwise>
    </xsl:choose>

    <xsl:apply-templates select="." mode="chart.options.specific">
      <xsl:with-param name="min" select="$min"/>
      <xsl:with-param name="max" select="$max"/>
    </xsl:apply-templates>

  </xsl:template>

  <!-- options specific of pie charts -->
  <xsl:template match="ch:view[@type='pie']" mode="chart.options.specific">
    <xsl:variable name="labelsType" select="ch:option[@name='labels']/text()"/>
    <xsl:choose>
      <xsl:when test="$labelsType='value'">
        <xsl:apply-templates select="../ch:data/ch:series" mode="chart.labels.pie"/>
      </xsl:when>
      <xsl:when test="$labelsType='name'">
      <xsl:apply-templates select="../ch:data/ch:labels" mode="chart.labels.pie"/>
      </xsl:when>
      <!-- default no labels (none)-->
        <!-- omit -->
    </xsl:choose>
    <xsl:choose>
      <xsl:when test="ch:option[@name='rotation']/text()!=''">
        <xsl:text>&amp;chp=</xsl:text>
        <xsl:value-of select="encode:forUriComponent(ch:option[@name='rotation']/text())"/>
      </xsl:when>
        <!-- omit -->
    </xsl:choose>
  </xsl:template>

  <xsl:template match="ch:series" mode="chart.labels.pie">
    <xsl:text>&amp;chl=</xsl:text>
    <xsl:choose>
      <xsl:when test="*">
        <xsl:apply-templates select="ch:element" mode="chart.labels.pie"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="encode:forUriComponent(ch:replace(text(),',','|'))"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template match="ch:labels" mode="chart.labels.pie">
    <xsl:text>&amp;chl=</xsl:text>
    <xsl:choose>
      <xsl:when test="*">
        <xsl:apply-templates select="ch:label" mode="chart.labels"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="encode:forUriComponent(ch:replace(text(),',','|'))"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template match="ch:element" mode="chart.labels.pie">
    <xsl:value-of select='encode:forUriComponent(text())'/>
    <xsl:if test="position() != last()">
       <xsl:text>%7C</xsl:text>
    </xsl:if>
  </xsl:template>

  <xsl:template match="ch:label" mode="chart.labels">
    <xsl:value-of select='encode:forUriComponent(text())'/>
    <xsl:if test="position() != last()">
       <xsl:text>%7C</xsl:text>
    </xsl:if>
  </xsl:template>

  <!-- options specific of line charts -->
  <xsl:template match="ch:view[@type='line']" mode="chart.options.specific">
    <xsl:param name="min"/>
    <xsl:param name="max"/>
    <xsl:variable name="showAxisLines">
      <xsl:choose>
        <xsl:when test="ch:option[@name='showAxisLines']/text()!=''">
          <xsl:value-of select="ch:option[@name='showAxisLines']/text()"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="'true'"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <xsl:variable name="showCategoryLabels">
      <xsl:choose>
        <xsl:when test="ch:option[@name='showCategoryLabels']/text()!=''">
          <xsl:value-of select="ch:option[@name='showCategoryLabels']/text()"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="$showAxisLines"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <xsl:variable name="showValueLabels">
      <xsl:choose>
        <xsl:when test="ch:option[@name='showValueLabels']/text()!=''">
          <xsl:value-of select="ch:option[@name='showValueLabels']/text()"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="$showAxisLines"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <xsl:choose>
      <xsl:when test="$showCategoryLabels='false' and $showValueLabels='false'">
        <!-- omit -->
      </xsl:when>
      <xsl:when test="$showCategoryLabels='true' and $showValueLabels='false'">
        <xsl:text>&amp;chxt=x</xsl:text>
        <xsl:apply-templates select="../ch:data/ch:labels" mode="chart.labels.line">
          <xsl:with-param name="min" select="$min"/>
          <xsl:with-param name="max" select="$max"/>
        </xsl:apply-templates>
      </xsl:when>
      <xsl:when test="$showCategoryLabels='false' and $showValueLabels='true'">
        <xsl:text>&amp;chxt=y</xsl:text>
      </xsl:when>
      <!--  default showCategoryLabels true and showValueLabels true -->
      <xsl:otherwise>
        <xsl:text>&amp;chxt=x,y</xsl:text>
        <xsl:apply-templates select="../ch:data/ch:labels" mode="chart.labels.line">
          <xsl:with-param name="min" select="$min"/>
          <xsl:with-param name="max" select="$max"/>
        </xsl:apply-templates>
      </xsl:otherwise>
    </xsl:choose>

    <xsl:choose>
      <xsl:when test="$showValueLabels='true'">
        <xsl:call-template name="chart.axis.range">
          <xsl:with-param name="min" select="$min"/>
          <xsl:with-param name="max" select="$max"/>
          <xsl:with-param name="showCategoryLabels" select="$showCategoryLabels"/>
        </xsl:call-template>
      </xsl:when>
        <!-- omit -->
    </xsl:choose>

  </xsl:template>

  <xsl:template name="chart.axis.range">
    <xsl:param name="min"/>
    <xsl:param name="max"/>
    <xsl:param name="showCategoryLabels"/>
    <xsl:choose>
      <xsl:when test="not(ch:option[@name='axisValueLabel']) and ch:option[@name='valueLabelsInterval']/text()!=''">
        <xsl:text>&amp;chxr=</xsl:text>
        <xsl:choose>
          <xsl:when test="$showCategoryLabels='false'">
            <xsl:text>0,</xsl:text>
          </xsl:when>
          <xsl:otherwise>
            <xsl:text>1,</xsl:text>
          </xsl:otherwise>
        </xsl:choose>
        <xsl:value-of select="encode:forUriComponent($min)"/>
        <xsl:text>,</xsl:text>
        <xsl:value-of select="encode:forUriComponent($max)"/>
        <xsl:text>,</xsl:text>
        <xsl:value-of select="encode:forUriComponent(ch:option[@name='valueLabelsInterval']/text())"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:text>&amp;chxr=</xsl:text>
        <xsl:choose>
          <xsl:when test="$showCategoryLabels='false'">
            <xsl:text>0,</xsl:text>
          </xsl:when>
          <xsl:otherwise>
            <xsl:text>1,</xsl:text>
          </xsl:otherwise>
        </xsl:choose>
        <xsl:value-of select="encode:forUriComponent($min)"/>
        <xsl:text>,</xsl:text>
        <xsl:value-of select="encode:forUriComponent($max)"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template match="ch:labels" mode="chart.labels.line">
    <xsl:param name="min"/>
    <xsl:param name="max"/>
    <xsl:text>&amp;chxl=</xsl:text>
    <xsl:choose>
      <xsl:when test="string-length(ch:emitLabels(ch:dataLabels(.), 'false'))!=0">
        <xsl:text>0:%7C</xsl:text>
        <xsl:value-of select="ch:emitLabels(ch:dataLabels(.), 'false')"/>
      </xsl:when>
        <!-- omit -->
    </xsl:choose>
    <xsl:if test="string-length(ch:emitLabels(ch:dataLabels(.), 'false'))!=0 and ../../ch:view/ch:option[@name='axisValueLabel']!=''">
      <xsl:text>%7C</xsl:text>
    </xsl:if>
    <xsl:choose>
      <xsl:when test="../../ch:view/ch:option[@name='axisValueLabel']!=''">
        <xsl:text>1:%7C</xsl:text>
        <xsl:value-of select="encode:forUriComponent($min)"></xsl:value-of>
        <xsl:text>%7C</xsl:text>
        <xsl:value-of select="($max + $min) div 2"></xsl:value-of>
        <xsl:text>%7C</xsl:text>
        <xsl:value-of select="encode:forUriComponent(../../ch:view/ch:option[@name='axisValueLabel']/text())"/>
        <xsl:text> </xsl:text>
        <xsl:value-of select="encode:forUriComponent($max)"/>
      </xsl:when>
        <!-- omit -->
    </xsl:choose>
  </xsl:template>

  <!-- options specific of bar charts -->
  <xsl:template match="ch:view[@type='bar']" mode="chart.options.specific">
    <xsl:param name="min"/>
    <xsl:param name="max"/>
    <xsl:variable name="showCategoryLabels">
      <xsl:choose>
        <xsl:when test="ch:option[@name='showCategoryLabels']/text()!=''">
          <xsl:value-of select="ch:option[@name='showCategoryLabels']/text()"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="'true'"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <xsl:variable name="showValueLabels">
      <xsl:choose>
        <xsl:when test="ch:option[@name='showValueLabels']/text()!=''">
          <xsl:value-of select="ch:option[@name='showValueLabels']/text()"/>
        </xsl:when>
        <xsl:otherwise>
           <xsl:value-of select="'true'"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <xsl:choose>
      <xsl:when test="ch:option[@name='isVertical']='true'">
        <xsl:choose>
          <xsl:when test="$showCategoryLabels='false' and $showValueLabels='false'">
            <!-- omit -->
          </xsl:when>
          <xsl:when test="$showCategoryLabels='true' and $showValueLabels='false'">
            <xsl:text>&amp;chxt=x</xsl:text>
            <xsl:apply-templates select="../ch:data/ch:labels" mode="chart.labels.line">
              <xsl:with-param name="min" select="$min"/>
              <xsl:with-param name="max" select="$max"/>
            </xsl:apply-templates>
          </xsl:when>
          <xsl:when test="$showCategoryLabels='false' and $showValueLabels='true'">
            <xsl:text>&amp;chxt=y</xsl:text>
          </xsl:when>
          <!--  default showCategoryLabels true and showValueLabels true -->
          <xsl:otherwise>
            <xsl:text>&amp;chxt=x,y</xsl:text>
            <xsl:apply-templates select="../ch:data/ch:labels" mode="chart.labels.line">
              <xsl:with-param name="min" select="$min"/>
              <xsl:with-param name="max" select="$max"/>
            </xsl:apply-templates>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:when>
      <xsl:otherwise>
        <xsl:choose>
          <xsl:when test="$showCategoryLabels='false' and $showValueLabels='false'">
            <!-- omit -->
          </xsl:when>
          <xsl:when test="$showCategoryLabels='true' and $showValueLabels='false'">
            <xsl:text>&amp;chxt=y</xsl:text>
            <xsl:apply-templates select="../ch:data/ch:labels" mode="chart.labels.bar.horizontal">
              <xsl:with-param name="min" select="$min"/>
              <xsl:with-param name="max" select="$max"/>
            </xsl:apply-templates>
          </xsl:when>
          <xsl:when test="$showCategoryLabels='false' and $showValueLabels='true'">
            <xsl:text>&amp;chxt=x</xsl:text>
          </xsl:when>
          <!--  default showCategoryLabels true and showValueLabels true -->
          <xsl:otherwise>
            <xsl:text>&amp;chxt=y,x</xsl:text>
            <xsl:apply-templates select="../ch:data/ch:labels" mode="chart.labels.bar.horizontal">
              <xsl:with-param name="min" select="$min"/>
              <xsl:with-param name="max" select="$max"/>
            </xsl:apply-templates>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:otherwise>
    </xsl:choose>

     <xsl:choose>
        <xsl:when test="ch:option[@name='showValueLabels']/text()='false'">
          <!--  omnit -->
        </xsl:when>
        <xsl:otherwise>
           <xsl:choose>
             <xsl:when test="not(ch:option[@name='axisValueLabel']) and ch:option[@name='valueLabelsInterval']/text()!=''">
               <xsl:text>&amp;chxr=</xsl:text>
               <xsl:choose>
                 <xsl:when test="$showCategoryLabels='false'">
                   <xsl:text>0,</xsl:text>
                 </xsl:when>
                 <xsl:otherwise>
                   <xsl:text>1,</xsl:text>
                 </xsl:otherwise>
               </xsl:choose>
               <xsl:value-of select="encode:forUriComponent($min)"/>
               <xsl:text>,</xsl:text>
               <xsl:value-of select="encode:forUriComponent($max)"/>
               <xsl:text>,</xsl:text>
               <xsl:value-of select="encode:forUriComponent(ch:option[@name='valueLabelsInterval']/text())"/>
             </xsl:when>
             <xsl:otherwise>
               <xsl:text>&amp;chxr=</xsl:text>
               <xsl:choose>
                 <xsl:when test="$showCategoryLabels='false'">
                   <xsl:text>0,</xsl:text>
                 </xsl:when>
                 <xsl:otherwise>
                   <xsl:text>1,</xsl:text>
                 </xsl:otherwise>
               </xsl:choose>
              <xsl:value-of select="encode:forUriComponent($min)"/>
              <xsl:text>,</xsl:text>
              <xsl:value-of select="encode:forUriComponent($max)"/>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template match="ch:labels" mode="chart.labels.bar.horizontal">
    <xsl:param name="min"/>
    <xsl:param name="max"/>
    <xsl:text>&amp;chxl=</xsl:text>
    <xsl:choose>
      <xsl:when test="string-length(ch:emitLabels(ch:dataLabels(.), 'true'))!=0">
        <xsl:text>0:%7C</xsl:text>
        <xsl:value-of select="ch:emitLabels(ch:dataLabels(.), 'true')"/>
      </xsl:when>
        <!-- omit -->
    </xsl:choose>
    <xsl:if test="string-length(ch:emitLabels(ch:dataLabels(.), 'false'))!=0 and ../../ch:view/ch:option[@name='axisValueLabel']!=''">
      <xsl:text>%7C</xsl:text>
    </xsl:if>
    <xsl:choose>
      <xsl:when test="../../ch:view/ch:option[@name='axisValueLabel']!=''">
        <xsl:text>1:%7C</xsl:text>
        <xsl:value-of select="encode:forUriComponent($min)"></xsl:value-of>
        <xsl:text>%7C</xsl:text>
        <xsl:value-of select="($max + $min) div 2"></xsl:value-of>
        <xsl:text>%7C</xsl:text>
        <xsl:value-of select="encode:forUriComponent($max)"/>
        <xsl:text> </xsl:text>
        <xsl:value-of select="encode:forUriComponent(../../ch:view/ch:option[@name='axisValueLabel']/text())"/>
      </xsl:when>
        <!-- omit -->
    </xsl:choose>
  </xsl:template>

  <!-- Legend construction (depends on chart type) -->

  <xsl:template match="ch:chart" mode="chart.legend">
    <xsl:choose>
      <xsl:when test="ch:view/ch:option[@name='legend']/text()='none'">
        <!-- omit -->
      </xsl:when>
      <xsl:otherwise>
        <xsl:choose>
          <xsl:when test="ch:view[@type='pie']">
            <xsl:apply-templates select="ch:data" mode="chart.legend.pie"/>
          </xsl:when>
          <xsl:when test="ch:view[@type='line']">
            <xsl:apply-templates select="ch:data" mode="chart.legend.line.bar"/>
          </xsl:when>
          <xsl:when test="ch:view[@type='bar']">
            <xsl:apply-templates select="ch:data" mode="chart.legend.line.bar"/>
          </xsl:when>
        </xsl:choose>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template match="ch:data" mode="chart.legend.pie">
    <xsl:text>&amp;chdl=</xsl:text>
    <xsl:apply-templates select="./ch:labels" mode="chart.legend.pie"/>
  </xsl:template>

  <xsl:template match="ch:labels" mode="chart.legend.pie">
    <xsl:choose>
      <xsl:when test="*">
        <xsl:apply-templates select="./ch:label" mode="chart.legend.pie"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:for-each select="str:tokenize(text(), ',')">
          <xsl:value-of select="encode:forUriComponent(normalize-space(.))"/>
            <xsl:if test="position() != last()">
              <xsl:text>%7C</xsl:text>
            </xsl:if>
         </xsl:for-each>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>

  <xsl:template match="ch:label" mode="chart.legend.pie">
    <xsl:value-of select='encode:forUriComponent(text())'/>
    <xsl:if test="position() != last()">
       <xsl:text>%7C</xsl:text>
    </xsl:if>
  </xsl:template>

  <xsl:template match="ch:data" mode="chart.legend.line.bar">
    <xsl:if test="ch:series[@name]">
      <xsl:text>&amp;chdl=</xsl:text>
      <xsl:apply-templates select="ch:series[@name]" mode="chart.legend.line.bar"/>
    </xsl:if>
  </xsl:template>

  <xsl:template match="ch:series[@name]" mode="chart.legend.line.bar">
    <xsl:value-of select='encode:forUriComponent(@name)'/>
    <xsl:if test="position() != last()">
       <xsl:text>%7C</xsl:text>
    </xsl:if>
  </xsl:template>


  <!-- Encoded chart data -->

  <xsl:template match="ch:data" mode="chart.data">
    <xsl:param name="min"/>
    <xsl:param name="max"/>

    <xsl:text>&amp;chd=e:</xsl:text>

    <xsl:apply-templates select="ch:series" mode="chart.data.encode">
      <xsl:with-param name="min" select="$min"/>
      <xsl:with-param name="max" select="$max"/>
    </xsl:apply-templates>
  </xsl:template>

  <xsl:template match="ch:series" mode="chart.data.encode">
    <xsl:param name="min"/>
    <xsl:param name="max"/>
    <xsl:value-of select="ch:dataEncodeSet(ch:dataSerie(), $min, $max)"/>
    <xsl:if test="position() != last()">
      <xsl:text>,</xsl:text>
    </xsl:if>
  </xsl:template>


  <!-- Utility templates and functions -->

  <xsl:template name="chart.data.min">
    <xsl:variable name="relmin">
      <nodeset>
        <xsl:for-each select="ch:data/ch:series">
          <node><xsl:value-of select="math:min(ch:dataSerie())"/></node>
        </xsl:for-each>
      </nodeset>
    </xsl:variable>
    <xsl:value-of select="math:min(exsl:node-set($relmin)/nodeset/node)"/>
  </xsl:template>

  <xsl:template name="chart.data.max">
    <xsl:variable name="relmax">
      <nodeset>
        <xsl:for-each select="ch:data/ch:series">
          <node><xsl:value-of select="math:max(ch:dataSerie())"/></node>
        </xsl:for-each>
      </nodeset>
    </xsl:variable>
    <xsl:value-of select="math:max(exsl:node-set($relmax)/nodeset/node)"/>
  </xsl:template>

  <xsl:template name="chart.data.max.stacked">
    <xsl:variable name="series" select="ch:data/ch:series"/>
    <xsl:variable name="stacks">
      <nodeset>
        <xsl:for-each select="ch:dataSerie($series[1])">
          <xsl:call-template name="chart.data.element.stacked">
            <xsl:with-param name="series" select="$series"/>
          </xsl:call-template>
        </xsl:for-each>
      </nodeset>
    </xsl:variable>
    <xsl:value-of select="math:max(exsl:node-set($stacks)/nodeset/node/text())"/>
  </xsl:template>

  <xsl:template name="chart.data.element.stacked">
    <xsl:param name="series"/>
    <xsl:variable name="pos" select="position()"/>
    <node>
      <!--
        We cannot simply use sum() since we may mix <ch:element> and comma
        separated strings. So once again build the nodeset of numbers to sum
      -->
      <xsl:variable name="stackvalues">
        <nodeset>
          <xsl:for-each select="$series">
            <node><xsl:value-of select="ch:dataSerie()[$pos]"/></node>
          </xsl:for-each>
        </nodeset>
      </xsl:variable>
      <xsl:value-of select="sum(exsl:node-set($stackvalues)/nodeset/node/text())"/>
    </node>
  </xsl:template>

  <!--
    Get nodeset of values of a ch:serie supporting both ch:element
    syntax or comma-separated syntax
  -->
  <func:function name="ch:dataSerie">
    <xsl:param name="serie" select="." />
    <xsl:choose>
      <xsl:when test="$serie/ch:element">
        <func:result select="$serie/ch:element/text()"/>
      </xsl:when>
      <xsl:otherwise>
        <!-- Remove empty values from the list to avoid NaN when calculating max and min -->
        <xsl:variable name="raw" select="str:tokenize(normalize-space($serie/text()), ',')"/>
        <func:result select="$raw[normalize-space(.) != '']"/>
      </xsl:otherwise>
    </xsl:choose>
  </func:function>

  <!--
    Get nodeset of values of a ch:labels supporting both ch:label
    syntax or comma-separated syntax
  -->
  <func:function name="ch:dataLabels">
    <xsl:param name="labels" select="." />
    <xsl:choose>
      <xsl:when test="$labels/ch:label">
        <func:result select="$labels/ch:label"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:variable name="labelsNormalized">
          <xsl:for-each select="str:tokenize(text(), ',')">
            <xsl:value-of select="normalize-space(.)"/>
              <xsl:if test="position() != last()">
                <xsl:text>,</xsl:text>
              </xsl:if>
          </xsl:for-each>
        </xsl:variable>
         <func:result select="str:tokenize($labelsNormalized, ',')"/>
      </xsl:otherwise>
    </xsl:choose>
  </func:function>

  <func:function name="ch:emitLabels">
    <xsl:param name="labels" />
    <xsl:param name="reverse" select="'false'"/>
    <xsl:variable name="result">
      <xsl:choose>
        <xsl:when test="$reverse='true'">
          <xsl:for-each select="$labels">
            <xsl:sort select="position()" data-type="number" order="descending"/>
            <xsl:value-of select="encode:forUriComponent(text())"/>
            <xsl:if test="position() != last()">
              <xsl:text>%7C</xsl:text>
            </xsl:if>
          </xsl:for-each>
        </xsl:when>
        <xsl:otherwise>
          <xsl:for-each select="$labels">
            <xsl:value-of select="encode:forUriComponent(text())"/>
            <xsl:if test="position() != last()">
              <xsl:text>%7C</xsl:text>
            </xsl:if>
          </xsl:for-each>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <func:result select="string($result)"/>
  </func:function>

  <!--
    Encode data according to google chart extended encoding spec
  -->
  <xsl:variable name="CHART_ENCODING_MAP">ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-.</xsl:variable>
  <xsl:variable name="CHART_ENCODING_MAP_LEN" select = "string-length($CHART_ENCODING_MAP)"/>
  <func:function name="ch:dataEncode">
    <xsl:param name="val" select="''" />
    <xsl:param name="min" select="''" />
    <xsl:param name="max" select="''" />
    <func:result>
      <xsl:variable name="valScaled" select="$val - $min"/>
      <xsl:variable name="maxScaled" select="$max - $min"/>
      <xsl:variable name="scaled" select="floor($CHART_ENCODING_MAP_LEN * $CHART_ENCODING_MAP_LEN * $valScaled div $maxScaled)"/>
      <xsl:choose>
        <xsl:when test="$scaled &gt; $CHART_ENCODING_MAP_LEN * $CHART_ENCODING_MAP_LEN - 1">
          <xsl:text>..</xsl:text>
        </xsl:when>
        <xsl:when test="$scaled &lt; 0">
          <xsl:text>__</xsl:text>
        </xsl:when>
        <xsl:otherwise>
          <xsl:variable name="quotient" select="floor($scaled div $CHART_ENCODING_MAP_LEN)"/>
          <xsl:variable name="remainder" select="$scaled - $CHART_ENCODING_MAP_LEN * $quotient"/>
          <xsl:value-of select="concat(substring($CHART_ENCODING_MAP,$quotient+1,1), substring($CHART_ENCODING_MAP,$remainder+1,1))"/>
        </xsl:otherwise>
      </xsl:choose>
    </func:result>
  </func:function>

  <func:function name="ch:dataEncodeSet">
    <xsl:param name="values" select="''" />
    <xsl:param name="min" select="''" />
    <xsl:param name="max" select="''" />
    <func:result>
      <xsl:for-each select="$values">
        <xsl:value-of select='ch:dataEncode(., $min, $max)'/>
      </xsl:for-each>
    </func:result>
  </func:function>

  <func:function name="ch:replace">
    <xsl:param name="text" />
    <xsl:param name="from" />
    <xsl:param name="to" />
    <func:result>
      <xsl:choose>
        <xsl:when test="contains($text, $from)">
          <xsl:variable name="before" select="substring-before($text, $from)" />
          <xsl:variable name="after" select="substring-after($text, $from)" />
          <xsl:value-of select="concat($before, $to, ch:replace($after, $from, $to))" />
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="$text" />
        </xsl:otherwise>
      </xsl:choose>
    </func:result>
  </func:function>

  <func:function name="ch:colorSubstring">
    <xsl:variable name="COLOR_ENCODING_MAP">0369CF</xsl:variable>
    <xsl:variable name="random" >
          <xsl:value-of select="floor(math:random()*6)+1"/>
    </xsl:variable>
    <xsl:variable name="colorSubstring" >
      <xsl:value-of select="substring($COLOR_ENCODING_MAP,$random,1)"/>
    </xsl:variable>
    <func:result>
        <xsl:value-of select="concat($colorSubstring,$colorSubstring)"/>
    </func:result>
  </func:function>

  <func:function name="ch:color">
    <func:result>
      <xsl:value-of select="concat(ch:colorSubstring(),ch:colorSubstring(),ch:colorSubstring())"/>
    </func:result>
  </func:function>

</xsl:stylesheet>
