<?xml version="1.0" encoding="UTF-8"?>
<!--
  * Copyright 2023-2025 by NI SP Software GmbH, All rights reserved.
 * Copyright 1999-2023 by Nice, srl., All rights reserved.
 *
 * This software includes confidential and proprietary information
 * of NI SP Software GmbH ("Confidential Information").
 * You shall not disclose such Confidential Information
 * and shall use it only in accordance with the terms of
 * the license agreement you entered into with NI SP Software.-->
<xsl:stylesheet version="1.0"
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
  xmlns:m="http://informatik.hu-berlin.de/merge"
  xmlns:dyn="http://exslt.org/dynamic"
  xmlns:str="http://exslt.org/strings"
  xmlns:func="http://exslt.org/functions"
  xmlns:date="http://exslt.org/dates-and-times"
  xmlns:exsl="http://exslt.org/common"
  xmlns:xalan="http://xml.apache.org/xalan"
  xmlns:regexp="http://exslt.org/regular-expressions"
  xmlns:filter="http://www.enginframe.com/2010/EnginFrame/Filter"
  xmlns:java="http://xml.apache.org/xalan/java"
  extension-element-prefixes="m dyn str func date exsl regexp filter"
  exclude-result-prefixes="m dyn str func date exsl xalan regexp filter java"
>

  <!-- output format -->
  <xsl:output
    indent="yes"
    xalan:indent-amount="2"
    method="xml"
    encoding="UTF-8"
    media-type="text/xml"
    omit-xml-declaration="no" />


  <!-- path to configuration file -->
  <xsl:param name="conf" />


  <!-- input flags and parameters -->
  <xsl:param name="filtered" />
  <xsl:param name="sorted" />
  <xsl:param name="paginated" />
  <xsl:param name="filter" />
  <xsl:param name="sort-by" />
  <xsl:param name="max-results" />
  <xsl:param name="start-index" />
  <xsl:param name="debug" />
  <xsl:param name="emit-conf" />
  <xsl:param name="merge-by" />
  <xsl:param name="merge-mode" />


  <!-- default values of input flags and parameters -->
  <xsl:param name="filtered_default"   >false</xsl:param>
  <xsl:param name="sorted_default"     >false</xsl:param>
  <xsl:param name="paginated_default"  >false</xsl:param>
  <xsl:param name="filter_default"     ></xsl:param>
  <xsl:param name="sort-by_default"    ></xsl:param>
  <xsl:param name="max-results_default">10</xsl:param>
  <xsl:param name="start-index_default">1</xsl:param>
  <xsl:param name="debug_default"      >false</xsl:param>
  <xsl:param name="emit-conf_default"  >false</xsl:param>
  <xsl:param name="merge-by_default"   ></xsl:param>
  <xsl:param name="merge-mode_default" >append</xsl:param>


  <!-- definition of operators -->
  <xsl:variable name="eq"       >=</xsl:variable>
  <xsl:variable name="ieq"      >?=</xsl:variable>
  <xsl:variable name="quot"     >"</xsl:variable>
  <xsl:variable name="apos"     >'</xsl:variable>
  <xsl:variable name="lparen"   >(</xsl:variable>
  <xsl:variable name="rparen"   >)</xsl:variable>
  <xsl:variable name="negate"   >!</xsl:variable>
  <xsl:variable name="neq"      >!=</xsl:variable>
  <xsl:variable name="regexp"   >~=</xsl:variable>
  <xsl:variable name="glob"     >*=</xsl:variable>
  <xsl:variable name="contains" >:=</xsl:variable>
  <xsl:variable name="icontains">?:=</xsl:variable>
  <xsl:variable name="or"       > or </xsl:variable>
  <xsl:variable name="and"      > and </xsl:variable>
  <xsl:variable name="not"      > not </xsl:variable>
  <xsl:variable name="gt"       >&gt;</xsl:variable>
  <xsl:variable name="lt"       >&lt;</xsl:variable>
  <xsl:variable name="gte"      >&gt;=</xsl:variable>
  <xsl:variable name="lte"      >&lt;=</xsl:variable>
  <xsl:variable name="comma"    >,</xsl:variable>


  <!-- build variable with values from <xsl:param> -->
  <xsl:variable name="_my_params_RTF">
    <filter:params
         filtered="{$filtered}"
           sorted="{$sorted}"
        paginated="{$paginated}"
           filter="{$filter}"
          sort-by="{$sort-by}"
      max-results="{$max-results}"
      start-index="{$start-index}"
            debug="{$debug}"
        emit-conf="{$emit-conf}"
         merge-by="{$merge-by}"
       merge-mode="{$merge-mode}"
    />
  </xsl:variable>
  <!-- store variable in a real node-set, not a RTF -->
  <xsl:variable name="my_params"
    select="exsl:node-set($_my_params_RTF)/filter:params" />


  <!-- build variable with default values -->
  <xsl:variable name="_my_default_RTF">
    <filter:default
         filtered="{$filtered_default}"
           sorted="{$sorted_default}"
        paginated="{$paginated_default}"
           filter="{$filter_default}"
          sort-by="{$sort-by_default}"
      max-results="{$max-results_default}"
      start-index="{$start-index_default}"
            debug="{$debug_default}"
        emit-conf="{$emit-conf_default}"
         merge-by="{$merge-by_default}"
       merge-mode="{$merge-mode_default}"
    />
  </xsl:variable>
  <!-- store variable in a real node-set, not a RTF -->
  <xsl:variable name="my_default"
    select="exsl:node-set($_my_default_RTF)/filter:default" />


  <!-- get initial configuration -->
  <xsl:variable name="_raw-mapper-list_RTF">
    <xsl:variable name="external-mapper-list">
      <!-- first copy the param -->
      <xsl:choose>
        <!-- param is a node set, so simply copy it -->
        <xsl:when test="exsl:object-type($conf)='node-set'">
          <xsl:copy-of select="$conf" />
        </xsl:when>
        <!-- param is a string, so load the document -->
        <xsl:when test="$conf and not($conf='')">
          <xsl:copy-of select="document($conf)" />
        </xsl:when>
      </xsl:choose>
    </xsl:variable>
    <!-- then get all other <filter:mapper-list> sorted by @priority -->
    <xsl:copy-of select="filter:sort-by-priority(
        exsl:node-set($external-mapper-list)/filter:mapper-list |
        //filter:mapper-list
      )" />
  </xsl:variable>
  <!-- store variable in a real node-set, not a RTF -->
  <xsl:variable name="raw-mapper-list"
    select="exsl:node-set($_raw-mapper-list_RTF)/filter:mapper-list" />


  <!-- root element from $raw-mapper-list -->
  <xsl:variable name="my_root" select="$raw-mapper-list/@root" />


  <!-- row element from $raw-mapper-list -->
  <xsl:variable name="my_row" select="$raw-mapper-list/@row" />


  <!-- build the final <filter:mapper-list> -->
  <xsl:key name="mappers-by-id" match="filter:mapper" use="@id" />
  <xsl:variable name="_final-mapper-list_RTF">
    <filter:mapper-list>
      <!-- merge all attributes -->
      <xsl:copy-of select="$raw-mapper-list/@*" />
      <!-- merge all children -->
      <xsl:for-each select="$raw-mapper-list/filter:mapper[count(.|key('mappers-by-id',@id)[last()])=1]" >
        <xsl:copy>
          <xsl:copy-of select="@*" />
          <xsl:attribute name="xpath">
            <xsl:value-of select="filter:expand-namespace(@select,$my_rownamespaceprefix,$my_rownamespaceuri)" />
          </xsl:attribute>
        </xsl:copy>
      </xsl:for-each>
    </filter:mapper-list>
  </xsl:variable>
  <!-- store variable in a real node-set, not a RTF -->
  <xsl:variable name="mapper-list"
    select="exsl:node-set($_final-mapper-list_RTF)/filter:mapper-list" />


  <!-- namespace URI of root and rows -->
  <xsl:variable name="my_rootnamespaceuri" select="namespace-uri(//*[name()=$my_root][1])" />
  <xsl:variable name="my_rownamespaceuri" select="namespace-uri(//*[name()=$my_root]/*[name()=$my_row][1])" />


  <!-- namespace prefix of root and rows -->
  <xsl:variable name="my_rootnamespaceprefix" select="concat(substring-before($my_root,':'),':')" />
  <xsl:variable name="my_rownamespaceprefix" select="concat(substring-before($my_row,':'),':')" />


  <!-- <filter:header> elements sorted by priority -->
  <xsl:variable name="_prioritized-header_RTF"
    select="filter:sort-by-priority(//filter:header)" />
  <xsl:variable name="prioritized-header"
    select="exsl:node-set($_prioritized-header_RTF)/*" />


  <!-- root elements sorted by priority -->
  <xsl:variable name="_prioritized-root-elements_RTF"
    select="filter:sort-by-priority(//*[name()=$my_root])" />
  <xsl:variable name="prioritized-root-elements"
    select="exsl:node-set($_prioritized-root-elements_RTF)/*" />


  <!-- build the final <filter:header> -->
  <xsl:variable name="_header_RTF">
    <filter:header>
      <!-- merge all attributes from all <filter:header> -->
      <xsl:copy-of select="$prioritized-header/@*" />
      <!-- merge all attributes from all root-elements -->
      <xsl:copy-of select="$prioritized-root-elements/@*" />
    </filter:header>
  </xsl:variable>
  <!-- store variable in a real node-set, not a RTF -->
  <xsl:variable name="header"
    select="exsl:node-set($_header_RTF)/filter:header" />


  <!-- xpath expression to identify the rows to merge -->
  <xsl:variable name="my_merge-by">
    <xsl:variable name="user-defined_merge-by" select="filter:get-value('merge-by')" />
    <xsl:choose>
      <!-- first, check if there is an user-defined value -->
      <xsl:when test="$user-defined_merge-by and not($user-defined_merge-by='')">
        <xsl:value-of select="$user-defined_merge-by" />
      </xsl:when>
      <!-- else, check if the row tag has an attribute "id" -->
      <xsl:when test="$prioritized-root-elements/*[name()=$my_row]/@id">@id</xsl:when>
      <!-- otherwise, do not merge at all the row tags -->
      <xsl:otherwise>generate-id()</xsl:otherwise>
    </xsl:choose>
  </xsl:variable>


  <!-- input flags and parameters from xsl:param or @attribute or filter:header -->
  <xsl:variable name="my_filtered" select="filter:get-value('filtered') = 'true'" />
  <xsl:variable name="my_sorted" select="filter:get-value('sorted') = 'true'" />
  <xsl:variable name="my_paginated" select="filter:get-value('paginated') = 'true'" />
  <xsl:variable name="my_filter" select="filter:get-value('filter')" />
  <xsl:variable name="my_sort-by" select="filter:get-value('sort-by')" />
  <xsl:variable name="my_max-results" select="filter:get-value('max-results')" />
  <xsl:variable name="my_start-index" select="filter:get-value('start-index')" />
  <xsl:variable name="my_debug" select="filter:get-value('debug')" />
  <xsl:variable name="my_emit-conf" select="filter:get-value('emit-conf')" />
  <xsl:variable name="my_merge-mode" select="filter:get-value('merge-mode')" />


  <!-- should I filter, sort or paginate? -->
  <xsl:variable name="must-filter" select="
      not($my_filtered) and
      $my_filter and not($my_filter='')
    " />
  <xsl:variable name="must-sort" select="
      not($my_sorted) and
      $my_sort-by and not($my_sort-by='')
    " />
  <xsl:variable name="must-paginate" select="
      not($my_paginated) and
      $my_start-index and not($my_start-index='') and
      $my_max-results and not($my_max-results='')
    " />


  <!-- build my_filter-by string -->
  <xsl:variable name="my_compact-filter-by">
    <xsl:if test="$must-filter">
      <xsl:value-of select="filter:parse-filter($my_filter)" />
    </xsl:if>
  </xsl:variable>
  <!-- expand namespaces -->
  <xsl:variable name="my_filter-by"
    select="filter:expand-namespace(
        $my_compact-filter-by,
        $my_rownamespaceprefix,
        $my_rownamespaceuri
      )" />


  <!-- unique identifier of sort from input parameter --> 
  <xsl:variable name="my_sort-id">
    <xsl:if test="$must-sort">
      <xsl:choose>
        <xsl:when test="contains($my_sort-by,':')">
          <xsl:value-of select="substring-before($my_sort-by,':')" />
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="$my_sort-by" />
        </xsl:otherwise>
      </xsl:choose>
    </xsl:if>
  </xsl:variable>


  <!-- data-type from configuration file -->
  <xsl:variable name="sort-type"
    select="filter:read-conf($my_sort-id,'type')" />
  <xsl:variable name="my_sort-type">
    <xsl:if test="$must-sort">
      <xsl:choose>
        <xsl:when test="$sort-type = 'text' or $sort-type = 'number'">
          <xsl:value-of select="$sort-type" />
        </xsl:when>
        <xsl:when test="$sort-type = 'unit'">
          <xsl:text>number</xsl:text>
        </xsl:when>
        <xsl:when test="$sort-type = 'date'">
          <xsl:text>text</xsl:text>
        </xsl:when>
        <xsl:otherwise>text</xsl:otherwise>
      </xsl:choose>
    </xsl:if>
  </xsl:variable>


  <!-- unique identifier of column to be sorted --> 
  <xsl:variable name="sort-select"
    select="filter:read-conf($my_sort-id,'select')" />
  <xsl:variable name="my_compact-sort-select">
    <xsl:if test="$must-sort">
      <xsl:choose>
        <xsl:when test="$sort-type = 'text' or $sort-type = 'number'">
          <xsl:value-of select="$sort-select" />
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="concat(
              'filter:',
              $sort-type,
              '(',$sort-select,')'
            )" />
        </xsl:otherwise>
      </xsl:choose>
    </xsl:if>
  </xsl:variable>
  <!-- expand namespaces -->
  <xsl:variable name="my_sort-select"
    select="filter:expand-namespace(
        $my_compact-sort-select,
        $my_rownamespaceprefix,
        $my_rownamespaceuri
      )" />


  <!-- sort order --> 
  <xsl:variable name="my_sort-order" >
    <xsl:if test="$must-sort">
      <xsl:choose>
        <xsl:when test="substring-after($my_sort-by,':')='desc'">
          <xsl:text>descending</xsl:text>
        </xsl:when>
        <xsl:otherwise>
          <xsl:text>ascending</xsl:text>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:if>
  </xsl:variable>


  <!-- start-index that must not be greater than the total -->
  <xsl:variable name="my_start">
    <xsl:choose>
      <xsl:when test="$my_max-results = -1">1</xsl:when>
      <xsl:when test="$my_start-index > $my_final-results">
        <xsl:value-of select="
            floor($my_final-results div $my_max-results)
            * $my_max-results
            + 1
          " />
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="$my_start-index" />
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>


  <!-- calculate stop-index -->
  <xsl:variable name="my_stop">
    <xsl:choose>
      <xsl:when test="$my_max-results = -1">
        <xsl:value-of select="$my_final-results" />
      </xsl:when>
      <xsl:when test="$must-paginate">
        <xsl:value-of select="$my_start+ $my_max-results - 1" />
      </xsl:when>
    </xsl:choose>
  </xsl:variable>


  <!-- merge all rows -->
  <xsl:key name="rows" match="//*[name()=$my_root]/*[name()=$my_row]" use="dyn:evaluate($my_merge-by)" />
  <xsl:variable name="_my_merged_xml_RTF">
    <xsl:for-each select="$prioritized-root-elements/*[name()=$my_row
        and count( . | key('rows', dyn:evaluate($my_merge-by) )[1] )=1
      ]" >
      <xsl:copy>
        <xsl:copy-of select="key('rows',dyn:evaluate($my_merge-by))/@*" />
        <xsl:choose>
          <xsl:when test="$my_merge-mode='replace'">
            <xsl:copy-of select="filter:merge(key('rows',dyn:evaluate($my_merge-by)))" />
          </xsl:when>
          <xsl:otherwise><!-- $my_merge-mode='append' or something else -->
            <xsl:copy-of select="key('rows',dyn:evaluate($my_merge-by))/*" />
          </xsl:otherwise>
        </xsl:choose>
      </xsl:copy>
    </xsl:for-each>
  </xsl:variable>
  <!-- store xml in a real node-set, not a RTF -->
  <xsl:variable name="my_merged_xml" select="exsl:node-set($_my_merged_xml_RTF)" />


  <!-- apply all templates with mode="filter" to merged rows -->
  <xsl:variable name="_my_xml_RTF">
    <xsl:apply-templates select="node()" mode="filter:get-xml" />
  </xsl:variable>
  <!-- store xml in a real node-set, not a RTF -->
  <xsl:variable name="my_xml" select="exsl:node-set($_my_xml_RTF)" />


  <!-- include first root element -->
  <xsl:template match="*[name()=$my_root][1]" mode="filter:get-xml">
    <xsl:copy>
      <xsl:copy-of select="$header/@*" />
      <xsl:apply-templates select="$my_merged_xml/*" mode="filter" />
      <xsl:copy-of select="$prioritized-root-elements/*[not(name()=$my_row)]" />
    </xsl:copy>
  </xsl:template>

  <xsl:template match="*[name()=$my_row and @hidden='true']" mode="filter" />

  <!-- exclude other root elements -->
  <xsl:template match="*[name()=$my_root][position()>1]" mode="filter:get-xml" />


  <!-- do not propagate processing instructions with name()="xml-stylesheet" -->
  <xsl:template match="processing-instruction()[name()='xml-stylesheet']" mode="filter:get-xml" />


  <!-- identity template -->
  <xsl:template match="node()|@*" mode="filter:get-xml">
    <xsl:copy>
      <xsl:apply-templates select="node()|@*" mode="filter:get-xml" />
    </xsl:copy>
  </xsl:template>


  <!-- total number of rows without pagination without filter -->
  <xsl:variable name="my_total-results" select="count($my_xml//*[name()=$my_root]/*[name()=$my_row])" />

  <!-- total number of rows without pagination with filter -->
  <xsl:variable name="my_final-results" select="count($my_rows)" />


  <!-- match the whole document -->
  <xsl:template match="/">
    <xsl:text>&#x0A;</xsl:text>
    <xsl:if test="$my_debug='true'">
      <xsl:copy-of select="$my_info" />
    </xsl:if>
    <xsl:text>&#x0A;</xsl:text>
    <xsl:apply-templates select="$my_xml" mode="filter:identity" />
  </xsl:template>

 
  <!-- write transformation info inside a comment -->
  <xsl:variable name="my_info">
    <xsl:comment>
      <xsl:value-of select="concat('&#x0A;',
          'root:          ', filter:tag-description($my_root,$my_rootnamespaceuri),'&#x0A;',
          'row:           ', filter:tag-description($my_row,$my_rownamespaceuri),'&#x0A;',
          'merge-by:      ', $my_merge-by,'&#x0A;',
          'merge-mode:    ', $my_merge-mode,'&#x0A;',
          'filtered:      ', $my_filtered,'&#x0A;',
          'sorted:        ', $my_sorted,'&#x0A;',
          'paginated:     ', $my_paginated,'&#x0A;',
          'filter:        ', $my_filter,'&#x0A;',
          'sort-by:       ', $my_sort-by,'&#x0A;',
          'max-results:   ', $my_max-results,'&#x0A;',
          'start-index:   ', $my_start-index,'&#x0A;',
          'debug:         ', $my_debug,'&#x0A;',
          'emit-conf:     ', $my_emit-conf,'&#x0A;',
          'must-filter:   ', $must-filter,'&#x0A;',
          'must-sort:     ', $must-sort,'&#x0A;',
          'must-paginate: ', $must-paginate,'&#x0A;',
          'total-results: ', $my_total-results,'&#x0A;',
          'final-results: ', $my_final-results,'&#x0A;',
          'max-results:   ', $my_max-results,'&#x0A;',
          'filter-by1:    ', $my_compact-filter-by,'&#x0A;',
          'filter-by2:    ', $my_filter-by,'&#x0A;',
          'sort-by1:      ', $my_compact-sort-select,'&#x0A;',
          'sort-by2:      ', $my_sort-select,'&#x0A;',
          'sort-order:    ', $my_sort-order,'&#x0A;',
          'sort-type:     ', $my_sort-type,'&#x0A;',
          'range:         ', $my_start, '-', $my_stop,'&#x0A;',
          ''
        )" />
    </xsl:comment>
  </xsl:variable>


  <!-- identity template for mode="filter" -->
  <xsl:template match="node()|@*" mode="filter">
    <xsl:copy>
      <xsl:apply-templates select="node()|@*" mode="filter"/>
    </xsl:copy>
  </xsl:template>


  <!-- filter and sort rows if requested, no pagination at this moment -->
  <xsl:variable name="_my_rows_RTF">
    <xsl:choose>
      <xsl:when test="$must-sort">
        <xsl:apply-templates mode="filter:identity"
          select="$my_xml//*[name()=$my_root]/*[name()=$my_row and not($must-filter) or dyn:evaluate($my_filter-by) = 'true']">
          <xsl:sort select="dyn:evaluate($my_sort-select)"
                    data-type="{$my_sort-type}"
                    order="{$my_sort-order}" />
        </xsl:apply-templates>
      </xsl:when>
      <xsl:otherwise>
        <xsl:apply-templates mode="filter:identity"
          select="$my_xml//*[name()=$my_root]/*[name()=$my_row and not($must-filter) or dyn:evaluate($my_filter-by) = 'true']" />
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <!-- store rows in a real node-set, not a RTF -->
  <xsl:variable name="my_rows" select="exsl:node-set($_my_rows_RTF)/*" />


  <!-- number of rows to display -->
  <xsl:variable name="my_results">
    <xsl:choose>
      <xsl:when test="$must-paginate">
        <xsl:value-of select="count(
            $my_rows[position() >= $my_start and $my_stop >= position()]
          )" />
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="$my_total-results" />
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>


  <!-- match the root element -->
  <xsl:template match="*[name()=$my_root]" mode="filter:identity">
    <!-- copy root element -->
    <xsl:copy>

      <!-- set the proper attributes -->
      <xsl:call-template name="filter:setAttributes" />

      <!-- paginate if requested -->
      <xsl:choose>
        <xsl:when test="$must-paginate">
          <xsl:copy-of select="$my_rows[
              position() >= $my_start and $my_stop >= position()
            ]" />
        </xsl:when>
        <xsl:otherwise>
          <xsl:copy-of select="$my_rows" />
        </xsl:otherwise>
      </xsl:choose>

      <!-- copy all other tags -->
      <xsl:copy-of select="*[not(name()=$my_row)]" />

      <xsl:if test="$my_emit-conf='true'">
        <!-- copy header for further use -->
        <xsl:text>&#x0A;  </xsl:text>
        <xsl:copy-of select="$header" />
        <!-- copy mapper list for further use -->
        <xsl:text>&#x0A;  </xsl:text>
        <xsl:copy-of select="$mapper-list" />
      </xsl:if>

      <!-- pretty print -->
      <xsl:text>&#x0a;</xsl:text>
    </xsl:copy>
    <!-- pretty print -->
    <xsl:text>&#x0a;</xsl:text>
  </xsl:template>


  <!-- do not match these tags, because they are managed elsewhere -->
  <xsl:template match="filter:header|filter:mapper-list" mode="filter:identity" />


  <!-- match everything else -->
  <xsl:template match="node()|@*" mode="filter:identity">
    <xsl:copy>
      <xsl:apply-templates select="node()|@*" mode="filter:identity"/>
    </xsl:copy>
  </xsl:template>


  <!-- set the proper attributes -->
  <xsl:template name="filter:setAttributes">
    <!-- copy non-reserved attributes -->
    <xsl:copy-of select="@*[not(
        name()='filtered' or
        name()='sorted' or
        name()='paginated' or
        name()='filter' or
        name()='sort-by' or
        name()='max-results' or
        name()='start-index' or
        name()='results' or
        name()='total-results' or
        name()='conf'
      )]" />
    <!-- set reserved attributes -->
    <xsl:attribute name="filtered">
      <xsl:text>true</xsl:text>
    </xsl:attribute>
    <xsl:attribute name="sorted">
      <xsl:text>true</xsl:text>
    </xsl:attribute>
    <xsl:attribute name="paginated">
      <xsl:text>true</xsl:text>
    </xsl:attribute>
    <xsl:attribute name="filter">
      <xsl:value-of select="$my_filter"/>
    </xsl:attribute>
    <xsl:attribute name="sort-by">
      <xsl:value-of select="$my_sort-by"/>
    </xsl:attribute>
    <xsl:attribute name="max-results">
      <xsl:value-of select="$my_max-results"/>
    </xsl:attribute>
    <xsl:attribute name="start-index">
      <xsl:value-of select="$my_start"/>
    </xsl:attribute>
    <xsl:attribute name="results">
      <xsl:value-of select="$my_results" />
    </xsl:attribute>
    <xsl:attribute name="total-results">
      <xsl:value-of select="$my_final-results" />
    </xsl:attribute>
  </xsl:template>


  <!-- extension function for xalan -->
  <xalan:component prefix="regexp" functions="matchRegexp">
    <xalan:script lang="javaclass"
      src="xalan://com.enginframe.common.utils.RegexpUtil"/>
  </xalan:component>

  <func:function name="filter:globMatch">
    <xsl:param name="string" />
    <xsl:param name="glob" />
    <func:result select="java:com.enginframe.common.utils.GlobMatcher.globMatch(string($string), string($glob))" />
  </func:function>

  <func:function name="filter:equalsIgnoreCase">
    <xsl:param name="string1" />
    <xsl:param name="string2" />
    <func:result select="java:equalsIgnoreCase(string($string1), string($string2))" />
  </func:function>

  <func:function name="filter:containsIgnoreCase">
    <xsl:param name="string1" />
    <xsl:param name="string2" />
    <func:result select="java:indexOf(java:toLowerCase(string($string1)),java:toLowerCase(string($string2))) > -1" />
  </func:function>

  <!-- parse filter string -->
  <func:function name="filter:parse-filter">
    <xsl:param name="string" />
    <xsl:param name="close" />
    <xsl:variable name="operator" select="filter:get-operator($string)" />
    <xsl:variable name="result">
      <xsl:choose>
        <xsl:when test="$operator=''">
          <xsl:value-of select="$string" />
        </xsl:when>
        <xsl:when test='$operator=$apos or $operator=$quot'>
          <xsl:variable name="quote" select="filter:get-quote($string, $operator)" />
          <xsl:value-of select="substring-before($string, $operator)" />
          <xsl:value-of select="$quote" />
          <xsl:value-of select="$close" />
          <xsl:value-of select="filter:parse-filter(substring-after($string,$quote))" />
        </xsl:when>
        <xsl:when test="$operator=$comma">
          <xsl:variable name="key" select="filter:trim(substring-before($string, $operator))" />
          <xsl:value-of select="filter:read-conf($key,'select')" />
          <xsl:value-of select="concat(' ',$operator,' ')" />
          <xsl:value-of select="filter:parse-filter(substring-after($string, $operator))" />
        </xsl:when>
        <xsl:when test="$operator=$and or $operator = $or or $operator = $not
          or $operator = $lparen or $operator = $rparen">
          <xsl:value-of select="substring-before($string, $operator)" />
          <xsl:value-of select="concat(' ',$operator,' ')" />
          <xsl:value-of select="filter:parse-filter(substring-after($string, $operator))" />
        </xsl:when>
        <xsl:when test="$operator=$ieq">
          <xsl:variable name="key" select="filter:trim(substring-before($string, $operator))" />
          <xsl:text>filter:equalsIgnoreCase(</xsl:text>
          <xsl:value-of select="filter:read-conf($key,'select')" />
          <xsl:text>,</xsl:text>
          <xsl:value-of select="filter:parse-filter(substring-after($string, $operator),')')" />
        </xsl:when>
        <xsl:when test="$operator=$icontains">
          <xsl:variable name="key" select="filter:trim(substring-before($string, $operator))" />
          <xsl:text>filter:containsIgnoreCase(</xsl:text>
          <xsl:value-of select="filter:read-conf($key,'select')" />
          <xsl:text>,</xsl:text>
          <xsl:value-of select="filter:parse-filter(substring-after($string, $operator),')')" />
        </xsl:when>
        <xsl:when test="$operator=$contains">
          <xsl:variable name="key" select="filter:trim(substring-before($string, $operator))" />
          <xsl:text>contains(</xsl:text>
          <xsl:value-of select="filter:read-conf($key,'select')" />
          <xsl:text>,</xsl:text>
          <xsl:value-of select="filter:parse-filter(substring-after($string, $operator),')')" />
        </xsl:when>
        <xsl:when test="$operator=$regexp">
          <xsl:variable name="key" select="filter:trim(substring-before($string, $operator))" />
          <xsl:variable name="my_close" select="concat(', ',filter:read-conf($key,'select'),', 0)')" />
          <xsl:text>regexp:matchRegexp(</xsl:text>
          <xsl:value-of select="filter:parse-filter(substring-after($string, $operator),$my_close)" />
        </xsl:when>
        <xsl:when test="$operator=$glob">
          <xsl:variable name="key" select="filter:trim(substring-before($string, $operator))" />
          <xsl:text>filter:globMatch(</xsl:text>
          <xsl:value-of select="filter:read-conf($key,'select')" />
          <xsl:text>,</xsl:text>
          <xsl:value-of select="filter:parse-filter(substring-after($string, $operator),')')" />
        </xsl:when>
        <xsl:when test="$operator=$negate">
          <xsl:variable name="foo1" select="filter:trim(substring-after($string, $operator))" />
          <xsl:variable name="key">
            <xsl:choose>
              <xsl:when test="contains($foo1, ' ')">
                <xsl:value-of select="substring-before($foo1, ' ')" />
              </xsl:when>
              <xsl:otherwise>
                <xsl:value-of select="$foo1" />
              </xsl:otherwise>
            </xsl:choose>
          </xsl:variable>
          <xsl:variable name="xpath" select="filter:read-conf($key,'select')" />
          <xsl:text>not(</xsl:text>
          <xsl:value-of select="$xpath" />
          <xsl:text>) or </xsl:text>
          <xsl:value-of select="$xpath" />
          <xsl:text> = '' </xsl:text>
          <xsl:value-of select="filter:parse-filter(substring-after($string, $key))" />
        </xsl:when>
        <xsl:otherwise>
          <xsl:variable name="key" select="filter:trim(substring-before($string, $operator))" />
          <xsl:value-of select="filter:read-conf($key,'select')" />
          <xsl:value-of select="concat(' ',$operator,' ')" />
          <xsl:value-of select="filter:parse-filter(substring-after($string, $operator))" />
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <func:result>
      <xsl:value-of select="$result" />
    </func:result>
  </func:function>

  <func:function name="filter:get-operator">
    <xsl:param name="string" />
    <xsl:variable name="operator-list">
      <xsl:for-each select="exsl:node-set(filter:operator-list($string))/filter:operator[@index>0]">
        <xsl:sort select="@index" order="ascending" data-type="number" />
        <xsl:copy-of select="." />
      </xsl:for-each>
    </xsl:variable>
    <func:result>
      <xsl:value-of select="exsl:node-set($operator-list)/filter:operator[1]/@operator" />
    </func:result>
  </func:function>

  <func:function name="filter:get-quote">
    <xsl:param name="string" />
    <xsl:param name="operator" />
    <func:result>
      <xsl:value-of select="concat(
        $operator,
        substring-before(substring-after($string, $operator), $operator),
        $operator
      )" />
    </func:result>
  </func:function>

  <func:function name="filter:operator-list">
    <xsl:param name="string" />
    <func:result>
      <xsl:copy-of select="filter:operator($string,$gt)" />
      <xsl:copy-of select="filter:operator($string,$lt)" />
      <xsl:copy-of select="filter:operator($string,$gte)" />
      <xsl:copy-of select="filter:operator($string,$lte)" />
      <xsl:copy-of select="filter:operator($string,$or)" />
      <xsl:copy-of select="filter:operator($string,$and)" />
      <xsl:copy-of select="filter:operator($string,$not)" />
      <xsl:copy-of select="filter:operator($string,$neq)" />
      <xsl:copy-of select="filter:operator($string,$regexp)" />
      <xsl:copy-of select="filter:operator($string,$glob)" />
      <xsl:copy-of select="filter:operator($string,$contains)" />
      <xsl:copy-of select="filter:operator($string,$icontains)" />
      <xsl:copy-of select="filter:operator($string,$eq)" />
      <xsl:copy-of select="filter:operator($string,$ieq)" />
      <xsl:copy-of select="filter:operator($string,$quot)" />
      <xsl:copy-of select="filter:operator($string,$apos)" />
      <xsl:copy-of select="filter:operator($string,$lparen)" />
      <xsl:copy-of select="filter:operator($string,$rparen)" />
      <xsl:copy-of select="filter:operator($string,$negate)" />
      <xsl:copy-of select="filter:operator($string,$comma)" />
    </func:result>
  </func:function>

  <func:function name="filter:operator">
    <xsl:param name="string" />
    <xsl:param name="operator" />
    <func:result>
      <xsl:element name="filter:operator">
        <xsl:attribute name="operator">
          <xsl:value-of select="$operator" />
        </xsl:attribute>
        <xsl:attribute name="index">
          <xsl:value-of select="filter:index-of($string,$operator)" />
        </xsl:attribute>
      </xsl:element>
    </func:result>
  </func:function>


  <!-- trim spaces on the left -->
  <func:function name="filter:ltrim">
    <xsl:param name="string" />
    <xsl:variable name="s-no-ws" select="translate($string,' &#9;&#10;&#13;','')"/>
    <xsl:variable name="s-first-non-ws" select="substring($s-no-ws,1,1)"/>
    <xsl:variable name="s-no-leading-ws" select="concat($s-first-non-ws,substring-after($string,$s-first-non-ws))"/>
    <func:result>
      <xsl:value-of select="$s-no-leading-ws" />
    </func:result>
  </func:function>


  <!-- trim spaces on the right -->
  <func:function name="filter:rtrim">
    <xsl:param name="string"/>
    <xsl:param name="i" select="string-length($string)"/>
    <func:result>
      <xsl:choose>
        <xsl:when test="translate(substring($string,$i,1),' &#9;&#10;&#13;','')">
          <xsl:value-of select="substring($string,1,$i)"/>
        </xsl:when>
        <xsl:when test="$i&lt;2"/>
        <xsl:otherwise>
          <xsl:value-of select="filter:rtrim($string, $i - 1)" />
        </xsl:otherwise>
      </xsl:choose>
    </func:result>
  </func:function>


  <!-- trim spaces on both sides -->
  <func:function name="filter:trim">
    <xsl:param name="string" />
    <func:result>
      <xsl:value-of select="filter:rtrim(filter:ltrim($string))" />
    </func:result>
  </func:function>


  <!-- position of substring starting with 1, 0 if missing -->
  <func:function name="filter:index-of">
    <xsl:param name="string" />
    <xsl:param name="search" />
    <xsl:variable name="result">
      <xsl:choose>
        <xsl:when test="contains($string, $search)">
          <xsl:value-of select="string-length(substring-before($string, $search)) + 1" />
        </xsl:when>
        <xsl:otherwise>0</xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <func:result>
      <xsl:value-of select="$result" />
    </func:result>
  </func:function>


  <!-- read attribute from configuration file -->
  <func:function name="filter:read-conf">
    <xsl:param name="key" />
    <xsl:param name="attr" />
    <xsl:variable name="value" select="$mapper-list/filter:mapper[@id=$key]/@*[name()=$attr]" />
    <xsl:variable name="result">
      <xsl:choose>
        <xsl:when test="$value and not($value='')">
          <xsl:value-of select="$value" />
        </xsl:when>
        <xsl:when test="$attr='select'">
          <xsl:value-of select='concat("*[local-name()=&apos;",$key,"&apos;]")' />
        </xsl:when>
        <xsl:when test="$attr='type'">
          <xsl:text>text</xsl:text>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="$key" />
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <func:result>
      <xsl:value-of select="$result" />
    </func:result>
  </func:function>


  <!-- get input flags and parameters from param attribute header -->
  <func:function name="filter:get-value">
    <xsl:param name="name" />
    <xsl:variable name="param"     select="$my_params/@*[name()=$name]" />
    <xsl:variable name="mapper"    select="$mapper-list/@*[name()=$name]" />
    <xsl:variable name="attribute" select="$header/@*[name()=$name]" />
    <xsl:variable name="default"   select="$my_default/@*[name()=$name]" />
    <xsl:variable name="result">
      <xsl:choose>
        <xsl:when test="$param and not($param='')">
          <xsl:value-of select="$param" />
        </xsl:when>
        <xsl:when test="$mapper and not($mapper='')">
          <xsl:value-of select="$mapper" />
        </xsl:when>
        <xsl:when test="$attribute and not($attribute='')">
          <xsl:value-of select="$attribute" />
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="$default" />
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <func:result>
      <xsl:value-of select="$result" />
    </func:result>
  </func:function>



  <!-- merge many nodes -->
  <func:function name="filter:merge">
    <xsl:param name="nodes" />
    <xsl:variable name="_merged_RTF">
      <xsl:for-each select="//*[name()=$my_root and name(*)=$my_row]/*[name()=$my_row][1]">
        <xsl:copy>
          <xsl:copy-of select="filter:merge-fields($nodes[1]/*,$nodes[2]/*)" />
        </xsl:copy>
      </xsl:for-each>
    </xsl:variable>
    <xsl:variable name="merged" select="exsl:node-set($_merged_RTF)/*" />
    <func:result>
      <xsl:choose>
        <xsl:when test="count($nodes)>2" >
          <xsl:copy-of select="filter:merge($merged|$nodes[position()>2])" />
        </xsl:when>
        <xsl:otherwise>
          <xsl:copy-of select="$merged/*" />
        </xsl:otherwise>
      </xsl:choose>
    </func:result>
  </func:function>


  <!-- variable replace is used by external merge routine -->
  <xsl:variable name="replace" select="'true'" />
  <!-- wrapper to external template -->
  <func:function name="filter:merge-fields">
    <xsl:param name="nodes1" />
    <xsl:param name="nodes2" />
    <func:result>
      <xsl:call-template name="m:merge">
        <xsl:with-param name="nodes1" select="$nodes1" />
        <xsl:with-param name="nodes2" select="$nodes2" />
      </xsl:call-template>
    </func:result>
  </func:function>


  <!-- build a string representation of a tag -->
  <func:function name="filter:tag-description">
    <xsl:param name="name" />
    <xsl:param name="namespace" />
    <func:result>
      <xsl:value-of select="concat('&lt;',$name)" />
      <xsl:if test="contains($name,':')">
        <xsl:value-of select="concat(' xmlns:',substring-before($name,':'),'=&quot;',$namespace,'&quot;')" />
      </xsl:if>
      <xsl:value-of select="'&gt;'" />
    </func:result>
  </func:function>


  <!-- replace tag name with namespace with a namespace-independent representation -->
  <func:function name="filter:expand-namespace">
    <xsl:param name="string" />
    <xsl:param name="prefix" />
    <xsl:param name="namespace" />
    <func:result>
      <xsl:choose>
        <xsl:when test="$prefix=':' or not(contains($string,$prefix))">
          <xsl:value-of select="$string" />
        </xsl:when>
        <xsl:otherwise>
          <xsl:variable name="before" select="substring-before($string,$prefix)" />
          <xsl:variable name="after" select="substring-after($string,$prefix)" />
          <xsl:variable name="local-name" select="filter:get-qname($after)" />
          <xsl:variable name="name" select="concat($prefix,$local-name)" />
          <xsl:variable name="others" select="substring-after($after,$local-name)" />
          <xsl:variable name="expanded" select='concat(
              "*[",
              "local-name()=&apos;",$local-name,"&apos;",
              " and ",
              "namespace-uri()=&apos;",$namespace,"&apos;",
              "]"
            )'/>
          <xsl:value-of select='concat($before,$expanded,filter:expand-namespace($others,$prefix,$namespace))' />
        </xsl:otherwise>
      </xsl:choose>
    </func:result>
  </func:function>


  <!-- generic replace function -->
  <func:function name="filter:replace">
    <xsl:param name="text" />
    <xsl:param name="from" />
    <xsl:param name="to" />
    <func:result>
      <xsl:choose>
        <xsl:when test="contains($text, $from)">
          <xsl:variable name="before" select="substring-before($text, $from)" />
          <xsl:variable name="after" select="substring-after($text, $from)" />
          <xsl:variable name="prefix" select="concat($before, $to)" />
          <xsl:value-of select="$before" />
          <xsl:value-of select="$to" />
          <xsl:copy-of select="filter:replace($after,$from,$to)" />
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="$text" />
        </xsl:otherwise>
      </xsl:choose>
    </func:result>
  </func:function>


  <!-- sort by @priority -->
  <func:function name="filter:sort-by-priority">
    <xsl:param name="nodes" />
    <func:result>
      <xsl:for-each select="$nodes[@priority &lt; 0]">
        <xsl:sort select="@priority" data-type="number" order="ascending" />
        <xsl:copy-of select="." />
      </xsl:for-each>
      <xsl:copy-of select="$nodes[
          not(@priority) or
          @priority='' or
          @priority='0' or
          string(number(@priority)) = 'NaN'
        ]" />
      <xsl:for-each select="$nodes[@priority &gt; 0]">
        <xsl:sort select="@priority" data-type="number" order="ascending" />
        <xsl:copy-of select="." />
      </xsl:for-each>
    </func:result>
  </func:function>


  <!-- extract a valid QName from the input string -->
  <func:function name="filter:get-qname">
    <xsl:param name="string" />
    <xsl:param name="isFirstChar" select="true()" />
    <xsl:variable name="first" select="substring($string,1,1)" />
    <xsl:variable name="others" select="substring($string,2)" />
    <func:result>
      <xsl:choose>
        <xsl:when test="$isFirstChar">
          <xsl:if test="filter:is-valid-qname-NameStartChar($first)">
            <xsl:value-of select="$first" />
            <xsl:value-of select="filter:get-qname($others,false())" />
          </xsl:if>
        </xsl:when>
        <xsl:when test="filter:is-valid-qname-NameChar($first)">
          <xsl:value-of select="$first" />
          <xsl:value-of select="filter:get-qname($others,false())" />
        </xsl:when>
        <xsl:otherwise>
        </xsl:otherwise>
      </xsl:choose>
    </func:result>
  </func:function>


  <!-- check if the first character is a valid NameStartChar -->
  <!-- [see http://www.w3.org/TR/xml/#NT-NameStartChar] -->
  <func:function name="filter:is-valid-qname-NameStartChar">
    <xsl:param name="character" />
    <xsl:variable name="regexp" select="concat(
      '^[',
      ':',
      'A-Z',
      '_',
      'a-z',
      '&#xC0;-&#xD6;',
      '&#xD8;-&#xF6;',
      '&#xF8;-&#x2FF;',
      '&#x370;-&#x37D;',
      '&#x37F;-&#x1FFF;',
      '&#x200C;-&#x200D;',
      '&#x2070;-&#x218F;',
      '&#x2C00;-&#x2FEF;',
      '&#x3001;-&#xD7FF;',
      '&#xF900;-&#xFDCF;',
      '&#xFDF0;-&#xFFFD;',
      '&#x10000;-&#xEFFFF;',
      ']$'
      )" />
    <func:result select="regexp:matchRegexp($regexp,$character, 0)" />
  </func:function>


  <!-- check if the first character is a valid NameChar -->
  <!-- [see http://www.w3.org/TR/xml/#NT-NameChar] -->
  <func:function name="filter:is-valid-qname-NameChar">
    <xsl:param name="character" />
    <xsl:variable name="regexp" select="concat(
      '^[',
      '-',
      '.',
      '0-9',
      '&#xB7;',
      '&#x0300;-&#x036F;',
      '&#x203F;-&#x2040;',
      ']$'
      )" />
    <func:result select="
        filter:is-valid-qname-NameStartChar($character) or
        regexp:matchRegexp($regexp,$character, 0)
      " />
  </func:function>


<!-- ====================================================================== -->
<!-- Merge Library by Oliver Becker                                         -->
<!-- http://www2.informatik.hu-berlin.de/~obecker/XSLT/#merge               -->
<!-- ====================================================================== -->

<!-- Normalize the contents of text, comment, and processing-instruction
     nodes before comparing?
     Default: yes -->
<xsl:param name="normalize" select="'yes'" />

<!-- Don't merge elements with this (qualified) name -->
<xsl:param name="dontmerge" />

<!-- If set to true, text nodes in file1 will be replaced -->
<xsl:param name="replace" select="false()" />

<!-- Variant 1: Source document looks like
     <?xml version="1.0"?>
     <merge xmlns="http://informatik.hu-berlin.de/merge">
        <file1>file1.xml</file1>
        <file2>file2.xml</file2>
     </merge>         
     The transformation sheet merges file1.xml and file2.xml.
-->
<xsl:template match="m:merge" >
   <xsl:variable name="file1" select="string(m:file1)" />
   <xsl:variable name="file2" select="string(m:file2)" />
   <xsl:message>
      <xsl:text />Merging '<xsl:value-of select="$file1" />
      <xsl:text />' and '<xsl:value-of select="$file2"/>'<xsl:text />
   </xsl:message>
   <xsl:if test="$file1='' or $file2=''">
      <xsl:message terminate="yes">
         <xsl:text>No files to merge specified</xsl:text>
      </xsl:message>
   </xsl:if>
   <xsl:call-template name="m:merge">
      <xsl:with-param name="nodes1" select="document($file1,/*)/node()" />
      <xsl:with-param name="nodes2" select="document($file2,/*)/node()" />
   </xsl:call-template>
</xsl:template>


<!-- Variant 2:
     The transformation sheet merges the source document with the
     document provided by the parameter "with".
-->
<xsl:param name="with" />

<xsl:template match="*">
   <xsl:message>
      <xsl:text />Merging input with '<xsl:value-of select="$with"/>
      <xsl:text>'</xsl:text>
   </xsl:message>
   <xsl:if test="string($with)=''">
      <xsl:message terminate="yes">
         <xsl:text>No input file specified (parameter 'with')</xsl:text>
      </xsl:message>
   </xsl:if>

   <xsl:call-template name="m:merge">
      <xsl:with-param name="nodes1" select="/node()" />
      <xsl:with-param name="nodes2" select="document($with,/*)/node()" />
   </xsl:call-template>
</xsl:template>


<!-- ============================================================== -->

<!-- The "merge" template -->
<xsl:template name="m:merge">
   <xsl:param name="nodes1" />
   <xsl:param name="nodes2" />

   <xsl:choose>
      <!-- Is $nodes1 resp. $nodes2 empty? -->
      <xsl:when test="count($nodes1)=0">
         <xsl:copy-of select="$nodes2" />
      </xsl:when>
      <xsl:when test="count($nodes2)=0">
         <xsl:copy-of select="$nodes1" />
      </xsl:when>

      <xsl:otherwise>
         <!-- Split $nodes1 and $nodes2 -->
         <xsl:variable name="first1" select="$nodes1[1]" />
         <xsl:variable name="rest1" select="$nodes1[position()!=1]" />
         <xsl:variable name="first2" select="$nodes2[1]" />
         <xsl:variable name="rest2" select="$nodes2[position()!=1]" />
         <!-- Determine type of node $first1 -->
         <xsl:variable name="type1">
            <xsl:apply-templates mode="m:detect-type" select="$first1" />
         </xsl:variable>

         <!-- Compare $first1 and $first2 -->
         <xsl:variable name="diff-first">
            <xsl:call-template name="m:compare-nodes">
               <xsl:with-param name="node1" select="$first1" />
               <xsl:with-param name="node2" select="$first2" />
            </xsl:call-template>
         </xsl:variable>

         <xsl:choose>
            <!-- $first1 != $first2 -->
            <xsl:when test="$diff-first='!'">
               <!-- Compare $first1 and $rest2 -->
               <xsl:variable name="diff-rest">
                  <xsl:for-each select="$rest2">
                     <xsl:call-template name="m:compare-nodes">
                        <xsl:with-param name="node1" select="$first1" />
                        <xsl:with-param name="node2" select="." />
                     </xsl:call-template>
                  </xsl:for-each>
               </xsl:variable>
      
               <xsl:choose>
                  <!-- $first1 is in $rest2 and 
                       $first1 is *not* an empty text node  -->
                  <xsl:when test="contains($diff-rest,'=') and
                                      not($type1='text' and
                                          normalize-space($first1)='')">
                     <!-- determine position of $first1 in $nodes2
                          and copy all preceding nodes of $nodes2 -->
                     <xsl:variable name="pos" 
                           select="string-length(substring-before(
                                                $diff-rest,'=')) + 2" />
                     <xsl:copy-of 
                           select="$nodes2[position() &lt; $pos]" />
                     <!-- merge $first1 with its equivalent node -->
                     <xsl:choose>
                        <!-- Elements: merge -->
                        <xsl:when test="$type1='element'">
                           <xsl:element name="{name($first1)}" 
                                         namespace="{namespace-uri($first1)}">
                              <xsl:copy-of select="$first1/namespace::*" />
                              <xsl:copy-of select="$first2/namespace::*" />
                              <xsl:copy-of select="$first1/@*" />
                              <xsl:call-template name="m:merge">
                                 <xsl:with-param name="nodes1" 
                                       select="$first1/node()" />
                                 <xsl:with-param name="nodes2" 
                                       select="$nodes2[position()=$pos]/node()" />
                              </xsl:call-template>
                           </xsl:element>
                        </xsl:when>
                        <!-- Other: copy -->
                        <xsl:otherwise>
                           <xsl:copy-of select="$first1" />
                        </xsl:otherwise>
                     </xsl:choose>
      
                     <!-- Merge $rest1 and rest of $nodes2 -->
                     <xsl:call-template name="m:merge">
                        <xsl:with-param name="nodes1" select="$rest1" />
                        <xsl:with-param name="nodes2" 
                              select="$nodes2[position() &gt; $pos]" />
                     </xsl:call-template>
                  </xsl:when>

                  <!-- $first1 is a text node and replace mode was
                       activated -->
                  <xsl:when test="$type1='text' and $replace">
                     <xsl:call-template name="m:merge">
                        <xsl:with-param name="nodes1" select="$rest1" />
                        <xsl:with-param name="nodes2" select="$nodes2" />
                     </xsl:call-template>
                  </xsl:when>

                  <!-- else: $first1 is not in $rest2 or
                       $first1 is an empty text node -->
                  <xsl:otherwise>
                     <xsl:copy-of select="$first1" />
                     <xsl:call-template name="m:merge">
                        <xsl:with-param name="nodes1" select="$rest1" />
                        <xsl:with-param name="nodes2" select="$nodes2" />
                     </xsl:call-template>
                  </xsl:otherwise>
               </xsl:choose>
            </xsl:when>

            <!-- else: $first1 = $first2 -->
            <xsl:otherwise>
               <xsl:choose>
                  <!-- Elements: merge -->
                  <xsl:when test="$type1='element'">
                     <xsl:element name="{name($first1)}" 
                                   namespace="{namespace-uri($first1)}">
                        <xsl:copy-of select="$first1/namespace::*" />
                        <xsl:copy-of select="$first2/namespace::*" />
                        <xsl:copy-of select="$first1/@*" />
                        <xsl:call-template name="m:merge">
                           <xsl:with-param name="nodes1" 
                                            select="$first1/node()" />
                           <xsl:with-param name="nodes2" 
                                            select="$first2/node()" />
                        </xsl:call-template>
                     </xsl:element>
                  </xsl:when>
                  <!-- Other: copy -->
                  <xsl:otherwise>
                     <xsl:copy-of select="$first1" />
                  </xsl:otherwise>
               </xsl:choose>

               <!-- Merge $rest1 and $rest2 -->
               <xsl:call-template name="m:merge">
                  <xsl:with-param name="nodes1" select="$rest1" />
                  <xsl:with-param name="nodes2" select="$rest2" />
               </xsl:call-template>
            </xsl:otherwise>
         </xsl:choose>
      </xsl:otherwise>
   </xsl:choose>
</xsl:template>


<!-- Comparing single nodes: 
     if $node1 and $node2 are equivalent then the template creates a 
     text node "=" otherwise a text node "!" -->
<xsl:template name="m:compare-nodes">
   <xsl:param name="node1" />
   <xsl:param name="node2" />
   <xsl:variable name="type1">
      <xsl:apply-templates mode="m:detect-type" select="$node1" />
   </xsl:variable>
   <xsl:variable name="type2">
      <xsl:apply-templates mode="m:detect-type" select="$node2" />
   </xsl:variable>

   <xsl:choose>
      <!-- Are $node1 and $node2 element nodes with the same name? -->
      <xsl:when test="$type1='element' and $type2='element' and
                       local-name($node1)=local-name($node2) and
                       namespace-uri($node1)=namespace-uri($node2) and
                       name($node1)!=$dontmerge and name($node2)!=$dontmerge">
         <!-- Comparing the attributes -->
         <xsl:variable name="diff-att">
            <!-- same number ... -->
            <xsl:if test="count($node1/@*)!=count($node2/@*)">.</xsl:if>
            <!-- ... and same name/content -->
            <xsl:for-each select="$node1/@*">
               <xsl:if test="not($node2/@*
                        [local-name()=local-name(current()) and 
                         namespace-uri()=namespace-uri(current()) and 
                         .=current()])">.</xsl:if>
            </xsl:for-each>
         </xsl:variable>
         <xsl:choose>
            <xsl:when test="string-length($diff-att)!=0">!</xsl:when>
            <xsl:otherwise>=</xsl:otherwise>
         </xsl:choose>
      </xsl:when>

      <!-- Other nodes: test for the same type and content -->
      <xsl:when test="$type1!='element' and $type1=$type2 and
                       name($node1)=name($node2) and 
                       ($node1=$node2 or 
                          ($normalize='yes' and
                           normalize-space($node1)=
                           normalize-space($node2)))">=</xsl:when>

      <!-- Otherwise: different node types or different name/content -->
      <xsl:otherwise>!</xsl:otherwise>
   </xsl:choose>
</xsl:template>


<!-- Type detection, thanks to M. H. Kay -->
<xsl:template match="*" mode="m:detect-type">element</xsl:template>
<xsl:template match="text()" mode="m:detect-type">text</xsl:template>
<xsl:template match="comment()" mode="m:detect-type">comment</xsl:template>
<xsl:template match="processing-instruction()" mode="m:detect-type">pi</xsl:template>

<!-- ====================================================================== -->
<!-- End of Merge Library                                                   -->
<!-- ====================================================================== -->


</xsl:stylesheet>
