/*global window, jQuery */
////////////////////////////////////////////////////////////////////////////////
// Copyright 2023-2025 by NI SP Software GmbH, All rights reserved.
// Copyright 1999-2023 by Nice, srl., All rights reserved.
//
// This software includes confidential and proprietary information
// of NI SP Software GmbH ("Confidential Information").
// You shall not disclose such Confidential Information
// and shall use it only in accordance with the terms of
// the license agreement you entered into with NI SP Software.
////////////////////////////////////////////////////////////////////////////////

// Configuration map for Interactive Session Cleanup columns
const sessionStatusColConfig = {
    result: {
        style: 'style="width:150px;"',
        title: 'Result'
    },
    uri: {
        title: 'Interactive Session URI'
    },
    status: {
        style: 'style="width:250px;"',
        title: 'Interactive Session Status'
    },
    message: {
        title: 'Reason'
    },
};

var adminAdvanced = {

    init: function () {
        adminAdvanced.renderInteractiveCleanupOutput();
    },

    toStandardCase: function (str){
        return str
            .replace(/_/g, ' ') // replace underscores with spaces
            .toLowerCase() // make everything lowercase
            .replace(/^\w/, c => c.toUpperCase()); // capitalize the first letter
    },

    renderResult: function (result) {
        const s = (result || '').toLowerCase();
        let cls = '';
        switch (result) {
            case 'success':
                cls = 'result-ok';
                break;
            case 'skipped':
            case 'no_sessions_cleaned':
                cls = 'result-warn';
                break;
            case 'completed_with_errors':
            default:
                if (result) cls = 'result-err';
                break;
        }
        return `<span class="result ${cls}">${adminAdvanced.toStandardCase(result)}</span>`;
    },

    getColConfig: function(columnName) {
        const key = columnName.toLowerCase();
        if (sessionStatusColConfig[key]) {
            return sessionStatusColConfig[key];
        }

        // Default fallback if column not in map
        return {
            style: '',
            title: columnName.capitalize()
        };
    },

    renderTable: function (items, result) {
        if (!items || items.length === 0) {
            return '<p><em>No items</em></p>';
        }
        let cols = Object.keys(items[0]);
        // Use jqgrid classes to have a look & feel similar to hydrogen tables
        let html = '<div class="ui-jqgrid">' +
            '<table class="ui-jqgrid-htable">' +
                '<tr>' +
                    //'<th class="ui-state-default ui-th-column ui-th-ltr">Result</th>' +
                    cols.map(c => {
                        const conf = adminAdvanced.getColConfig(c);
                        return `<th class="ui-state-default ui-th-column ui-th-ltr" ${conf.style || ''}>${conf.title}</th>`;
                    }).join('') +
                '</tr>';
        items.forEach(item => {
            html += '<tr class="jqgrow ui-row-ltr">';
            //html += `<td>${adminAdvanced.renderResult(result)}</td>`;
            cols.forEach(c => {
                let val = item[c] || '';
                if (c === 'result') val = adminAdvanced.renderResult(val);
                html += `<td>${val}</td>`;
            });
            html += '</tr>';
        });
        html += '</table></div>';
        return html;
    },

    renderInteractiveCleanupOutput: function () {
        let raw = $('#interactive-cleanup-output').text().trim();
        let data;
        try {
            data = JSON.parse(raw);
        } catch (e) {
            alert('Invalid JSON: ' + e.message);
            return;
        }

        let out = '';
        // If JSON has "error" key, render error block
        if (data.error) {
            // Example JSON: {"error": "", "message": ""};
            out += `<h2>Error</h2>
                    <p><strong>Error:</strong> ${data.error}</p>
                    <p><strong>Message:</strong> ${data.message || ''}</p>`;
        } else {
            // Success, example JSON
            // {  "mode": "session-list",
            //    "force": false,
            //    "ttl": "1m",
            //    "succeeded": [{
            //           "uri": "spooler:///...tmp18412220580383169937.session.ef",
            //           "status": "Closed",
            //           "message": "Successfully set TTL"
            //       }],
            //    "skipped": [],
            //    "failed": [],
            //    "status": "success"};
            out += `<h2>Interactive Sessions Cleanup Result</h2>
              <p><strong>Force:</strong> ${data.force}</p>
              <p><strong>Configured Time To Live (TTL):</strong> ${data.ttl}</p>
              <p><strong>Result:</strong> ${adminAdvanced.renderResult(data.status)}</p>`;

            let succeeded = data.succeeded || [];
            let skipped = data.skipped || [];
            let failed = data.failed || [];

            out += `<h2>Summary</h2>
              <p>Number or Sessions: <strong>${succeeded.length + skipped.length + failed.length}</strong><br />
              Succeded: ${succeeded.length}<br />
              Skipped: ${skipped.length}<br />
              Failed: ${failed.length}</p><hr/>`;

            // Merge all results into a single array with a "result" field
            const allResults = []
                .concat(succeeded.map(r => ({ result: "success" , ...r })))
                .concat(skipped.map(r => ({ result: "skipped", ...r })))
                .concat(failed.map(r => ({ result: "failed", ...r })));

            // Render one unified table
            out += `<h2>Results</h2>${adminAdvanced.renderTable(allResults)}`;
        }

        $('#interactive-cleanup-output').html(out).show();
    }
};

jQuery(document).ready(function () {
    adminAdvanced.init();
});
