#!/bin/bash

#########################################################################
##################################################################################
# Copyright 2023-2025 by NI SP Software GmbH, All rights reserved.
# Copyright 1999-2023 by Nice, srl., All rights reserved.
#
# This software includes confidential and proprietary information
# of NI SP Software GmbH ("Confidential Information").
# You shall not disclose such Confidential Information
# and shall use it only in accordance with the terms of
# the license agreement you entered into with NI SP Software.
################################################################################
################################################################################                                       ###
#########################################################################

# This script gathers a set of informations about EnginFrame portal
# installation and configuration

# support@ni-sp-software.com

# Source admin profile common script
. `dirname $0`/common


# Variable initialization
# =======================================================================
# List of file to include in final archive
EF_COLLECT_LIST="/tmp/ef_collect_filelist_$$.txt"

# Logfile of this service
EF_COLLECT_LOG="/tmp/ef_collect_log_$$.txt"

# Final archive filename (without extension or path)
EF_COLLECT_RESULT="ef_collect_$$"

# Check if we have to print what we're doing on screen
[ "${BEVERBOSE}" = "true" ] && EF_COLLECT_DEBUG=1 || EF_COLLECT_DEBUG=0

# Get NISP_ROOT directory
NISP_ROOT=`dirname "${EF_ROOT}"`

# How many tail lines to include
LOG_LINES="2000"



# Printing and logging
# =======================================================================

# Prints Ok in green and closes table row
printok () {
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "<td><font color=\"green\">"
  echo "$1" >> "${EF_COLLECT_LOG}"
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "&#160;&#160; $1"
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "</font></td></tr>"
}

# Prints Error in red and closes table row
printerror () {
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "<td><font color=\"red\">"
  echo "$1" >> "${EF_COLLECT_LOG}"
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "&#160;&#160; $1"
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "</font></td></tr>"
}

# Prints an error in red on a stand-alone row
printsoloerror () {
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "<tr><td colspan=\"2\"><br/><font color=\"red\">"
  echo "[`date '+%Y/%m/%d %H:%M:%S'`] [ERR ] $1 " >> "${EF_COLLECT_LOG}"
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "$1"
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "</font></td></tr>"
}

# Prints a warning in orange and closes table row
printwarning () {
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "<td><font color=\"orange\">"
  echo "$1" >> "${EF_COLLECT_LOG}"
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "&#160;&#160; $1"
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "</font></td></tr>"
}

# Prints a warning in orange on a stand-alone row
printsolowarning () {
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "<tr><td colspan=\"2\"><br/><font color=\"orange\">"
  echo "[`date '+%Y/%m/%d %H:%M:%S'`] [WARN] $1 " >> "${EF_COLLECT_LOG}"
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "$1"
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "</font></td></tr>"
}

# Starts a section i.e. opens a new table
startsection () {
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "<b>"
  echo "[`date '+%Y/%m/%d %H:%M:%S'`] [INFO] $1 " >> "${EF_COLLECT_LOG}"
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "$1"
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "</b><hr/>"
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "<table border=\"0\" cellpadding=\"0\" cellspacing=\"0\">"
}

# Closes a section i.e. closes a table
closesection () {
  echo "[`date '+%Y/%m/%d %H:%M:%S'`] [INFO] $1 " >> "${EF_COLLECT_LOG}"
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "</table>"
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "<br/><br/>"
}

# Prints a fatal error messages and exits
printexit () {
  echo "[`date '+%Y/%m/%d %H:%M:%S'`] [EXIT] $1 " >> "${EF_COLLECT_LOG}"
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && closesection
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "<br/><br/>"
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "<p>--- Fatal errors found. Please contact your EF Portal Administrator ---</p>"
}

# Print an entry (half row)
printvoice () {
  echo "[`date '+%Y/%m/%d %H:%M:%S'`] [INFO] $1 " >> "${EF_COLLECT_LOG}"
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "<tr><td>&#160;&#160;&#160;"
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "$1"
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "</td>"
}

# Prints an entry in a stand-alone row
printsolovoice () {
  echo "[`date '+%Y/%m/%d %H:%M:%S'`] [INFO] $1 " >> "${EF_COLLECT_LOG}"
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "<tr><td colspan=\"2\">&#160;&#160;&#160;"
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "$1"
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "</td></tr>"
}

# Prints an entry indented (half row)
printsubvoice () {
  printf "[`date '+%Y/%m/%d %H:%M:%S'`] [INFO] $1 " >> "${EF_COLLECT_LOG}"
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "<tr><td>&#160;&#160;&#160;&#160;&#160;&#160;&#160;&#160;"
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "$1"
  [ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "</td>"
}


#---------------------------------------------------------------------------- #
# ef_tail
# ============
# Executes tail with the correct syntax for host OS
#
# Arguments: the same you'd pass to tail
#-------------------------------------------------------------------------------

ef_tail() {

  # New tail syntax: tail -n -100 file (EF_TAIL_NEWSYNTAX=1)
  # Classic tail syntax: tail -100 file (EF_TAIL_NEWSYNTAX=0)
  # The same for the input command line (EF_INPUT_NEWSYNTAX)

  # Find if OS tail syntax is new or classic
  [ "`echo hello | tail -1 2>/dev/null`" = "hello" ] \
    && EF_TAIL_NEWSYNTAX=0 \
    || EF_TAIL_NEWSYNTAX=1

  # Copy options string
  EF_OPTIONS_STRING="$@"

  # Check input options
  EF_INPUT_NEWSYNTAX=0
  for i in $@; do
    # Check if there's a -<number> option
    MINUS_NUMBER=`echo $1 | egrep -e "-[0-9]"`
    # Check if there's a -n <number> option
    if [ "-n" = "$1" ]; then
      EF_TAIL_LINES=`echo $2 | sed 's/+//;s/-//'`
      EF_INPUT_NEWSYNTAX=1
      shift
    elif [ -n "${MINUS_NUMBER}" ]; then
      # Remove minus
      EF_TAIL_LINES=`echo ${MINUS_NUMBER} | sed 's/-//'`
    else
      # Copy other non-influent options
      EF_TAIL_OPTIONS="$EF_TAIL_OPTIONS $1"
    fi
    [ "$#" = "1" ] && break || shift
  done

  # execute tail
  [ $EF_TAIL_NEWSYNTAX -eq $EF_INPUT_NEWSYNTAX ] && tail ${EF_OPTIONS_STRING}
  [ $EF_TAIL_NEWSYNTAX -eq 0 -a $EF_INPUT_NEWSYNTAX -eq 1 ] && tail -${EF_TAIL_LINES} ${EF_TAIL_OPTIONS}
  [ $EF_TAIL_NEWSYNTAX -eq 1 -a $EF_INPUT_NEWSYNTAX -eq 0 ] && tail -n +${EF_TAIL_LINES} ${EF_TAIL_OPTIONS}
}


collect() {
  local _search="$1"
  local _list="$2"
  local _depth=""
  if [ -n "$3" ]; then
    _depth="-maxdepth $3"
  fi

  if [ -x "${_search}" ]; then
    for i in `find "${_search}" ${_depth} -type f`; do
      printsubvoice "$i:"
      if [ -r "$i" ]; then
        printok "Collected"
        echo "$i" >> "${_list}"
      else
        printwarning "protected"
      fi
    done
  else
    printsubvoice "${_search}:"
    printerror "Not accessible: ${_search}"
  fi
}

collect_logs() {
  _search="$1"
  _list="$2"
  #printsubvoice "${_search}:"
  if [ -x "${_search}" ]; then
    #printok "Ok"
    for i in `ls ${_search}`; do
      if [ ! -d "${_search}/$i" ]; then
        printsubvoice "${_search}/$i:"
        if [ -r "${_search}/$i" ]; then
          printok "Collected"
          echo " " >> "${_list}"
          echo "${_search}/$i" >> "${_list}"
          echo "===========================================================" >> "${_list}"
          ef_tail -${LOG_LINES} "${_search}/$i" >> "${_list}"
        else
          printwarning "protected"
        fi
      fi
    done
  else
    printsubvoice "${_search}:"
    printerror "Not accessible: ${_search}"
  fi
}


# Start
# =======================================================================
[ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "<p>Starting at `date`</p><br/>"
echo "<p>Starting at `date`</p><br/>" >> "${EF_COLLECT_LOG}"



startsection "Preliminary checks"
# -----------------------------------------------------------------------
# Preliminary write check on /tmp
# Temporary files would be written here
if [ -w "/tmp" ]; then
  printvoice "Preliminary check on writable /tmp: "; printok "ok"
  cd "/tmp"
else
  printerror "/tmp directoy in host `hostname` is not writable. Please contact your system administrator."
  printexit
fi

# Prints log and temporary file path and names
printvoice "Included file list: "; printok "${EF_COLLECT_LIST}"
printvoice "Collector log: "; printok "${EF_COLLECT_LOG}"

# Add log and temporary file names to final archive file list
echo "${EF_COLLECT_LOG}" >> "${EF_COLLECT_LIST}"
echo "${EF_COLLECT_LIST}" >> "${EF_COLLECT_LIST}"

closesection



startsection "Collecting environment info:"
# -----------------------------------------------------------------------
# Internally runs EF self check service in text mode (run_tests_text.sh)
printvoice "Retrieving system info:"
cat << EOF >> "${EF_COLLECT_LOG}"
Executing check script
===========================================================
`${EF_ROOT}/plugins/admin/bin/run_tests_text.sh 2>&1`
===========================================================
Processes
-----------------------------------------------------------------------
`ps -def 2>&1`
===========================================================
ls -alR
-----------------------------------------------------------------------
`ls -alR "${NISP_ROOT}" 2>&1`
===========================================================
===========================================================
===========================================================
EOF
printok "Ok"
closesection

# Load utility functions
export EF_VERSION_ROOT="${EF_ROOT}/.."
. "${EF_VERSION_ROOT}/bin/functions.sh" > /dev/null
load_env

# -----------------------------------------------------------------------
startsection "Collecting EF Portal configuration files:"
collect "${EF_CONF_ROOT}" "${EF_COLLECT_LIST}" 1
collect "${EF_CONF_ROOT}/enginframe" "${EF_COLLECT_LIST}"
collect "${EF_ROOT}/conf" "${EF_COLLECT_LIST}"
closesection

# -----------------------------------------------------------------------
startsection "Collecting EF Portal plugin configuration files:"
collect "${EF_CONF_ROOT}/plugins" "${EF_COLLECT_LIST}"
closesection

# -----------------------------------------------------------------------
startsection "Collecting Tomcat configuration files:"
collect "${CATALINA_BASE}/conf" "${EF_COLLECT_LIST}"
closesection

# -----------------------------------------------------------------------
startsection "Collecting Derby configuration files:"
collect "${DERBY_SYSTEM_HOME}" "${EF_COLLECT_LIST}"
closesection

# -----------------------------------------------------------------------
startsection "Collecting EF Portal log files:"
collect "${EF_LOGS_ROOT}" "${EF_COLLECT_LIST}"
closesection

echo "<p>Ending at `date`</p><br/>" >> "${EF_COLLECT_LOG}"

# -----------------------------------------------------------------------
startsection "Collecting License info:"
collect "${EF_LICENSE_PATH}" "${EF_COLLECT_LIST}"
closesection

# Check if we must include plugins
[ "${INCLUDEPLUGINS}" = "true" ] && echo "${EF_ROOT}/plugins" >> "${EF_COLLECT_LIST}"


startsection "Building archive:"
# -----------------------------------------------------------------------
printvoice "Building tar"

which tar > /dev/null 2>&1
if [ "$?" -eq "0" ]; then
  printok "Ok"
  TAR_RESULT=`cat "${EF_COLLECT_LIST}" | xargs tar cvf "/tmp/${EF_COLLECT_RESULT}.tar" 2>&1`
  if [ "${EF_COLLECT_DEBUG}" -eq 1 ]; then
    cat << EOF
      <tr><td colspan="2"><pre>`echo "${TAR_RESULT}" | grep -v "Removing leading"`</pre></td></tr>
EOF
fi
else
  printerror "tar executable not found in your path. Please contact your system administrator"
  printexit
fi

printvoice "Compressing"
# -------------------------------------------------------------------------

gzip --help > /dev/null 2>&1
if [ "$?" -eq "0" ]; then
  gzip -c /tmp/${EF_COLLECT_RESULT}.tar > /tmp/${EF_COLLECT_RESULT}.tgz
else
  which compress > /dev/null 2>&1
  if [ "$?" -eq "0" ]; then
    compress -c /tmp/${EF_COLLECT_RESULT}.tar > /tmp/${EF_COLLECT_RESULT}.tar.Z
  else
    printerror "no gzip nor compress executable found in your path. Please contact your system administrator"
    printexit
  fi
fi

printok "Ok"


closesection
[ "${EF_COLLECT_DEBUG}" -eq 1 ] && echo "<p>Ending at `date`</p><br/>"


# Try to copy result archive into spooler to download it
cp -f /tmp/${EF_COLLECT_RESULT}* $EF_SPOOLER/
ARCHIVE_PRESENT=`ls . | grep ${EF_COLLECT_RESULT} | sort -r | sed 1q`
if [ -r "${ARCHIVE_PRESENT}" ]; then
    cat << EOF
  <a href="/${EF_ROOT_CONTEXT}/download/${ARCHIVE_PRESENT}?_spooler=${EF_SPOOLER_URI}&amp;_file=${ARCHIVE_PRESENT}&amp;_plugin=ef"><b>Download compressed report</b></a>
EOF

# Clear temporary file on successfull operation
rm -f "/tmp/${EF_COLLECT_RESULT}.tar" 2>/dev/null
rm -f "/tmp/${EF_COLLECT_RESULT}.tgz" 2>/dev/null
rm -f "/tmp/${EF_COLLECT_RESULT}.tar.Z" 2>/dev/null
rm -f "${EF_COLLECT_LIST}"
rm -f "${EF_COLLECT_LOG}"
rm -f "${EF_COLLECT_EFLOGS}"
rm -f "${EF_COLLECT_TOMCATLOGS}"

else
  echo "<font color=\"red\">Error copying file from /tmp to current spooler. Please retrieve it manually: /tmp/${EF_COLLECT_RESULT}</font>"
fi
