#!/bin/bash

################################################################################
# Copyright 2023-2025 by NI SP Software GmbH, All rights reserved.
# Copyright 1999-2023 by Nice, srl., All rights reserved.
#
# This software includes confidential and proprietary information
# of NI SP Software GmbH ("Confidential Information").
# You shall not disclose such Confidential Information
# and shall use it only in accordance with the terms of
# the license agreement you entered into with NI SP Software.
################################################################################

# Maximum time to wait for du command
# Default is 10 seconds
DU_TIMEOUT=10s

# Maximum time to wait for df command
# Default is 15 seconds
DF_TIMEOUT=15s

# Limit listing to file systems TYPEs when using df
# This below is an example that excludes network mounts
# DF_EXCLUDE=("--exclude-type=nfs" "--exclude-type=cifs")
# Default is an empty array which means include all TYPE of mounts
DF_EXCLUDE=()


# Input: 
#   $1 : the target filesystem path to get information from

# Expected output:
#   <property name>=<property value>
#   ...
#
# Where expected properties are:
#  FS_PATH_USED_SIZE  : size in bytes of the portion of the filesystem identified by the target path.
#  FS_MOUNT_POINT     : mount point of the filesystem the target path belongs to
#  FS_USED_SIZE       : total number of bytes used in the filesystem
#  FS_AVAIL_SIZE      : total number of bytes available in the filesystem
#  FS_USED_INODES     : total number of inodes used in the filesystem
#  FS_AVAIL_INODES    : total number of free inodes in the filesystem

#---------------------------------------------------------------#
# Print the FS_PATH_USED_SIZE property
# Arguments:
#   $1 : the target filesystem path
#---------------------------------------------------------------#
print_path_info() {
  [ -z "$1" ] && return 1
  # E.g. _result="1851392 /opt/nisp/enginframe/spoolers/"
  #
  _result=$(timeout --preserve-status "$DU_TIMEOUT" du -s -B1 "$1" | tr '\t' ' ' | sed "s/ [ ]*/ /g")
  FS_PATH_USED_SIZE=$(echo "${_result}" | cut -d' ' -f1)

  if [ -n "${FS_PATH_USED_SIZE}" ]; then
    echo "FS_PATH_USED_SIZE=${FS_PATH_USED_SIZE}"
  fi
}

#---------------------------------------------------------------------------#
# Print the following properties: FS_MOUNT_POINT FS_USED_SIZE FS_AVAIL_SIZE
# Arguments:
#   $1 : the target filesystem path
#---------------------------------------------------------------------------#
print_fs_info() {
  [ -z "$1" ] && return 1
  # E.g. _result="/dev/mapper/VolGroup00-LogVol00 9457483776 7563792384 1405554688 85% /"
  _result=$(timeout --preserve-status "$DF_TIMEOUT" df "${DF_EXCLUDE[@]}" -B1 "$1" | tail -n +2 | tr -d '\n' | sed "s/ [ ]*/ /g")
  FS_USED_SIZE=$(echo "${_result}" | cut -d' ' -f3)
  FS_AVAIL_SIZE=$(echo "${_result}" | cut -d' ' -f4)
  FS_MOUNT_POINT=$(echo "${_result}" | cut -d' ' -f6)

  if [ -n "${FS_USED_SIZE}" ]; then
    echo "FS_USED_SIZE=${FS_USED_SIZE}"
  fi
  if [ -n "${FS_AVAIL_SIZE}" ]; then
    echo "FS_AVAIL_SIZE=${FS_AVAIL_SIZE}"
  fi
  if [ -n "${FS_MOUNT_POINT}" ]; then
    echo "FS_MOUNT_POINT=${FS_MOUNT_POINT}"
  fi
}

#---------------------------------------------------------------------------#
# Print the following properties: FS_USED_INODES FS_AVAIL_INODES
# Arguments:
#   $1 : the target filesystem path
#---------------------------------------------------------------------------#
print_fs_inodes() {
  [ -z "$1" ] && return 1
  # E.g. _result="/dev/mapper/VolGroup00-LogVol00 2392064 165001 2227063 7% /"
  _result=$(timeout --preserve-status "$DF_TIMEOUT" df "${DF_EXCLUDE[@]}" -B1 -i "$1" | tail -n +2 | tr -d '\n' | sed "s/ [ ]*/ /g")
  FS_USED_INODES=$(echo "${_result}" | cut -d' ' -f3)
  FS_AVAIL_INODES=$(echo "${_result}" | cut -d' ' -f4)

  if [ -n "${FS_USED_INODES}" ]; then
    echo "FS_USED_INODES=${FS_USED_INODES}"
  fi
  if [ -n "${FS_AVAIL_INODES}" ]; then
    echo "FS_AVAIL_INODES=${FS_AVAIL_INODES}"
  fi
}

#
# Main
#

if [ -z "$1" ]; then 
  echo "Target filesystem path argument is missing." >&2
  exit 1
fi

exec 2>/dev/null
print_path_info "$1"
print_fs_info "$1"
print_fs_inodes "$1"

# vi: syntax=sh ts=2 sw=2 et ai :
