/*
 * Copyright 2023-2025 by NI SP Software GmbH, All rights reserved.
 *
 * This software includes confidential and proprietary information
 * of NI SP Software GmbH ("Confidential Information").
 * You shall not disclose such Confidential Information
 * and shall use it only in accordance with the terms of
 * the license agreement you entered into with NI SP Software.
 */

/*** UTILS ***/

const Utils = {
    CHART_COLORS: {
        blue: 'rgb(54, 162, 235)',
        green: 'rgb(75, 192, 192)',
        yellow: 'rgb(255, 205, 86)',
        red: 'rgb(255, 99, 132)',
        orange: 'rgb(255, 159, 64)',
        purple: 'rgb(153, 102, 255)',
        grey: 'rgb(201, 203, 207)',
    },

    CHART_COLORS_TRANSPARENT: {
        blue: 'rgba(54, 162, 235, 0.2)',
        green: 'rgba(75, 192, 192, 0.2)',
        yellow: 'rgba(255, 205, 86, 0.2)',
        red: 'rgba(255, 99, 132, 0.2)',
        orange: 'rgba(255, 159, 64, 0.2)',
        purple: 'rgba(153, 102, 255, 0.2)',
        grey: 'rgba(201, 203, 207, 0.2)',
    },

    UI_FADE_DURATION_MS: 200,
    WIDGET_AUTOREFRESH_MS: 30000,
    ERROR_MAX_LENGTH: 300,

    // Load a JS file from the given path.
    // The path is relative to the root context of EF Portal.
    loadJs: function(file) {
        let js = document.createElement('script');
        let f = file.startsWith('/') ? file : '/' + file;

        js.type = 'text/javascript';
        js.src = '/' + jQuery.enginframe.rootContext + f + '?' + Date.now();
        jQuery('head').append(js);
    },

    // Load a CSS file from the given path.
    // The path is relative to the root context of EF Portal.
    loadCss: function(file) {
        let css = document.createElement('link');
        let f = file.startsWith('/') ? file : '/' + file;

        css.rel = 'stylesheet';
        css.href = '/' + jQuery.enginframe.rootContext + f + '?' + Date.now();
        jQuery('head').append(css);
    },

    invokeRest: function(endpoint, onSuccess, onError, method = 'GET', data = null) {
        const tokenEp = '/' + jQuery.enginframe.rootContext + '/eftoken/eftoken.xml?_uri=//eftoken/eftoken'
        const ep = endpoint.startsWith('/') ? endpoint : '/' + endpoint;
        const tokenStorage = [jQuery.enginframe.user.name, 'dashboard'].join('.');
        let token = sessionStorage.getItem(tokenStorage);

        // Try to call given endpoint using stored token
        jQuery.ajax({
            type: method,
            contentType : 'application/json',
            url: '/' + jQuery.enginframe.rootContext + '/rest' + ep,
            headers: {
                'accept': 'application/json',
                'Authorization': 'Bearer ' + token
            },
            data: data
        })
        .done((response) => {
            // All good, call the onSuccess callback
            onSuccess(response);
        })
        .error((error) => {
            var storedToken;

            // Check why the call failed
            if (error.status === 401) {
                // Token not valid or expired
                // Easy concurrency check: get the token again from the storage and check if it changed.
                // If changed, it's likely that another call already renewed it
                storedToken = sessionStorage.getItem(tokenStorage);
                if (!storedToken || storedToken === token) {
                    // Renew the token
                    jQuery.ajax({
                        type: 'POST',
                        url: tokenEp,
                        data: { 'EF_OUTPUT_MODE': 'rest' }
                    })
                    .done((response) => {
                        // Got a valid response, extract the new token and store it
                        let newToken = null;

                        response.split('\n').forEach(element => {
                            if(element.match(/^token=\w+$/)) {
                                newToken = element.split('=')[1];
                            }
                        });

                        // Store the token
                        sessionStorage.setItem(tokenStorage, newToken);
                        // Uncomment the code below to enable the following logic for additional security.
                        // The downside is that the server will generate new tokens much more frequently.
                        //
                        // If the token was not stored once, add a listener to remove it
                        // when the page is hidden (e.g. when the user visits another service or even when
                        // the user switches to another tab in some browsers).
                        // Event is automatically removed when invoked.
                        /*
                        if (!storedToken) {
                            addEventListener('pagehide', () => {
                                sessionStorage.removeItem(tokenStorage);
                            }, { once: true });
                        }
                        */

                        // Call again the endpoint using the new token
                        jQuery.ajax({
                            type: method,
                            contentType : 'application/json',
                            url: '/' + jQuery.enginframe.rootContext + '/rest' + ep,
                            headers: {
                                'accept': 'application/json',
                                'Authorization': 'Bearer ' + newToken
                            },
                            data: data
                        })
                        .done((response) => { onSuccess(response) })
                        .error((error) => { onError(error) });
                    })
                    .error((error) => {
                        // Get/Renew token failed, call the onError callback
                        onError(error);
                    });
                } else {
                    // Token already renewed, call the endpoint using the newly stored token
                    jQuery.ajax({
                        type: method,
                        contentType : 'application/json',
                        url: '/' + jQuery.enginframe.rootContext + '/rest' + ep,
                        headers: {
                            'accept': 'application/json',
                            'Authorization': 'Bearer ' + storedToken
                        },
                        data: data
                    })
                    .done((response) => { onSuccess(response) })
                    .error((error) => { onError(error) });
                }
            } else {
                // Other kind of error, call the onError callback immediately
                onError(error);
            }
        });
    },

    // Convert given value having given unit to MBytes
    toMBytes: function(value, unit) {
        const dividers = new Map([
            [ 'bytes', (1024 * 1024) ],
            [ 'kbytes', 1024 ],
            [ 'mbytes', 1 ],
            [ 'gbytes', (1 / 1024) ],
        ]);
        let divider;

        divider = dividers.get(unit);
        if (divider === undefined) {
            return value;
        }

        return value / divider;
    },

    // Convert any value to a trimmed string, which is empty if given value is null or undefined
    toString: function(value) {
        var s;

        if (value === null || value === undefined) {
            return '';
        }

        s = String(value).trim();

        return s.length ? s : '';
    },

    ellipsize: function(str, max = Utils.ERROR_MAX_LENGTH) {
      const s = Utils.toString(str);

      return s.length > max ? s.slice(0, Math.max(0, max - 1)) + '…<br/>Please check EF Portal log files.' : s;
    },

    // Format a JSON string coming from a failed REST API call into an error string suited for the UI
    formatError: function(jsonString, fallback = 'Please check EF Portal log files.') {
        let error, title, message;

        try {
            error = JSON.parse(jsonString);
        } catch {
            return fallback;
        }

        if (!error || typeof error !== 'object') {
            return fallback;
        }

        title = Utils.toString(error.title);
        message = Utils.ellipsize(error.message);

        if (!title && !message) {
            return fallback;
        }

        if (title && message) {
            return `${title}: ${message}`;
        }

        return title || message; // whichever is present
    },

    // Change the grid columns spanning to make the Storage widget a bit wider,
    // shrinking a bit the Licenses widget
    expandStorageWidget: function() {
        jQuery('#storage-widget').css('grid-column', '7 / span 5' );
        jQuery('#licenses-widget').css('grid-column', '12 / span 5' );
    },

    // Create an array of colors with defined values for each storage type
    // to be used when creating a storage pie chart
    getPieChartColors: function(storages) {
        return storages.map(storage => {
            switch (storage.storage) {
                case 'Spoolers':
                    return Utils.CHART_COLORS.yellow;
                case 'Repository':
                    return Utils.CHART_COLORS.purple;
                case 'Session Shared Folders':
                    return Utils.CHART_COLORS.orange;
                case 'Other':
                    return Utils.CHART_COLORS.blue;
                case 'Free':
                    return Utils.CHART_COLORS.green;
                default:
                    return Utils.CHART_COLORS.blue;
            }
        });
    }
};


/*** WIDGETS ***/

class LoadWidget {
    _content = null;
    _loading = null;
    _error = null;
    _config = {
        resource: 'unknown',
        label: 'unknown',
        yLabel: 'unknown',
        max: null,
        color: Utils.CHART_COLORS.blue,
        fillColor: Utils.CHART_COLORS_TRANSPARENT.blue

    };

    constructor(id, config) {
        this._loading = jQuery('#' + id + '-loading').html('Loading...');
        this._error = jQuery('#' + id + '-error');
        this._error.children('.close').on('click', function () {
            jQuery(this).parent().fadeOut(Utils.UI_FADE_DURATION_MS);
        });

        if (config) {
            if (config.resource) this._config.resource = config.resource;
            if (config.max) this._config.max = config.max;
            if (config.color) this._config.color = config.color;
            if (config.fillColor) this._config.fillColor = config.fillColor;
            if (config.yLabel) this._config.yLabel = config.yLabel;
            this._config.label = config.label || this._config.resource;
        }

        Utils.invokeRest(
            '/system/services',
            // onSuccess
            (response) => {
                const output = JSON.parse(response.output);
                // FIXME: add support for multiple servers (EF Ent)
                const host = output.servers[0].host;
                const server = output.servers[0].server;

                this._loading.hide();
                this._content = new Chart(jQuery('#' + id), {
                    type: 'line',
                    data: {
                        labels: server.map((d) => new Date(d.timestamp * 1000)),
                        datasets: [
                            {
                                label: `${this._config.label} on ${host}`,
                                data: server.map(d => {
                                    const item = d.data.find(i => i.resource === this._config.resource);
                                    return (item) ? Utils.toMBytes(item.value, item.unit) : null;
                                }),
                                spanGaps: false,
                                borderColor: this._config.color,
                                backgroundColor: this._config.fillColor,
                                fill: true
                            }
                        ]
                    },
                    options: {
                        maintainAspectRatio: false,
                        plugins: {
                            zoom: {
                                zoom: {
                                    drag: { enabled: true },
                                    mode: 'x'
                                }
                            }
                        },
                        scales: {
                            x: {
                                type: 'time',
                                time: {
                                    displayFormats: {
                                        second: 'HH:mm:ss',
                                        minute: 'HH:mm',
                                        hour: 'MMM dd HH:mm',
                                        day: 'MMM dd'
                                    },
                                    minUnit: 'second',
                                    tooltipFormat: 'MMM dd HH:mm:ss'
                                },
                                display: true,
                                title: { display: true, text: 'Date' }
                            },
                            y: {
                                beginAtZero: true,
                                max: this._config.max,
                                display: true,
                                title: { display: true, text: this._config.yLabel }
                            }
                        }
                    }
                });

                setInterval(() => this.refresh(), Utils.WIDGET_AUTOREFRESH_MS);
            },
            // onError
            (error) => {
                this._error
                    .children('.message')
                    .html(
                        `Load failed with status ${error.status}<br/>${Utils.formatError(error.responseText)}
                        <br/>Please try to reload the page.`
                    );

                this._loading.hide();
                this._error.fadeIn(Utils.UI_FADE_DURATION_MS);
            },
            // method
            'POST',
            // payload
            JSON.stringify({
                sdf: '/admin/com.enginframe.admin.xml',
                uri: '//com.enginframe.admin/server.load.data',
                options: []
            })
        );

        // Use arrow function to preserve the right 'this' in methods
        jQuery('#' + id + '-refresh').on('click', () => this.refresh());
        jQuery('#' + id + '-reset-zoom').on('click', () => this.resetZoom());
    }

    refresh() {
        const chart = this._content;
        const errorDiv = this._error;
        const loading = this._loading;
        const config = this._config;

        if (chart !== null) {
            errorDiv.fadeOut(Utils.UI_FADE_DURATION_MS);
            loading.show();

            Utils.invokeRest(
                '/system/services',
                (response) => {
                    const output = JSON.parse(response.output);
                    // FIXME: add support for multiple servers (EF Ent)
                    const server = output.servers[0].server;

                    chart.data.labels = server.map(d => new Date(d.timestamp * 1000));
                    // Datasets definition is hardcoded, so we can safely target them
                    chart.data.datasets[0].data = server.map(d => {
                        const item = d.data.find(i => i.resource === config.resource);
                        return (item) ? Utils.toMBytes(item.value, item.unit) : null;
                    });

                    loading.hide();
                    chart.update();
                },
                (error) => {
                    errorDiv
                        .children('.message')
                        .html(`Refresh failed with status ${error.status}<br/>${Utils.formatError(error.responseText)}`);

                    loading.hide();
                    errorDiv.fadeIn(Utils.UI_FADE_DURATION_MS);
                },
                'POST',
                JSON.stringify({
                    sdf: '/admin/com.enginframe.admin.xml',
                    uri: '//com.enginframe.admin/server.load.data',
                    options: []
                })
            );
        }
    }

    resetZoom() {
        const chart = this._content;

        if (chart !== null) {
            chart.resetZoom();
        }
    }
}

class LicensesWidget {
    _content = null;
    _loading = null;
    _error = null;

    constructor(id) {
        this._loading = jQuery('#' + id + '-loading').html('Loading...');
        this._error = jQuery('#' + id + '-error');
        this._error.children('.close').on('click', function() {
            jQuery(this).parent().fadeOut(Utils.UI_FADE_DURATION_MS)
        });

        Utils.invokeRest('/monitor/licenses',
            // onSuccess callback
            (response) => {
                this._loading.hide();
                this._content = new Chart(jQuery('#' + id), {
                    type: 'bar',
                    data: {
                        labels: response.map(license => license.component),
                        datasets: [
                            {
                                label: 'Used Licenses',
                                data: response.map(license => license['used-tokens']),
                                backgroundColor: Utils.CHART_COLORS_TRANSPARENT.blue,
                                borderColor: Utils.CHART_COLORS.blue,
                                borderWidth: 1
                            },
                            {
                                label: 'Total Licenses',
                                data: response.map(license => license['total-tokens']),
                                backgroundColor: Utils.CHART_COLORS_TRANSPARENT.green,
                                borderColor: Utils.CHART_COLORS.green,
                                borderWidth: 1
                            }
                        ]
                    },
                    options: {
                        maintainAspectRatio: false,
                        indexAxis: 'y',
                        scales: {
                            y: {
                                stacked: true,
                                beginAtZero: true
                            }
                        }
                    }
                });

                setInterval(() => this.refresh(), Utils.WIDGET_AUTOREFRESH_MS);
            },
            // onError callback
            (error) => {
                this._error
                    .children('.message')
                    .html(
                        `Load failed with status ${error.status}<br/>${Utils.formatError(error.responseText)}
                        <br/>Please try to reload the page.`
                    );

                this._loading.hide();
                this._error.fadeIn(Utils.UI_FADE_DURATION_MS);
            }
        );

        // Use arrow function to preserve the right 'this' in methods
        jQuery('#' + id + '-refresh').on('click', () => this.refresh());
    }

    refresh() {
        const chart = this._content;
        const errorDiv = this._error;
        const loading = this._loading;

        if (chart !== null) {
            errorDiv.fadeOut(Utils.UI_FADE_DURATION_MS);
            loading.show();

            Utils.invokeRest('/monitor/licenses',
                (response) => {
                    // Datasets definition is hardcoded, so we can safely target them
                    chart.data.datasets[0].data = response.map(license => license['used-tokens']);
                    chart.data.datasets[1].data = response.map(license => license['total-tokens']);

                    loading.hide();
                    chart.update();
                },
                (error) => {
                    errorDiv
                        .children('.message')
                        .html(`Refresh failed with status ${error.status}<br/>${Utils.formatError(error.responseText)}`);

                    loading.hide();
                    errorDiv.fadeIn(Utils.UI_FADE_DURATION_MS);
                }
            );
        }
    }
};

class StorageWidget {
    _contents = new Map();
    _loading = null;
    _error = null;

    constructor(id) {
        this._loading = jQuery('#' + id + '-loading').html('Loading...');
        this._error = jQuery('#' + id + '-error');
        this._error.children('.close').on('click', function() {
            jQuery(this).parent().fadeOut(Utils.UI_FADE_DURATION_MS)
        });

        Utils.invokeRest('/system/services',
            // onSuccess callback
            (response) => {
                const output = JSON.parse(response.output);
                const mounts = new Map(output.map(item => [item.mount, item.storages]));
                const $container = jQuery('#' + id);

                // If we have more than one mountpoint, use a grid layout for the widget content
                // (2 charts per row) and make some more space for the Storage widget
                if (mounts.size > 1) {
                    let gap = mounts.size === 2 ? '25px' : '4px'
                    $container.css({
                        display: 'grid',
                        width: '100%',
                        height: '100%',
                        gap: gap,
                        gridTemplateColumns: 'repeat(2, minmax(0, 1fr))',
                        gridAutoRows: 'minmax(0, auto)'
                    });
                    Utils.expandStorageWidget();
                }

                this._loading.hide();
                // Create and add one canvas for each mountpoint found
                for (const [mount, storages] of mounts) {
                    const $canvasContainer = jQuery('<div/>', {
                        id: `chart-container-${CSS.escape(mount)}`,
                        style: 'position: relative; max-height: 260px'
                    }).appendTo($container);
                    const $canvas = jQuery('<canvas/>', {
                        id: `chart-${CSS.escape(mount)}`,
                    }).appendTo($canvasContainer);
                    // Filter out empty storages
                    const nonEmptyStorages = storages.filter(storage => storage.size > 0);

                    this._contents.set(mount, new Chart($canvas, {
                        type: 'pie',
                        data: {
                            labels: nonEmptyStorages.map(storage => storage.storage),
                            datasets: [
                                {
                                    label: 'Size (MB)',
                                    data: nonEmptyStorages.map(storage => Utils.toMBytes(storage.size, storage.unit)),
                                    backgroundColor: Utils.getPieChartColors(nonEmptyStorages)
                                }
                            ]
                        },
                        options: {
                            maintainAspectRatio: false,
                            plugins: {
                                title: {
                                    display: true,
                                    text: `Mount point: ${mount}`,
                                    padding: 4
                                },
                                legend: {
                                    position: 'top',
                                    labels: {
                                        boxWidth: 12,
                                        padding: 7,
                                    }
                                }
                            }
                        }
                    }));
                }

                setInterval(() => this.refresh(), Utils.WIDGET_AUTOREFRESH_MS);
            },
            // onError callback
            (error) => {
                this._error
                    .children('.message')
                    .html(
                        `Load failed with status ${error.status}<br/>${Utils.formatError(error.responseText)}
                        <br/>Please try to reload the page.`
                    );

                this._loading.hide();
                this._error.fadeIn(Utils.UI_FADE_DURATION_MS);
            },
            // method
            'POST',
            // data sent to REST endpoint
            JSON.stringify({
                sdf: '/admin/com.enginframe.admin.xml',
                uri: '//com.enginframe.admin/storage.data',
                options: []
            })
        );

        // Use arrow function to preserve the right 'this' in methods
        jQuery('#' + id + '-refresh').on('click', () => this.refresh());
    }

    refresh() {
        const charts = this._contents;
        const loading = this._loading;
        const errorDiv = this._error;

        if (charts.size !== 0) {
            errorDiv.fadeOut(Utils.UI_FADE_DURATION_MS);
            loading.show();

            Utils.invokeRest('/system/services',
                (response) => {
                    const output = JSON.parse(response.output);
                    const mounts = new Map(output.map(item => [item.mount, item.storages]));

                    loading.hide();
                    for (const [mount, storages] of mounts) {
                        if (charts.has(mount)) {
                            const chart = charts.get(mount);
                            const nonEmptyStorages = storages.filter(storage => storage.size > 0);

                            chart.data.labels = nonEmptyStorages.map(storage => storage.storage),
                            // Datasets definition is hardcoded, so we can safely target them
                            chart.data.datasets[0].data = nonEmptyStorages
                                .map(storage => Utils.toMBytes(storage.size, storage.unit))
                            chart.update();
                        }
                    }
                },
                (error) => {
                    errorDiv
                        .children('.message')
                        .html(`Refresh failed with status ${error.status}<br/>${Utils.formatError(error.responseText)}`);

                    loading.hide();
                    errorDiv.fadeIn(Utils.UI_FADE_DURATION_MS);
                },
                'POST',
                JSON.stringify({
                    sdf: '/admin/com.enginframe.admin.xml',
                    uri: '//com.enginframe.admin/storage.data',
                    options: []
                })
            );
        }
    }
};
