#!/bin/bash

################################################################################
# Copyright 2023-2025 by NI SP Software GmbH, All rights reserved.
# Copyright 1999-2023 by Nice, srl., All rights reserved.
#
# This software includes confidential and proprietary information
# of NI SP Software GmbH ("Confidential Information").
# You shall not disclose such Confidential Information
# and shall use it only in accordance with the terms of
# the license agreement you entered into with NI SP Software.
################################################################################

# This script delete the AWS WorkSpaces Core instance associated to an Interactive DCV Session.
# This is complementary to the sample.workspaces.session.pre.submit.hook
# It is meant to be used with DCV 2017 (and later) interactive sessions only.

# The script does not require any configuration parameter. The required info
# will be propagated by the pre.submit hook through the SESSION_WORKSPACES_* env variables.


# Configuration parameters from pre.submit hook
REGION="$SESSION_WORKSPACES_AWS_REGION"
REQUIRED_SESSION_CLASS_PREFIX="$SESSION_WORKSPACES_REQUIRED_SESSION_CLASS_PREFIX"
INSTANCE_ACTION_AT_SESSION_CLOSE="$SESSION_WORKSPACES_INSTANCE_ACTION_AT_SESSION_CLOSE"

# Runtime variables from pre.submit hook
WORKSPACES_INSTANCE_ID="$SESSION_WORKSPACES_INSTANCE_ID"
EC2_INSTANCE_ID="$SESSION_WORKSPACES_EC2_INSTANCE_ID"


# Check for required tools
check_dependencies() {
  # Check for AWS CLI
  if ! command -v aws >/dev/null 2>&1; then
    echo "Error: AWS CLI not found."
    exit 1
  else
    # Try to retrieve the list of regions
    local _regions
    _regions=$(aws ec2 describe-regions --query "Regions[].RegionName" --output text 2>/dev/null)

    if [ -z "${_regions}" ]; then
        echo "AWS CLI is installed but failed to retrieve regions. Check your credentials or network."
        exit 1
    fi
  fi

  # Check for jq
  if ! command -v jq >/dev/null 2>&1; then
      echo "Error: jq not found."
      exit 1
  fi
}


# Print environment variables from pre submit hook
print_session_info() {
  printf "\nSession %s for OS %s executed on cluster %s\n\n" "${EF_SPOOLER_NAME}" "${INTERACTIVE_SESSION_OS}" "${INTERACTIVE_SESSION_CLUSTER}"

  printf "Environment variables with SESSION_ prefix from pre.submit hook:\n\n"
  env | grep '^SESSION_' | sort
}


# Deallocate the WorkSpace instance (and terminate related EC2 instance)
delete_workspace_instance() {
  local _workspace_instance_id="$1"
  local _region="$2"

  if [ -z "${_workspace_instance_id}" ]; then
    echo "Error: WorkSpace Instance Id not available. Unable to delete it."
    exit 1
  else
    echo "Deleting WorkSpace instance ${_workspace_instance_id}..."
    aws workspaces-instances delete-workspace-instance \
     --region "${_region}" --workspace-instance-id "${_workspace_instance_id}"
   fi
}


# Stop the EC2 instance associated to the WorkSpace instance
stop_ec2_instance() {
  local _ec2_instance="$1"
  local _region="$2"

  if [ -z "${_ec2_instance}" ]; then
    echo "Error: EC2 Instance Id not available. Unable to stop it."
    exit 1
  else
    echo "Stopping EC2 instance ${_ec2_instance}..."
    aws ec2 stop-instances --region "${_region}" --instance-ids "${_ec2_instance}" >/dev/null
   fi
}


# Print WorkSpace and EC2 instance state
print_instances_status() {
  local _workspace_instance_id="$1"
  local _region="$2"

  local _status_output
  local _provision_state
  local _instance_state
  local _ec2_instance_id

  if [ -z "${_workspace_instance_id}" ]; then
    echo "Warning: WorkSpace Instance Id not available. Skipping status check."
  else
    _status_output=$(aws workspaces-instances get-workspace-instance \
      --region "${_region}" \
      --workspace-instance-id "${_workspace_instance_id}")
    # Output is something like:
    # { "ProvisionState": "ALLOCATED", "WorkspaceInstanceId": "wsinst-123", "EC2ManagedInstance": { "InstanceId": "i-123" } }
    _provision_state=$(echo "${_status_output}" | jq -r '.ProvisionState')
    printf "Current WorkSpace Instance state: %s\n" "${_provision_state}"

    _ec2_instance_id=$(echo "${_status_output}" | jq -r '.EC2ManagedInstance.InstanceId')
    _instance_state=$(aws ec2 describe-instances \
      --instance-ids "${_ec2_instance_id}" \
      --region "${_region}" \
      --query "Reservations[*].Instances[*].State.Name" \
      --output text)
    printf "Current EC2 Instance state: %s\n" "${_instance_state}"
  fi
}


main () {
  if [[ $INTERACTIVE_SESSION_CLASS == $REQUIRED_SESSION_CLASS_PREFIX* ]]; then
    check_dependencies
    print_session_info

    # INSTANCE_ACTION_AT_SESSION_CLOSE is a configuration parameter from the pre.submit hook and propagated to the closing hook
    if [ -z "$INSTANCE_ACTION_AT_SESSION_CLOSE" ]; then
      printf "\nINSTANCE_ACTION_AT_SESSION_CLOSE not set — Preserving both WorkSpace and EC2 instance.\n"
    elif [ "$INSTANCE_ACTION_AT_SESSION_CLOSE" = "delete" ]; then
      printf "\nINSTANCE_ACTION_AT_SESSION_CLOSE set to 'delete' — Deallocating WorkSpace instance and terminating related EC2 instance.\n"
      delete_workspace_instance "$WORKSPACES_INSTANCE_ID" "$REGION"
    elif [ "$INSTANCE_ACTION_AT_SESSION_CLOSE" = "stop" ]; then
      printf "\nINSTANCE_ACTION_AT_SESSION_CLOSE set to 'stop' — Preserving WorkSpace instance but stopping related EC2 instance.\n"
      stop_ec2_instance "$EC2_INSTANCE_ID" "$REGION"
    else
      printf "\nINSTANCE_ACTION_AT_SESSION_CLOSE is %s. This is an unexpected value. Preserving both WorkSpace and EC2 instance.\n" "$INSTANCE_ACTION_AT_SESSION_CLOSE"
    fi

    print_instances_status "$WORKSPACES_INSTANCE_ID" "$REGION"
    printf "\nInteractive Session closing hook completed.\n\n"
  else
    # Skip hook for sessions where Session Class does not have "workspaces_hook" as prefix
    echo "INTERACTIVE_SESSION_CLASS is '$INTERACTIVE_SESSION_CLASS'. It does not have '$REQUIRED_SESSION_CLASS_PREFIX' has prefix. Skipping the hook."
  fi
}


main
