#!/bin/bash

##############################################################################
##################################################################################
# Copyright 2023-2025 by NI SP Software GmbH, All rights reserved.
# Copyright 1999-2023 by Nice, srl., All rights reserved.
#
# This software includes confidential and proprietary information
# of NI SP Software GmbH ("Confidential Information").
# You shall not disclose such Confidential Information
# and shall use it only in accordance with the terms of
# the license agreement you entered into with NI SP Software.
################################################################################
###################################################################################
##############################################################################
##
## This wrapper scripts calls <ef-ks.sh> script to manage DB Credentials
## stored in a keystore
##
##########################################################################

# EF_DB_CREDENTIALS_KEYS
# Ref. com.enginframe.server.Configuration
#
EF_DB_ADMIN_NAME="ef.db.admin.name"
EF_DB_ADMIN_PASSWORD="ef.db.admin.password"
EF_DB_USER_NAME="ef.db.user.name"
EF_DB_USER_PASSWORD="ef.db.user.password"
SCRIPT_DIR="$( cd -- "$( dirname -- "${BASH_SOURCE[0]}" )" &> /dev/null && pwd )"

usage() {
    local _cmd=$(basename "$0")
    cat <<EOF
Usage: $_cmd --action setDBAdmin|deleteDBAdmin|setDBUser|deleteDBUser|listKeys [--name <name>] [--password <password>]
  --action <action>
                 Carry out the specified action. Possible values are:
                 * setDBAdmin: add the name/password pair for the admin to the keystore
                 * deleteDBAdmin: delete the name/password for the specified admin from the keystore 
                 * setDBUser: add the name/password pair for the user to the keystore
                 * deleteDBUser: deletes the name/password pair for the user from the keystore 
                 * listKeys: list all the keys in the keystore

  For setDBAdmin and setDBUser (all required):
  --name <user or admin name>
                 The user or admin name

  --password <user or admin password>
                 The user or admin password

EOF
}

# Detect the EnginFrame directory from the current script.
# Known limitations: don't resolve symlinks
#
get_enginframe_dir() {
    local -- _target="${BASH_SOURCE[0]}"
    (cd "$(dirname "${_target}")/.." 2>/dev/null && pwd)
}

bail() {
    echo -e "$1"
    exit 1
}

main() {
    export EF_ROOT="$(get_enginframe_dir)"

    java_home=$(cat $EF_ROOT/conf/enginframe.conf | grep JAVA_HOME | cut -d'=' -f2 | tr -d '"')
    ks_name=$(cat $EF_ROOT/conf/enginframe/server.conf | grep EF_DB_KEY_STORE= | cut -d'/' -f4)
    plain_pwd=$(cat $EF_ROOT/conf/enginframe/server.conf | grep EF_KEY_STORE_PASSWORD | cut -d'=' -f2)

    [ ! -z "$java_home" ] || bail "JAVA_HOME in $EF_ROOT/conf/enginframe.conf should not be empty"
    [ ! -z "$ks_name" ] || bail "EF_DB_KEY_STORE in $EF_ROOT/conf/enginframe/server.conf should not be empty"
    [ ! -z "$plain_pwd" ] || bail "EF_KEY_STORE_PASSWORD in $EF_ROOT/conf/enginframe/server.conf should not be empty"

    ks_pwd=$(echo -n ${plain_pwd} | sha256sum | cut -d' ' -f1)

    ks="$EF_ROOT/conf/enginframe/certs/$ks_name"

    if [ $# -eq 0 ]; then
        usage
        exit 1
    fi

    while [ $# -gt 0 ] ; do
        case "$1" in
            --action)
                _action="$2"
                shift
                ;;
            --name)
                _name="$2"
                shift
                ;;
            --password)
                _password="$2"
                shift
                ;;
            *)
                usage
                exit 1
                ;;
        esac
        shift
    done

    [ -z "$_action" ] && bail "Command line argument 'action' not provided"

    case "$_action" in
        setDBAdmin)
            [ -z "$_name" ] && bail "Command line argument 'name' for DBAmin not provided"
            [ -z "$_password" ] && bail "Command line argument 'password' for DBAmin not provided"
            setDBAdmin $_name $_password
            ;;
        deleteDBAdmin)
            deleteUser $EF_DB_ADMIN_NAME $EF_DB_ADMIN_PASSWORD "DBAdmin"
            ;;
        setDBUser)
            [ -z "$_name" ] && bail "Command line argument 'name' for DBUser not provided"
            [ -z "$_password" ] && bail "Command line argument 'password' for DBUser not provided"
            setDBUser $_name $_password
            ;;
        deleteDBUser)
            deleteUser $EF_DB_USER_NAME $EF_DB_USER_PASSWORD "DBUser"
            ;;
        listKeys)
            listKeys
            ;;
        *)
            bail "Unsupported action: ${_action}"
            ;;
    esac
}

setDBAdmin() {
  local _name="$1"
  local _password="$2"
  setUser $EF_DB_ADMIN_NAME $_name $EF_DB_ADMIN_PASSWORD $_password "DBAdmin"
}

setDBUser() {
  local _name="$1"
  local _password="$2"
  setUser $EF_DB_USER_NAME $_name $EF_DB_USER_PASSWORD $_password "DBUser"
}

setUser() {
  local _name_key="$1"
  local _name_value="$2"
  local _password_key="$3"
  local _password_value="$4"
  local _user_name="$5"
  echo "Setting $_user_name Name"
  ${SCRIPT_DIR}/enginframe.keystore.sh --javahome "$java_home" --keystore "$ks" --password "$ks_pwd" --action addAlias --alias "$_name_key" --value "$_name_value"
  echo "Setting $_user_name Password"
  ${SCRIPT_DIR}/enginframe.keystore.sh --javahome "$java_home" --keystore "$ks" --password "$ks_pwd" --action addAlias --alias "$_password_key" --value "$_password_value"
}

deleteUser() {
  local _name_key="$1"
  local _password_key="$2"
  local _user_name="$3"
  echo "Deleting $_user_name Name"
  ${SCRIPT_DIR}/enginframe.keystore.sh --javahome "$java_home" --keystore "$ks" --password "$ks_pwd" --action deleteKey --alias "$_name_key"
  echo "Deleting $_user_name Password"
  ${SCRIPT_DIR}/enginframe.keystore.sh --javahome "$java_home" --keystore "$ks" --password "$ks_pwd" --action deleteKey --alias "$_password_key"
}

listKeys() {
  ${SCRIPT_DIR}/enginframe.keystore.sh --javahome "$java_home" --keystore "$ks" --password "$ks_pwd" --action listKeys
}

main "$@"
