#!/bin/bash

############################################################################
##################################################################################
# Copyright 2023-2025 by NI SP Software GmbH, All rights reserved.
# Copyright 1999-2023 by Nice, srl., All rights reserved.
#
# This software includes confidential and proprietary information
# of NI SP Software GmbH ("Confidential Information").
# You shall not disclose such Confidential Information
# and shall use it only in accordance with the terms of
# the license agreement you entered into with NI SP Software.
################################################################################
#################################################################################
############################################################################
##
## KeyTool docs
## https://docs.oracle.com/javase/8/docs/technotes/tools/unix/keytool.html
##
## Note: keytool is designed to prevent extraction of private keys and secret keys.
## To extract a value associated with a key in the keystore Java or other tools must be used.
##
##########################################################################

usage() {
    local _cmd=$(basename "$0")
    cat <<EOF
Usage: $_cmd --action addAlias|listKeys|containsKey|deleteKey --javahome <java home> --keystore <keystore> --password <password> [--alias <alias>] [--value <value>]
  --action <action>
                 Carry out the specified action. Possible values are:
                 * addAlias: add a key with the given alias to the keystore
                 * listKeys: list all the keys in the keystore
                 * containsKey: check that the keystore contains the provided key
                 * deleteKey: delete the provided key from the keystore

  --javahome <java home>
                 The Java home

  --password <password>
                 The password

  --keystore <keystore>
                 The keystore

  For addAlias, containsKey and deleteKey (all required):
  --alias <value>
                 The key alias

  For addAlias (required):
  --value <value>
                 The key value

EOF
}

bail()
{
    echo -e "$1"
    exit 1
}

main() {
    if [ $# -eq 0 ]; then
        usage
        exit 1
    fi

    while [ $# -gt 0 ] ; do
        case "$1" in
            --javahome)
                _javahome="$2"
                shift
                ;;
            --action)
                _action="$2"
                shift
                ;;
            --keystore)
                _keystore="$2"
                shift
                ;;
            --password)
                _password="$2"
                shift
                ;;
            --alias)
                _alias="$2"
                shift
                ;;
            --value)
                _value="$2"
                shift
                ;;
            *)
                usage
                exit 1
                ;;
        esac
        shift
    done

    [ -z "$_javahome" ] && bail "Command line argument 'javahome' not provided"

    case "$_action" in
        addAlias)
            [ -z "$_keystore" ] && bail "Command line argument 'keystore' not provided"
            [ -z "$_password" ] && bail "Command line argument 'password' not provided"
            [ -z "$_alias" ] && bail "Command line argument 'alias' not provided"
            [ -z "$_value" ] && bail "Command line argument 'value' not provided"
            response=$(addalias $_javahome $_keystore $_password $_alias $_value)
            ;;
        listKeys)
            [ -z "$_keystore" ] && bail "Command line argument 'keystore' not provided"
            [ -z "$_password" ] && bail "Command line argument 'password' not provided"
            response=$(listKeys $_javahome $_keystore $_password)
            ;;
        containsKey)
            [ -z "$_keystore" ] && bail "Command line argument 'keystore' not provided"
            [ -z "$_password" ] && bail "Command line argument 'password' not provided"
            [ -z "$_alias" ] && bail "Command line argument 'alias' not provided"
            response=$(containsKey $_javahome $_keystore $_password $_alias)
            ;;
        deleteKey)
            [ -z "$_keystore" ] && bail "Command line argument 'keystore' not provided"
            [ -z "$_password" ] && bail "Command line argument 'password' not provided"
            [ -z "$_alias" ] && bail "Command line argument 'alias' not provided"
            response=$(deleteKey $_javahome $_keystore $_password $_alias)
            ;;
        *)
            bail "Unsupported action"
            ;;
    esac

    status_code=$(echo $?)
    prefix_to_strip="keytool error: java.lang.Exception: "
    stripped_response=${response#"$prefix_to_strip"}
    if [ ! -z "$stripped_response" ]; then
        echo -e "$stripped_response"
    fi
    exit $status_code
}

addalias() {
  local _javahome="$1"
  local _keystore="$2"
  local _password="$3"
  local _alias="$4"
  local _value="$5"

  deleteKey "$_javahome" "$_keystore" "$_password" "$_alias"
  echo -e "$_value\n$_value" | "$_javahome"/bin/keytool -importpass -storetype pkcs12 -alias "$_alias" -keystore "$_keystore" -storepass "$_password"
}

listKeys() {
  local _javahome="$1"
  local _keystore="$2"
  local _password="$3"
  echo -e "$_value\n$_value" | "$_javahome"/bin/keytool -list -keystore "$_keystore" -storepass "$_password"
}

containsKey() {
  local _javahome="$1"
  local _keystore="$2"
  local _password="$3"
  local _alias="$4"
  result=$("$_javahome"/bin/keytool -list -alias "$_alias" -keystore "$_keystore" -storepass "$_password")
  if [ $? -eq 0 ]; then
    echo "TRUE"
  else
    echo "FALSE"
  fi
}

deleteKey() {
  local _javahome="$1"
  local _keystore="$2"
  local _password="$3"
  local _alias="$4"

  result=$(containsKey "$_javahome" "$_keystore" "$_password" "$_alias")
  if [ "$result" = "TRUE" ]; then
    "$_javahome"/bin/keytool -delete -alias "$_alias" -keystore "$_keystore" -storepass "$_password"
  fi
}

check() {
  printenv >> /tmp/remove-me.log
  date >> /tmp/remove-me.log
}

install() {
  echo "install"
}

main "$@"
