#!/bin/bash

usage() {
    local _cmd=$(basename "$0")
    cat <<EOF
Usage: $_cmd --roles=server,agent,derby [--hostname=<host name>] [--boot=y|n] [--default-agent=<agent host name>] [--rmi-host=<agent host name>]
  -r <comma separated roles>, --roles=<comma separated roles>
                 configures the EnginFrame services to start on the host
  -H <host name>, --hostname=<host name>
                 specify the host name the roles apply to
                 If not specified uses the current hostname.
  -b <y|n>, --boot=<y|n>
                 configures if EnginFrame must be started at boot on this host
                 It cannot be specified together with --hostname

  For server role only:
  --default-agent=<agent host name>
                 configures the default agent EnginFrame server will use

  For agent role only:
  --rmi-host=<agent host name>
                 configures the host name that should be associated with remote stubs for locally created remote objects

EOF
}

error() {
    echo "ERROR: $1"
    echo ""
    exit 1
}

error_usage() {
    echo "ERROR: $1"
    echo ""
    usage
    exit 1
}

print_conf_file() {
    local _conf_file="$1"
    if [ -e "${_conf_file}" ]; then
        echo "--- ${_conf_file} ---"
        cat "${_conf_file}"
        echo ""
    else
        echo "--- ${_conf_file} - Not present ---"
    fi
}

main() {
    local _roles
    local _hostname
    local _boot

    _boot="n"
    _hostname=$(hostname)

    while [ $# -gt 0 ] ; do
        case "$1" in
            -r|--roles)
                _roles="$2"
                shift
                ;;
            --roles=*)
                _roles="${1#*=}"
                ;;
            -H|--hostname)
                _hostname="$2"
                shift
                ;;
            --hostname=*)
                _hostname="${1#*=}"
                ;;
            -b|--boot)
                _boot="$2"
                shift
                ;;
            --boot=*)
                _boot="${1#*=}"
                ;;
            --default-agent)
                _default_agent="$2"
                shift
                ;;
            --default-agent=*)
                _default_agent="${1#*=}"
                ;;
            --rmi-host)
                _rmi_host="$2"
                shift
                ;;
            --rmi-host=*)
                _rmi_host="${1#*=}"
                ;;
            *)
                usage
                exit 1
                ;;
        esac
        shift
    done

    if [ -z "${_roles}" ]; then
        echo "No roles specified."
    fi

    if [ "${_boot}" = "y" -a "${_hostname}" != "$(hostname)" ]; then
        error_usage "You cannot specify --boot=y and a host name different from the local host name"
    fi

    OLD_IFS="${IFS}"
    IFS=","
    _server_role="false"
    _agent_role="false"
    _derby_role="false"
    for _role in ${_roles} ; do
      case "${_role}" in
        "server")
          _server_role="true"
          ;;
        "agent")
          _agent_role="true"
          ;;
        "derby")
          _derby_role="true"
          ;;
      esac
    done
    IFS="${OLD_IFS}"

    if [ "${_server_role}" = "false" ]; then
        if [ -n "${_default_agent}" ]; then
            error_usage "Option --default-agent invalid for specified roles"
        fi
    fi

    if [ "${_agent_role}" = "false" ]; then
        if [ -n "${_rmi_host}" ]; then
            error_usage "Option --rmi-host invalid for specified roles"
        fi
    fi

    EF_VERSION_ROOT="$(dirname "$0")/../.."
    pushd "${EF_VERSION_ROOT}" > /dev/null 2>/dev/null || error "Unable to enter version root directory ${EF_VERSION_ROOT}"
    EF_VERSION_ROOT=$(pwd)
    popd > /dev/null

    if [ -z "${EF_TOP}" ]; then
        _ef_top="$(dirname ${EF_VERSION_ROOT}/../..)"
        pushd "${_ef_top}" > /dev/null 2>/dev/null || error "Unable to enter directory ${_ef_top}"
        EF_TOP=$(pwd)
        popd > /dev/null
    fi

    if [ -z "${EF_CONF}" ]; then
        _default_conf_root="${EF_VERSION_ROOT}/../conf"
        pushd "${_default_conf_root}" > /dev/null 2>/dev/null || error "Unable to enter configuration directory ${_default_conf_root}"
        EF_CONF="$(pwd)/enginframe.conf"
        popd > /dev/null
    fi

    . "${EF_CONF}"
    if [ -z "${EF_CONF_ROOT}" ]; then
        EF_CONF_ROOT="$(dirname "${EF_CONF}")"
    fi

    _local_conf="${EF_CONF_ROOT}/enginframe.${_hostname}.conf"
    _local_server_conf="${EF_CONF_ROOT}/enginframe/server.${_hostname}.conf"
    _local_agent_conf="${EF_CONF_ROOT}/enginframe/agent.${_hostname}.conf"

    if [ "${_boot}" = "y" ]; then
        export EF_VERSION_ROOT
        export EF_CONF
        export EF_TOP
        echo "Configuring system to start ${EF_VERSION_ROOT} with conf ${EF_CONF} at boot..."
        "${EF_VERSION_ROOT}/enginframe/tools/enginframe.service.sh" install
    fi

    if [ -n "${_roles}" ]; then
        echo "Writing host configuration files..."

        # Writing enginframe.<hostname>.conf
        echo "# Generated on $(date)"                   > "${_local_conf}"
        echo "EF_ROLES=\"${_roles}\""                  >> "${_local_conf}"

        # Writing server.<hostname>.conf
        if [ "${_server_role}" = "true" ]; then
          echo "# Generated on $(date)"                   > "${_local_server_conf}"
          if [ -n "${_default_agent}" ]; then
              echo "EF_AGENT_HOST=${_default_agent}"         >> "${_local_server_conf}"
          fi
        fi

        # Writing agent.<hostname>.conf
        if [ "${_agent_role}" = "true" ]; then
            echo "# Generated on $(date)"                   > "${_local_agent_conf}"
            if [ -n "${_rmi_host}" ]; then
                echo "java.rmi.server.hostname=${_rmi_host}"   >> "${_local_agent_conf}"
            fi
        fi
    fi

    echo ""
    echo "Configuration files for this host:"
    print_conf_file "${_local_conf}"
    print_conf_file "${_local_server_conf}"
    print_conf_file "${_local_agent_conf}"
}

main "$@"
