#!/bin/bash

compute_item() {
  local _file="$1"
  local _stat

  if [ -L "${_file}" -a ! -e "${_file}" ]; then
    _stat=$(LANG=C stat --printf='"%F","%G:%U","%a"' "${_file}")'"N/A"'
    _checksum="Broken link"
  elif [ -d "${_file}" ]; then
    _stat=$(LANG=C stat --printf='"%F","%G:%U","%a"' "${_file}")'"N/A"'
    _checksum="N/A"
  else
    _stat=$(LANG=C stat --printf='"%F","%G:%U","%a","%s"' "${_file}")
    if [ -r "${_file}" ]; then
        case "${_checksum_command}" in
            sha1sum)
                _checksum=$(sha1sum "${_file}" | sed 's/\([^ ]\) .*/\1/')
            ;;
            openssl)
                _checksum=$(openssl sha1 "${_file}" | sed 's/^.* //')
            ;;
            *)
                echo "No SHA1 command available"
                exit 1;
        esac
    else
        _checksum="N/A"
    fi
  fi
  echo "\"${_file}\",${_stat},\"${_checksum}\""
}

check() {
    local _new_report="$1"
    local _reference_report="$2"

    (
      echo "#### REFERENCE ####"
      cat "${_reference_report}"
      echo "#### CURRENT STATUS ####"
      cat "${_new_report}"
    ) | awk -f "${PROGRAM_ROOT}/compare-reports.awk"
    _exit_code=$?
    return ${_exit_code}
}

main() {
    local _directory
    local _input_report
    local _output_report
    local _temp_dir
    local _changes
    local _exit_code

    case "$1" in
      "check")
        _directory="$2"
        _input_report="$3"
        _changes_file="$4"

        _temp_dir=$(mktemp -d /tmp/EF_REPORT.$(date +"%Y%d%m-%H%M%S").XXXXXX)
        _output_report="${_temp_dir}/report.txt"

        echo -n "Storing temporary report of \"${_directory}\" to \"${_output_report}\": "
        compute "${_directory}" > "${_output_report}"
        echo "DONE"

        echo -n "Checking \"${_output_report}\" against report \"${_input_report}\": "
        check "${_output_report}" "${_input_report}" > "${_changes_file}"
        _exit_code=$?

        if [ "$_exit_code" = "0" ]; then
            echo "SUCCESS"
        else
            echo "FAILED"
            echo ""
            echo "Changes:"
            cat "${_changes_file}"
        fi

        rm -rf "${_temp_dir}"
        exit ${_exit_code}
        ;;

      "compute")
        _directory="$2"
        _output_report="$3"

        echo -n "Storing report of \"${_directory}\" to \"${_output_report}\": "
        compute "${_directory}" > "${_output_report}"
        echo "DONE"
        ;;

      *)
        echo "Error" >&2
        exit 1
        ;;
    esac
}

compute() {
    local _directory="$1"

    if [ -z "${_checksum_command}" ]; then
        which "sha1sum" >/dev/null
        if [ "$?" = "0" ]; then
            _checksum_command="sha1sum"
        else
            which "openssl" >/dev/null
            if [ "$?" = "0" ]; then
                _checksum_command="openssl"
            fi
        fi
    fi

    pushd "${_directory}" >/dev/null
    IFS=$'\n'
    for item in $(find . \! \( -regex '^\./install$' -or -regex '^\./install/.*$' \)); do
      compute_item "${item}"
    done
    popd >/dev/null
}

PROGRAM_ROOT="$(dirname "$0")"
REPORT_SCRIPT="${PROGRAM_ROOT}/report.sh"

main "$@"
