#!/bin/bash

# Variables that must be set in the environment:
# JAVA_HOME

. "$(dirname $0)/certificates-utils.sh"

# Files name
_server_keystore="server.keystore"
_server_cert="server.cert"
_server_truststore="server.truststore"
_agent_keystore="agent.keystore"
_agent_cert="agent.cert"
_agent_truststore="agent.truststore"
_keystore_password="keystore.password"

# Constants
_CNS="EFPortal-Server"
_CNA="EFPortal-Agent"
_OU="EFPortal"
_O="NI SP Software GmbH"
_L="Tuebingen"
_S="Tuebingen"
_C="DE"

# 10 year validity
_validity=3650

fail() {
    echo "ERROR: $1" >&2
    exit 1
}

log() {
    echo "INFO : $1"
}

syntax() {
    _cmd=$(basename "$0")
    echo "Syntax"
    echo "  $_cmd generate"
    echo "      Generates self-signed certificates for EF Server and EF Agent into the"
    echo "      EF_CONF_ROOT/enginframe/certs/ folder."
    echo "      The certificates password is read from the EF_CONF_ROOT/enginframe/certs/${_keystore_password} file."
    echo ""
    echo "      -"
    echo "          The certificates password is provided as standard input,"
    echo "          and will be stored into a ${_keystore_password} file in the same location as the certificates files."
    echo ""
    echo "      --tomcatuser user, --tomcatuser=user"
    echo "          Specify the user running Tomcat"
    echo "          (needed to set right permission on the certificates)."
    echo ""
    echo "  $_cmd list"
    echo "      List certificates in the EF_CONF_ROOT/enginframe/certs/ folder."
    echo ""
    echo "  $_cmd clean"
    echo "      Removes all certificates except the ${_keystore_password} file. Use with care!!!"
    echo ""
    echo ""
    echo "Examples"
    echo "  echo \"myPassword\" | $_cmd - generate --tomcatuser efnobody"
    echo "  $_cmd generate --tomcatuser=efnobody"
    echo "  $_cmd list"
    echo "  $_cmd clean"
    echo ""
}

generate() {
    # Variables that must be set in the environment:
    # _tomcatuser
    # _ef_cert_folder
    # _keystore_password_file
    # _password
    # _validity
    # _server_keystore
    # _server_cert
    # _server_truststore
    # _agent_keystore
    # _agent_cert
    # _agent_truststore

    [ -z "${_tomcatuser}" ] && fail "TOMCAT_USER must be set into environment."
    [ -z "${_ef_cert_folder}" ] && fail "EF certificate folder must be set into environment."
    [ -z "${_keystore_password_file}" ] && fail "Keystore password file must be set into environment."
    [ -z "${_password}" ] && fail "Password must be set into environment."
    [ -z "${_validity}" ] && fail "Certificate validity must be set into environment."
    [ -z "${_server_keystore}" ] && fail "EF server keystore name must be set into environment."
    [ -z "${_server_cert}" ] && fail "EF server certificate name must be set into environment."
    [ -z "${_server_truststore}" ] && fail "EF server truststore name must be set into environment."
    [ -z "${_agent_keystore}" ] && fail "EF agent keystore name must be set into environment."
    [ -z "${_agent_cert}" ] && fail "EF agent certificate name must be set into environment."
    [ -z "${_agent_truststore}" ] && fail "EF agent truststore name must be set into environment."

    mkdir -p "${_ef_cert_folder}" || fail "Unable to create ${_ef_cert_folder}."

    echo "${_password}" > "${_keystore_password_file}" || fail "Unable to write to \"${_keystore_password_file}\"."
    log "Keystore password file created: ${_keystore_password_file}"
    chmod 600 "${_keystore_password_file}" || fail "Unable to chmod \"${_keystore_password_file}\"."

    # Create the EF Server Private Keys and Public Certificates
    _ef_server_keystore="${_ef_cert_folder}/${_server_keystore}"
    _ef_server_dname="CN=${_CNS},OU=${_OU},O=${_O},L=${_L},S=${_S},C=${_C}"
    _ef_server_cert="${_ef_cert_folder}/${_server_cert}"
    create_cert "${_ef_server_keystore}" "${_CNS}" "${_ef_server_dname}" "${_tomcatuser}" "${_password}"
    export_cert "${_ef_server_keystore}" "${_CNS}" "${_ef_server_cert}" "${_tomcatuser}" "${_password}"

    # Create the EF Agent Private Keys and Public Certificates
    _ef_agent_keystore="${_ef_cert_folder}/${_agent_keystore}"
    _ef_agent_dname="CN=${_CNA},OU=${_OU},O=${_O},L=${_L},S=${_S},C=${_C}"
    _ef_agent_cert="${_ef_cert_folder}/${_agent_cert}"
    create_cert "${_ef_agent_keystore}" "${_CNA}" "${_ef_agent_dname}" "root" "${_password}"
    export_cert "${_ef_agent_keystore}" "${_CNA}" "${_ef_agent_cert}" "root" "${_password}"

    # Build a cross trust between EF server and EF agent
    _ef_server_truststore="${_ef_cert_folder}/${_server_truststore}"
    _ef_agent_truststore="${_ef_cert_folder}/${_agent_truststore}"
    # cross_trust "${_ef_agent_truststore}" "${_CNA}" "${_ef_agent_cert}" "root" "${_password}" "${_ef_server_truststore}" "${_CNS}" "${_ef_server_cert}" "${_tomcatuser}" "${_password}"
    trust_cert "${_ef_agent_cert}" "${_CNA}" "${_ef_server_truststore}" "${_tomcatuser}" "${_password}"
    trust_cert "${_ef_server_cert}" "${_CNS}" "${_ef_agent_truststore}" "root" "${_password}"
}

init() {
    _cuid="$(id -u)"

    # script location is EF_ROOT/tools
    local -x EF_CONF_ROOT="$(readlink -f "$(dirname "$0")/../../../conf")"
    [ -d "${EF_CONF_ROOT}/enginframe" ] || fail "Could not find a valid EF_CONF_ROOT/enginframe folder. Please check script location."

    [ -x "${JAVA_HOME}/bin/java" ] || fail "Please specify a valid JAVA_HOME environment variable: '${JAVA_HOME}/bin/java' not found."
    [ -x "${JAVA_HOME}/bin/keytool" ] || fail "Please specify a valid JAVA_HOME environment variable: '${JAVA_HOME}/bin/keytool' not found."

    _ef_cert_folder="${EF_CONF_ROOT}/enginframe/certs"
    _keystore_password_file="${_ef_cert_folder}/${_keystore_password}"
}

parse_options() {
    _command=''
    _input_passwd=false
    _tomcatuser=''

    if [ $# -eq 0 ]; then
        syntax
        exit 0
    fi

    while [ $# -gt 0 ] ; do
        case "$1" in
            generate|list|clean)
               [ -n "${_command}" ] && fail "Only one command at a time can be used."
                _command="$1"
            ;;
            -)
                _input_passwd=true
            ;;
            --tomcatuser)
                _tomcatuser="$2"
                shift
            ;;
            --tomcatuser=*)
                _tomcatuser="${1#*=}"
            ;;
            -h|--help|help)
                syntax
                exit 0
            ;;
            *)
                syntax
                fail "Unrecognized option '$1'"
            ;;
        esac
        shift
    done
}

check_options() {
    if [ "${_command}" = "generate"  -o "${_command}" = "list" ]; then
        if [ "${_input_passwd}" = "true" ]; then
            _password=$(cat -)
        else
            if [ -f "${_keystore_password_file}" ]; then
                _password=$(cat "${_keystore_password_file}" | head -n 1)
            else
                fail "${_ef_cert_folder}/${_keystore_password} file does not exist, use - to pass password to STDIN to generate the file."
            fi
        fi
    fi

    if [ "${_command}" = "generate" ]; then
        [ -z "${_tomcatuser}" ] && fail "Please specify --tomcatuser option with valid TOMCAT_USER user as value."
        id "${_tomcatuser}"  > /dev/null 2>&1 || fail "Please specify a valid TOMCAT_USER user: '${_tomcatuser}' not found."
    fi
}

do_command() {
    case "${_command}" in
        clean)
            read -p "All certificates, keystore and trustore files are going to be removed. The ${_keystore_password} file will NOT be removed. Do you want to continue? [y/N] " -n 1 -r
            echo
            if [[ $REPLY =~ ^[Yy]$ ]]
            then
                for _cert in "${_ef_cert_folder}/${_server_keystore}" \
                             "${_ef_cert_folder}/${_server_truststore}" \
                             "${_ef_cert_folder}/${_server_cert}" \
                             "${_ef_cert_folder}/${_agent_keystore}" \
                             "${_ef_cert_folder}/${_agent_truststore}" \
                             "${_ef_cert_folder}/${_agent_cert}"; do
                    rm "${_cert}" > /dev/null 2>&1
                done
            fi
        ;;
        generate)
            generate
        ;;
        list)
            for _cert in "${_ef_cert_folder}/${_server_keystore}" \
                         "${_ef_cert_folder}/${_server_truststore}" \
                         "${_ef_cert_folder}/${_agent_keystore}" \
                         "${_ef_cert_folder}/${_agent_truststore}"; do
                list_cert "${_cert}" "${_password}"
            done
        ;;
    esac
}

main() {
    init
    parse_options "$@"
    check_options
    do_command
}

main "$@"
