/*
 * Copyright 2023-2025 by NI SP Software GmbH, All rights reserved.
 *
 * This software includes confidential and proprietary information
 * of NI SP Software GmbH ("Confidential Information").
 * You shall not disclose such Confidential Information
 * and shall use it only in accordance with the terms of
 * the license agreement you entered into with NI SP Software.
 */
const dotenv = require('dotenv');
const express = require('express');
const { createProxyMiddleware } = require('http-proxy-middleware');
const fs = require('fs');
const http = require('http');
const https = require('https');
const path = require('path');
const yaml = require('js-yaml');
const helmet = require('helmet');
const morgan = require('morgan');
const rateLimit = require('express-rate-limit');

const DEFAULT_PROXY_PORT = 3001;
const DEFAULT_TIMEOUT = 30000;
const DEFAULT_RATE_LIMIT_WINDOW_MS = 15 * 60 * 1000; // 15 minutes
const DEFAULT_RATE_LIMIT_REQUESTS = 1000; // Limit each IP to 1000 requests per window ms

// Logging
const LOG_LEVELS = { ERROR: 0, WARN: 1, INFO: 2, DEBUG: 3 };
const DEFAULT_LOG_LEVEL = "WARN";

// Required root key in the config
const REQUIRED_GLOBAL_KEYS = ["proxies"];
// Keys required for static routing
const REQUIRED_PROXY_STATIC_KEYS = ["path", "target"];
// Keys required for dynamic routing
// hostRegex is optional (defaults to localhost)
const REQUIRED_PROXY_DYNAMIC_KEYS = ["path", "portRegex"];


function detectRoutingMode(proxy) {
    const hasTarget = typeof proxy.target === "string" && proxy.target.trim() !== "";
    const hasPortRegex = typeof proxy.portRegex === "string" && proxy.portRegex.trim() !== "";

    return {
        isStatic: hasTarget,
        isDynamic: hasPortRegex
    };
}

function validateProxyEntry(proxy, index) {
    const { isStatic, isDynamic } = detectRoutingMode(proxy);

    // Must be exactly one mode
    if (!isStatic && !isDynamic) {
        console.error(
            `Proxy entry #${index + 1} must define either static routing (${REQUIRED_PROXY_STATIC_KEYS.join(", ")}) ` +
            `or dynamic routing (${REQUIRED_PROXY_DYNAMIC_KEYS.join(", ")}).`
        );
        process.exit(1);
    }
    if (isStatic && isDynamic) {
        console.error(`Proxy entry #${index + 1} cannot define both 'target' and 'portRegex'. Choose static or dynamic routing.`);
        process.exit(1);
    }

    // Validate required keys for static or dynamic routing
    const validateKeys = (proxy, keys, mode, index) =>
    keys.forEach(key => {
        if (!(key in proxy)) {
            console.error(`${mode} proxy entry #${index + 1} is missing required key '${key}'.`);
            process.exit(1);
        }
    });

    if (isStatic) validateKeys(proxy, REQUIRED_PROXY_STATIC_KEYS, "Static", index);
    if (isDynamic) validateKeys(proxy, REQUIRED_PROXY_DYNAMIC_KEYS, "Dynamic", index);

    // Validate regex correctness
    try {
        new RegExp(proxy.path);
        if (proxy.portRegex) new RegExp(proxy.portRegex);
        if (proxy.hostRegex) new RegExp(proxy.hostRegex);
    } catch (err) {
        console.error(`Invalid regex in proxy entry #${index + 1}: ${err.message}`);
        process.exit(1);
    }
}

function validateConfig(config) {
    if (!config.proxies || !Array.isArray(config.proxies)) {
        console.error("The 'proxies' key must be an array in proxy.config.yaml");
        process.exit(1);
    }

    config.proxies.forEach(validateProxyEntry);
}


// Load environment variables
dotenv.config({ path: '.env.local' });
if (process.env.NODE_ENV === 'production') {
    dotenv.config({ path: process.env.EF_CONF_ROOT + '/proxy/env' });
}

// Enable logging
const configuredLogLevelName = String(process.env.PROXY_LOG_LEVEL ?? DEFAULT_LOG_LEVEL).toUpperCase();
const configuredLogLevel = LOG_LEVELS[configuredLogLevelName] ?? LOG_LEVELS[DEFAULT_LOG_LEVEL];
const isLogLevelEnabled = (logLevel) => configuredLogLevel >= logLevel;

function debugLog(...args) {
    if (isLogLevelEnabled(LOG_LEVELS.DEBUG)) {
        console.debug(...args);
    }
}
function infoLog(...args) {
    if (isLogLevelEnabled(LOG_LEVELS.INFO)) {
        console.log(...args);
    }
}

const app = express();

// enable morgan logging middlewere for INFO and above
if (isLogLevelEnabled(LOG_LEVELS.INFO)) {
    // Combined format for production, dev format otherwise
    const format = process.env.NODE_ENV === 'production' ? 'combined' : 'dev';
    app.use(require('morgan')(format));
}

// Security middleware. We have to disable CSP for DCV
app.use(helmet({ contentSecurityPolicy: false }));

// Rate limiting (basic protection against brute-force or flooding)
const limiter = rateLimit({
    windowMs: process.env.PROXY_RATE_LIMIT_WINDOW_MS || DEFAULT_RATE_LIMIT_WINDOW_MS,
    max: process.env.PROXY_RATE_LIMIT_REQUESTS || DEFAULT_RATE_LIMIT_REQUESTS,
});
app.use(limiter);

// Load configuration from YAML (use EF_CONF_ROOT if defined)
const confRoot = process.env.EF_CONF_ROOT || __dirname;
const configPath = path.join(confRoot, 'proxy', 'proxy.config.yaml');

let config;
try {
    config = yaml.load(fs.readFileSync(configPath, 'utf8'));
    validateConfig(config);
} catch (err) {
    console.error('Failed to load proxy configuration:', err);
    process.exit(1);
}

// Create a storage array for the proxies
const configuredProxies = [];

// Iterate over proxies defined in YAML and create middleware
config.proxies.forEach(proxy => {
    try {
        debugLog("Mounting proxy:", proxy.path, "with options:", proxy);

        // Support both dynamic paths with regex and static paths
        const pathRegex = new RegExp(proxy.path);

        // protocol is required to use WS when jupyter starts in HTTP, or to use WSS with DCV.
        const protocol = proxy.protocolRewrite || 'http';

        // Populate proxy options
        const proxyOptions = {
            target: proxy.target, // Undefined for dynamic proxyes
            changeOrigin: true, // rewrites the Host header so the backend sees the request as coming directly to it
            ws: true, // Required for WebSocket communications
            logLevel: proxy.logLevel || configuredLogLevelName.toLowerCase(),
            secure: proxy.secure !== false, // can be false for self-signed certificates
            timeout: proxy.timeout || DEFAULT_TIMEOUT,
            // Set connection timeout (how long to wait for a connection to be established)
            proxyTimeout: proxy.timeout || DEFAULT_TIMEOUT,
            // Rewrite protocol must be set to http for example if Jupyter is started in http and it requires ws:// for the terminal
            protocolRewrite: protocol,
            // Ensure the Host and Protocol headers are correct for the target
            autoRewrite: true,
            pathRewrite: (path, req) => {
                // Use originalUrl if valid (HTTP), fallback to url (WS)
                // Express modifies the req object, stripping the mount path to create req.url and saving the full path in req.originalUrl.
                // WebSockets happen at the network level before Express touches the request.
                // Therefore, req.originalUrl doesn't exist yet, and req.url is the full path.
                const urlToUse = req.originalUrl || req.url;

                if (proxy.rewrite && proxy.rewrite !== false) {
                    // Classic rewrite, according to configuration
                    const regex = new RegExp(proxy.rewrite);
                    const rewrittenPath = urlToUse.replace(regex, '');
                    debugLog(`Rewriting path from ${urlToUse} to ${rewrittenPath}`);
                    return rewrittenPath;
                } else {
                    // no rewrite --> return original url
                    // Note: by default express would have returned req.url without the prefix instead.
                    // Return the full URL to ensure Jupyter receives the prefix
                    debugLog(`Rewriting path to ${urlToUse}`);
                    return urlToUse;
                }
            },
            router: proxy.hostRegex || proxy.portRegex ? (req) => {
                // if host or port regex are configured it is a dynamic routing proxy

                // Use originalUrl if valid (HTTP), fallback to url (WS)
                const urlToUse = req.originalUrl || req.url;
                debugLog(`Router sees URL: ${urlToUse}`);

                const hostRegex = proxy.hostRegex ? new RegExp(proxy.hostRegex) : null;
                const portRegex = proxy.portRegex ? new RegExp(proxy.portRegex) : null;
                let hostname = 'localhost'; // Default to localhost
                let port = null;

                // Extract Port
                if (portRegex) {
                    // Match the port regex against the URL path
                    const portMatch = urlToUse.match(portRegex);
                    if (portMatch && portMatch.length >= 2) {
                        port = portMatch[1];
                    }
                }
                // Extract Hostname (if defined)
                if (hostRegex) {
                    // Match the host regex against the URL path
                    const hostMatch = urlToUse.match(hostRegex);
                    if (hostMatch && hostMatch.length >= 2) {
                        hostname = hostMatch[1];
                    }
                }
                // Determine Target URL
                if (port) {
                    const targetUrl = `${protocol}://${hostname}:${port}`;
                    debugLog(`Dynamic routing: Hostname='${hostname}', Port='${port}' -> ${targetUrl}`);
                    return targetUrl;
                }

                // Failure. If no port is found (or the regexes failed), return nothing to prevent connection.
                debugLog(`Routing failure: Could not find port using portRegex '${proxy.portRegex}' for URL: ${urlToUse}`);
                return;
            } : undefined
        };

        // Create the middleware instance
        const proxyMiddleware = createProxyMiddleware(proxyOptions);
        // Mount to Express
        app.use(pathRegex, proxyMiddleware);

        // Store the middleware and path logic for WebSocket handling
        configuredProxies.push({
            pathRegex: pathRegex,
            proxyConfig: proxy, // keep hostRegex, portRegex, etc.
            middleware: proxyMiddleware
        });

    } catch (err) {
        console.error(`Failed to mount proxy for path ${proxy.path}:`, err);
    }
});

app.use((req, res, next) => {
    infoLog("Incoming request:", req.method, req.url);
    next();
});

// Error handling
app.use((err, req, res, next) => {
    console.error('Unhandled error:', err.stack || err);
    res.status(500).send('Internal Server Error');
});

// Start Express server
const port = process.env.PROXY_PORT || DEFAULT_PROXY_PORT;
const host = process.env.PROXY_HOST || '0.0.0.0';

// Check if key and certificate variables are defined to start the server in HTTPs
const keyPath = process.env.PROXY_PRIVATE_KEY;
const certPath = process.env.PROXY_CERTIFICATE;

let server;
if (keyPath && certPath) {
    // Try to read the key and certificate files
    try {
        const privateKey = fs.readFileSync(keyPath, 'utf8');
        const certificate = fs.readFileSync(certPath, 'utf8');
        const credentials = { key: privateKey, cert: certificate };
        // HTTPS server
        server = https.createServer(credentials, app);
        server.listen(port, host, () => {
            infoLog(`Proxy HTTPS server listening on https://${host}:${port}`);
        });
    } catch (err) {
        console.error('Failed to read SSL files:', err);
        // Fallback to HTTP if reading files fails
        server = http.createServer(app);
        server.listen(port, host, () => {
            infoLog(`Proxy HTTP fallback listening on http://${host}:${port}`);
        });
    }
} else {
    // Start HTTP server if env vars are not set
    server = http.createServer(app);
    server.listen(port, host, () => {
        infoLog(`Proxy HTTP server listening on http://${host}:${port}`);
    });
}

// Web socket handling
server.on('upgrade', (req, socket, head) => {
    debugLog("WebSocket upgrade requested:", req.url);
    debugLog("URL:", req.url);
    debugLog("Headers:", req.headers);

    // Find proxy configuration matching the request URL
    const matchingProxy = configuredProxies.find(cp => {
        return cp.pathRegex.test(req.url);
    });

    if (matchingProxy) {
        // Handshake accepted & proxied
        debugLog("Matched proxy path:", matchingProxy.pathRegex.toString());
        debugLog("Routing WebSocket to proxy middleware");
        // Manually pass the upgrade request to the http-proxy-middleware instance
        // The middleware will add its own listeners and manage the lifecycle.
        matchingProxy.middleware.upgrade(req, socket, head);
    } else {
        // Connection rejected and closed
        debugLog("No matching proxy found for WebSocket upgrade. Destroying socket.");
        socket.destroy();
    }
});

// Graceful shutdown
process.on('SIGINT', () => {
    console.log('Received SIGINT, shutting down...');
    server.close(() => {
        debugLog('Server closed');
        process.exit(0);
    });
});
process.on('SIGTERM', () => {
    console.log('Received SIGTERM, shutting down...');
    server.close(() => {
        debugLog('Server closed');
        process.exit(0);
    });
});
