const fs        = require('fs');
const http      = require('http');
const https     = require('https');
const path      = require('path');
const WebSocket = require('ws');
const express   = require('express');
const pty       = require('node-pty');
const hbs       = require('hbs');
const dotenv    = require('dotenv');
const Tokens    = require('csrf');
const url       = require('url');
const yaml      = require('js-yaml');
const glob      = require('glob');
const helpers   = require('./utils/helpers');
const host_path_rx = /^\/ssh\/([^\/?]+)(\/[^?]*)?(?:\?.*)?$/;


// Read in environment variables
dotenv.config({path: '.env.local'});
if (process.env.NODE_ENV === 'production') {
    dotenv.config({path: process.env.EF_CONF_ROOT + '/shell/env'});
}

// Load color themes
var color_themes = {dark: [], light: []};
glob.sync('./color_themes/light/*').forEach(f => color_themes.light.push(require(path.resolve(f))));
glob.sync('./color_themes/dark/*').forEach(f => color_themes.dark.push(require(path.resolve(f))));
color_themes.json_array = JSON.stringify([...color_themes.light, ...color_themes.dark]);

const tokens = new Tokens({});
const secret = tokens.secretSync();

// Create all your routes
var router = express.Router();
router.get(['/', '/ssh'], function (req, res) {
    res.redirect(req.baseUrl + '/ssh/default');
});

router.get('/ssh*', function (req, res) {
    var theHost, theDir;
    [theHost, theDir] = host_and_dir_from_url(req.url);
    res.render('index',
    {
        baseURI: req.baseUrl,
        csrfToken: tokens.create(secret),
        host: theHost,
        dir: theDir,
        colorThemes: color_themes,
        siteTitle: "EF Portal Remote Shell",
    });
});

router.use(express.static(path.join(__dirname, 'public')));

// Setup app
var app = express();

// Setup template engine
app.set('view engine', 'hbs');
app.set('views', path.join(__dirname, 'views'));

// Mount the routes at the base URI
app.use('/', router);

// Setup websocket server
var server;

// Check if both environment variables are set and not empty
const keyPath = process.env.SHELL_PRIVATE_KEY;
const certPath = process.env.SHELL_CERTIFICATE;

if (keyPath && certPath) {
  // Try to read the key and certificate files
  try {
    const privateKey = fs.readFileSync(keyPath, 'utf8');
    const certificate = fs.readFileSync(certPath, 'utf8');
    const credentials = { key: privateKey, cert: certificate };

    server = https.createServer(credentials, app);
  } catch (err) {
    console.error('Failed to read SSL files:', err);
    // Fallback to HTTP if reading files fails
    server = http.createServer(app);
  }
} else {
  // Start HTTP server if env vars are not set
  server = http.createServer(app);
}

const wss = new WebSocket.Server({ noServer: true });

const inactiveTimeout = (process.env.SHELL_INACTIVE_TIMEOUT_MS || 300000);
const maxShellTime = (process.env.SHELL_MAX_DURATION_MS || 3600000);

let host_allowlist = [];
if (process.env.SHELL_SSHHOST_ALLOWLIST) {
    host_allowlist = Array.from(new Set(process.env.SHELL_SSHHOST_ALLOWLIST.split(':')));
}

let hosts = helpers.definedHosts();
let default_sshhost = hosts['default'];
hosts['hosts'].forEach((host) => {
    host_allowlist.push(host);
});
if (!host_allowlist.includes(default_sshhost)) {
    host_allowlist.push(default_sshhost);
}

function host_and_dir_from_url(url) {
    let match = url.match(host_path_rx),
    hostname = null,
    directory = null;

    if (match) {
        hostname = match[1] === "default" ? default_sshhost : match[1];
        directory = match[2] ? decodeURIComponent(match[2]) : null;
    }
    return [hostname, directory];
}

function custom_server_origin(default_value = null) {
    var custom_origin = null;

    if(process.env.SHELL_ORIGIN_CHECK) {
        // if ENV is set, do not use default!
        if(process.env.SHELL_ORIGIN_CHECK.startsWith('http')) {
            custom_origin = process.env.SHELL_ORIGIN_CHECK;
        }
    } else {
        custom_origin = default_value;
    }

    return custom_origin;
}

function default_server_origin(headers) {
    var origin = null;

    if (headers['x-forwarded-proto'] && headers['x-forwarded-host']) {
        origin = headers['x-forwarded-proto'] + "://" + headers['x-forwarded-host']
    }

    return origin;
}

function detect_auth_error(requestToken, client_origin, server_origin, host) {
    var theHost = host.includes('@') ? host.split('@')[1] : host;
    if (client_origin && client_origin.startsWith('http') && server_origin && client_origin !== server_origin) {
        return "Invalid Origin.";
    } else if (!tokens.verify(secret, requestToken)) {
        return "Bad CSRF Token.";
    } else if (!helpers.hostInAllowList(host_allowlist, theHost)) {
        return `Host "${theHost}" not specified in allowlist or cluster configs.`;
    } else {
        return null;
    }
}

wss.on('connection', function connection (ws, req) {
    var dir,
        term,
        args,
        host,
        cmd = process.env.SHELL_SSH_WRAPPER || 'ssh';
  
    ws.isAlive = true;
    ws.startedAt = Date.now();
    ws.lastActivity = Date.now();
  
    console.log('Connection established');
    
    [host, dir] = host_and_dir_from_url(req.url);
    
    // Verify authentication
    token = req.url.match(/csrf=([^&]*)/)[1];
    authError = detect_auth_error(token, req.origin, custom_server_origin(default_server_origin(req)), host);
    if (authError) {
        // 3146 has no meaning, any number between 3000-3999 is fair to use
        ws.close(3146, authError);
    } else {
        let cmdParts;

        // Extract hostname (remove username@ prefix if present)
        const hostname = host.includes('@') ? host.split('@')[1] : host;
        const user = host.includes('@') ? host.split('@')[0] : '';
        
        if (cmd.includes(' ')) {
            // Split command and replace %NODE% placeholder in all parts
            cmdParts = cmd.split(' ').map(part => part.replace(/%NODE%/g, hostname));
        } else {
            // Single command, just replace %NODE% placeholder
            cmdParts = [cmd.replace(/%NODE%/g, hostname)];
        }
        
        // Extract the main command and its arguments
        let mainCmd = cmdParts[0];  // Changed const to let
        let args = cmdParts.slice(1);
        
        // For commands without spaces, add the hostname at the end, host = user@server 
        if (!cmd.includes(' ')) {
            if (dir) {
                args = args.concat([host, '-t', 'cd \'' + dir.replace(/\'/g, "'\\''") + '\' ; exec ${SHELL} -l']);
            } else {
                args = args.concat([host]);
            }
        } else {
            // For commands with spaces, wrap in SSH
            const execParam = args.join(' ');
            mainCmd = "ssh";
            modifiedHost = host.includes('@') ? `${user}@localhost` : 'localhost';
            args = [modifiedHost, "-t", cmdParts.join(' ')];  // Rebuild args array properly
        }
        
        console.log(mainCmd); console.log(args);
        
        process.env.LANG = 'en_US.UTF-8';
        
        term = pty.spawn(mainCmd, args, {
            name: 'xterm-16color',
            cols: 80,
            rows: 30
        });
        
        console.log('Opened terminal: ' + term.pid);

        term.onData(function (data) {
            ws.send(data, function (error) {
                if (error) console.log('Send error: ' + error.message);
           });
           ws.lastActivity = Date.now();
        });

        term.onExit(function (_exitData) {
            ws.close();
        });

        ws.on('message', function (msg) {
            msg = JSON.parse(msg);
            if (msg.input)  {
                term.write(msg.input);
                this.lastActivity = Date.now();
            }
            if (msg.resize) term.resize(parseInt(msg.resize.cols), parseInt(msg.resize.rows));
        });

        ws.on('close', function () {
            term.end();
            this.isAlive = false;
            console.log('Closed terminal: ' + term.pid);
        });

        ws.on('pong', function () {
            this.isAlive = true;
        });
    }
});

const interval = setInterval(function ping() {
    wss.clients.forEach(function each(ws) {
        const timeUsed = Date.now() - ws.startedAt;
        const inactiveFor = Date.now() - ws.lastActivity;
        if (ws.isAlive === false || inactiveFor > inactiveTimeout || timeUsed > maxShellTime) {
            return ws.terminate();
        }

        ws.isAlive = false;
        ws.ping();
    });
}, 30000);

wss.on('close', function close() {
    clearInterval(interval);
});

server.on('upgrade', function upgrade(request, socket, head) {
    wss.handleUpgrade(request, socket, head, function done(ws) {
        wss.emit('connection', ws, request);
    });
});

const port = (process.env.SHELL_PORT || 3000);
server.listen(port, '0.0.0.0', function () {
    console.log('Listening on ' + port);
});

// Graceful shutdown
process.on('SIGINT', () => {
    console.log('Received SIGINT, shutting down...');
    server.close(() => {
        console.log('Server closed');
        process.exit(0);
    });
});

process.on('SIGTERM', () => {
    console.log('Received SIGTERM, shutting down...');
    server.close(() => {
        console.log('Server closed');
        process.exit(0);
    });
});
