#!/bin/bash

################################################################################
################################################################################
# Copyright 2023-2025 by NI SP Software GmbH, All rights reserved.
# Copyright 1999-2023 by Nice, srl., All rights reserved.
#
# This software includes confidential and proprietary information
# of NI SP Software GmbH ("Confidential Information").
# You shall not disclose such Confidential Information
# and shall use it only in accordance with the terms of
# the license agreement you entered into with NI SP Software.
################################################################################
################################################################################
################################################################################
# $EF_ROOT/plugins/lsf/bin/functions.lib.sh - common functions for LSF Plugin
################################################################################


################################################################################
# Backward-Compatibility settings/functions
################################################################################
#-------------------------------------------------------------------------------
# xml_start()
#-------------------------------------------------------------------------------
xml_start() {
  TAG="$1"
  [ -z "${TAG}" ] && TAG="ef:output"
  if [ -z "${EF_FIRST_TAG}" ] ; then
    EF_FIRST_TAG="${TAG}"
    export EF_FIRST_TAG
    echo "<${TAG} ${EF_XMLNS_ef}>"
  fi
}

#-------------------------------------------------------------------------------
# xml_end()
#-------------------------------------------------------------------------------
xml_end() {
  echo "</${EF_FIRST_TAG}>"
  exit 1
}

#-------------------------------------------------------------------------------
# warning()
#-------------------------------------------------------------------------------
warning() {
  ef_error "${1}" "${2}"
}

# This variable was set in old common, don't know why
EF_AGENT="com.enginframe.lsf"
################################################################################
# end of Backward-Compatibility Section
################################################################################


################################################################################
# lsf common functions - these are the real ones
################################################################################
ef_bjobs2xml() {

[ "${USE_PROJECT_FOR_USER}" = 'false' ] && PROJECT_FILTER=""
export PROJECT_FILTER

BJOBS_TEMPFILE=`ef_create_tempfile`
if [ "$?" -ne 0 ]; then
    echo '<ef:error>'
    echo "  <ef:title><![CDATA[${EF_PLUGIN_NAME} Error]]></ef:title>"
    echo "  <ef:message><![CDATA[Cannot create a temporary file]]></ef:message>"
    echo "  <ef:command><![CDATA[${EF_TEMPFILE_ERROR}]]></ef:command>"
    echo '</ef:error>'
    exit 1
fi

(
echo "Year	`date '+%Y	%m%d'`"

if [ "${LSF_VERSION_MAJOR}" -ge 8 ]; then
    export LSB_DISPLAY_YEAR='n'
fi

LSB_SHORT_HOSTLIST=0 ${EF_LSF_BJOBS} ${PROJECT_FILTER} -l -UF $* \
2>"${BJOBS_TEMPFILE}" \
| ${EF_AWK} -f "${EF_ROOT}/plugins/lsf/bin/bjobs.fix.awk" \
| sed 's/, *Copy Files[^,]*,/,/;s/></> </g;s/>>/> >/g;s/>>*\(;\|,\|;,\)/\
/g' \
| sed 's/^[ ]*//;s/[ ]*</	/;s/>;//g;s/>  *</	/g'
) \
| ${EF_AWK} \
  -v GRIDML_COMPRESSION="${EF_GRIDML_COMPRESSION}" \
  -v COMPRESS_ARRAY="${COMPRESS_JOB_ARRAY}" \
  -v CLUSTER_ID="${LSF_CLUSTER_ID}" \
  -v EF_XMLNS_grid="${EF_XMLNS_grid}" \
  -v GRID_TAG_SIZE_LIMIT="${GRID_TAG_SIZE_LIMIT}" \
  -v "statusmap=${EF_ROOT}/plugins/lsf/conf/grid.job.status.mapping" \
  -f "${EF_ROOT}/plugins/ef/lib/awk/utils.awk" \
  -f "${EF_ROOT}/plugins/lsf/bin/bjobs.parse.awk"

  LSF2XML_BJOBS_AWKEXITCODE=$?

  LSF2XML_BJOBS_STDERR=`cat "${BJOBS_TEMPFILE}"`
  rm -f "${BJOBS_TEMPFILE}"

  # If there is something on standard error...
  if [ -n "${LSF2XML_BJOBS_STDERR}" ]; then

    # Check if the error is "Job <123> is not found" --> exit with error
    echo "${LSF2XML_BJOBS_STDERR}" | grep -e 'Job .* not .*found' >/dev/null
    LSF2XML_MISSING_JOB="$?"

    # Check if the error is "No jobs found" --> exit silently
    echo "${LSF2XML_BJOBS_STDERR}" | grep -e '[nN]o .*found' >/dev/null
    LSF2XML_NO_JOB="$?"

    if [ "${LSF2XML_MISSING_JOB}" = 0 -a "${LSF2XML_BJOBS_AWKEXITCODE}" = "1" ]; then
        ef_error "${LSF2XML_BJOBS_STDERR}" "EnginFrame LSF Plugin Error" "${EF_LSF_BJOBS}"
        exit 1
    fi

    if [ "${LSF2XML_NO_JOB}" = 0 ]; then
      # return empty tag
      echo "<grid:job-list type=\"lsf\" ${EF_XMLNS_grid} />"
      exit 0
    fi

    if [ ! "${LSF2XML_MISSING_JOB}" = 0 ]; then
      # In any other situations --> exit with error
      ef_error "${LSF2XML_BJOBS_STDERR}" "EnginFrame LSF Plugin Error" "${EF_LSF_BJOBS}"
      exit 1
    fi
  fi

  if [ -z "${sort_order}" ]; then
    cat <<EOF
      <ef:session>
        <ef:option id="sort_order">descending</ef:option>
      </ef:session>
EOF
  fi

  if [ "${_sortby}" = "${job_sortby}" -a -n "${job_sortby}" ]; then
    if [ "${sort_order}" = "ascending" ]; then
      cat <<EOF
        <ef:session>
          <ef:option id="sort_order">descending</ef:option>
        </ef:session>
EOF
      sort_order="descending"
    else
      cat <<EOF
        <ef:session>
          <ef:option id="sort_order">ascending</ef:option>
        </ef:session>
EOF
      sort_order="ascending"
    fi
  else
    sort_order="ascending"

    if [ ! "x${_sortby}" = "x" ]; then
      cat <<EOF
        <ef:session>
          <ef:option id="sort_order">ascending</ef:option>
        </ef:session>
        <ef:session>
          <ef:option id="job_sortby">$(ef_xml_escape_content -i "${_sortby}")</ef:option>
        </ef:session>
EOF
    fi
  fi
}


ef_array_2xml() {
[ "x${USE_PROJECT_FOR_USER}" = "xfalse" ] && PROJECT_FILTER=""
export PROJECT_FILTER

(
echo "Year	`date '+%Y	%m%d'`"

if [ "${LSF_VERSION_MAJOR}" -ge 8 ]; then
    export LSB_DISPLAY_YEAR='n'
fi

LSB_SHORT_HOSTLIST=0 ${EF_LSF_BJOBS} ${PROJECT_FILTER} -l -A -UF $* 2>/dev/null \
| ${EF_AWK} \
    -f "${EF_ROOT}/plugins/lsf/bin/bjobs.fix-array.awk" \
| sed 's/></> </g;s/>>*\(;\|,\|;,\)/\
/g' \
| sed 's/^[ ]*//;s/[ ]*</	/;s/>;//g;s/>  *</	/g;s/^Re-runnable;//g'
) \
| ${EF_AWK} \
    -v GRIDML_COMPRESSION="${EF_GRIDML_COMPRESSION}" \
    -v CLUSTER_ID="${LSF_CLUSTER_ID}" \
    -v EF_XMLNS_grid="${EF_XMLNS_grid}" \
    -v GRID_TAG_SIZE_LIMIT="${GRID_TAG_SIZE_LIMIT}" \
    -v "statusmap=${EF_ROOT}/plugins/lsf/conf/grid.job.status.mapping" \
    -f "${EF_ROOT}/plugins/ef/lib/awk/utils.awk" \
    -f "${EF_ROOT}/plugins/lsf/bin/bjobs.parse-array.awk"
}

# parse bsub output to get job id
ef_getjobid() {
    [ -n "${1}" ] && echo "${1}" | sed -n 's/^Job <\([0-9][0-9]*\)> is submitted.*$/\1/p'
}

ef_jobspooler() {
  _job="$1"

  # First try with bread
  unset _myURI

  if [ -n "${EF_LSF_BREAD}" ]; then
    _breadfile="/tmp/.ef$$"
    _breadfile2="${_breadfile}"
    ${EF_LSF_BREAD} -a "${_breadfile2}" "${_job}" > /dev/null 2>/dev/null
    if [ -f "${_breadfile}" ] ; then
      _myURI=`cat "${_breadfile}"`
    fi
    rm -f "${_breadfile}"
  fi

  # This branch means that bread does not work (so we probably have LAVA, not LSF)
  if [ -z "${_myURI}" ] ; then
    _myURI=`ef_bjobs2xml "$@" "${_job}"`
    [ "$?" -ne 0 ] && return
    _myURI=`echo "${_myURI}" | sed -n 's/^ *<grid:submission-directory>\(.*\)<\/grid:submission-directory>$/spooler:\/\/\1/p' | sed -n '1p'`
    _myURI=`eval echo "${_myURI}"`
  fi

  # This branch means that even <grid:submission-directory> does not work,
  # so we try with $URI, if it is defined
  if [ -z "${_myURI}" -a -n "${URI}" ] ; then
    _myURI="${URI}"
  fi

  case "${_myURI}" in
      *tmp*.ef)
        echo "${_myURI}"
        ;;
  esac
}

ef_jobinfo() {
  [ -n "$_message" ] && echo "<grid:message ${EF_XMLNS_grid}>$_message</grid:message>"

  if [ "${COMPRESS_JOB_ARRAY}" = "true" ]; then
    ef_array_2xml $* ${_jobid} 2>/dev/null
    COMPRESS_JOB_ARRAY="false"
    ef_bjobs2xml $* ${_jobid} 2>/dev/null
  else
    ef_bjobs2xml $* ${_jobid} 2>/dev/null
  fi
}

ef_jobdetails() {
  ef_jobinfo $*

  # Let's see if there is a spooler associated to this job
  _myURI=`ef_jobspooler ${_jobid}`

  # Eventually, if we get some value of $_myURI then show the spooler
  if [ -n "${_myURI}" ] ; then
    URI="${_myURI}"
    DIR=`spooler2file "${_myURI}"`
    if [ -d "${DIR}" ] ; then
      ${EF_ROOT}/plugins/ef/bin/ef.show.spooler "${URI}"
    fi
  fi

}

ef_killjob() {
  _message="`${EF_LSF_BKILL} ${_jobid} | ef_xml_escape -p 2>&1`"
  export _message
  ef_jobdetails
}

ef_switchjob() {
  if [ "x${USE_PROJECT_FOR_USER}" = "xtrue" ]
  then
    JOB_IS_MINE="`${EF_LSF_BJOBS} -a -P ${EF_USER} | grep ${_jobid}`"
    if [ "x${JOB_IS_MINE}" = "x" ]
    then
      _message="You can only modify your jobs"
      export _message
      ef_jobdetails
      exit
    fi
  fi
  _message="`${EF_LSF_BSWITCH} $_queue ${_jobid} | ef_xml_escape -p 2>&1`"
  export _message
  ef_jobdetails
}

ef_rename() {
  if [ "x${USE_PROJECT_FOR_USER}" = "xtrue" ]
  then
    JOB_IS_MINE="`${EF_LSF_BJOBS} -a -P ${EF_USER} | grep ${_jobid}`"
    if [ "x${JOB_IS_MINE}" = "x" ]
    then
      _message="You can only modify your jobs"
      export _message
      ef_jobdetails
      exit
    fi
  fi
  _message="`${EF_LSF_BMOD} -J \"$_name\" ${_jobid} | ef_xml_escape -p 2>&1`"
  export _message
  ef_jobdetails
}

ef_changestatus() {
  if [ "x${USE_PROJECT_FOR_USER}" = "xtrue" ]
  then
    JOB_IS_MINE="`${EF_LSF_BJOBS} -a -P ${EF_USER} | grep ${_jobid}`"
    if [ "x${JOB_IS_MINE}" = "x" ]
    then
      _message="You can only modify your jobs"
      export _message
      ef_jobdetails
      exit
    fi
  fi

  case "$_action" in
    kill) result="`${EF_LSF_BKILL} ${_jobid} 2>&1`" ;;
    stop) result="`${EF_LSF_BSTOP} ${_jobid} 2>&1`" ;;
    resume) result="`${EF_LSF_BRESUME} ${_jobid} 2>&1`" ;;
    *) result="Bad status requested"
  esac

  _message="`echo "$result" | ef_xml_escape -p`"
  export _message
  ef_jobdetails
}

# Job list (my or all)
ef_listjobs() {
  if [ "${COMPRESS_JOB_ARRAY}" = "true" -a -z "${host}" ]; then
    ef_bjobs2xml $* 2>&1
    ef_array_2xml $* 2>&1
  else
    [ -n "${host}" ] && COMPRESS_JOB_ARRAY=false
    ef_bjobs2xml $* 2>&1
  fi
}


jobname_sanitize() {
  local -- _jobname="${1:-$(</dev/stdin)}"

  # LSF Should not need any sanification. We however remove the "\r\n" which break the bjobs output.
  # LSF uses square brackets to label job arrays and percent sign to specify job slot limit: "job_name[index_list]%job_slot_limit".
  printf "%s" "${_jobname}" | tr '\n\r[]%' _
}


#
# vi: ts=4 sw=4 et syntax=sh
#

