#!/bin/sh

################################################################################
################################################################################
# Copyright 2023-2025 by NI SP Software GmbH, All rights reserved.
# Copyright 1999-2023 by Nice, srl., All rights reserved.
#
# This software includes confidential and proprietary information
# of NI SP Software GmbH ("Confidential Information").
# You shall not disclose such Confidential Information
# and shall use it only in accordance with the terms of
# the license agreement you entered into with NI SP Software.
################################################################################
################################################################################
################################################################################

################################################################################
# CVS $Id: test_checkpassword.sh 21429 2011-09-09 10:22:30Z robby $
################################################################################

#-------------------------------------------------------------------------------
# This script check if EnginFrame can authenticate users using the PAM
# authentication system.
#
# It asks four questions about:
#   - EF_ROOT     the root directory of EnginFrame software
#   - EF_AUTH     the user who performs PAM authentication
#   - EF_USERNAME the account to be tested
#   - EF_PASSWORD the password to be tested
#
# EF_ROOT must have at least these files/directories:
# $EF_ROOT/
# `-- plugins/
#     `-- pam/
#         |-- bin/
#         |   `-- ef.auth
#         `-- lib/
#             `-- checkpassword-pam.`uname` or checkpassword-pam.`uname`.`uname -m`
#
# EF_AUTH is usually EF_ADMIN
#
# This script prints out one of the following possible message:
#   - grant
#   - deny
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# trap signals
# A special function "ef_trap" is executed when the scripts receives a signal
#-------------------------------------------------------------------------------
trap ef_trap HUP INT QUIT TERM XCPU

#-------------------------------------------------------------------------------
# the "main" routine
#-------------------------------------------------------------------------------
ef_main(){

  # initialize the script
  ef_init

  _ef_root="$(get_ef_root)"
  # ask EF_ROOT, the location of NICE software.
  echo "${efmsg_asknice}"
  while : ; do
    printf "EF_ROOT (${_ef_root}): "
    read EF_ROOT
    [ -z "${EF_ROOT}" ] && EF_ROOT=${_ef_root}
  # check if EF_ROOT contains a valid EnginFrame
    ef_check_nice && break
  done
  EF_CONF_ROOT="$(get_ef_conf_root)"

  # ask EF_AUTH, the user who checks password
  echo "${efmsg_askefauth}"
  while : ; do
    printf "EF_AUTH: "
    read EF_AUTH
    # check if EF_AUTH is a valid user
    ef_check_user -auth "${EF_AUTH}" && break
  done

  # ask a valid username
  # note that stty -echo will not "echo" the password on the terminal
  echo "${efmsg_askaccount}"
  while : ; do
    printf "Username: "
    read EF_USERNAME
    # check if EF_USERNAME is a valid user
    ef_check_user "${EF_USERNAME}" && break
  done

  # ask the corresponding password
  while : ; do
    printf "Password: "
    stty -echo
    read EF_PASSWORD
    stty echo
    printf "\n"
    [ -n "${EF_PASSWORD}" ] && break
  done

  # do the real test
  echo "${efmsg_pleasewait}"

  printf '%s\0%s\0' "${EF_USERNAME}" "${EF_PASSWORD}" | \
  su - "${EF_AUTH}" -c "/bin/sh -c '\
    export EF_ROOT=\"${EF_ROOT}\" && \
    export EF_CONF_ROOT=\"${EF_CONF_ROOT}\" && \
    export EF_USER=\"${EF_AUTH}\" && \
    \"${EF_ROOT}/plugins/pam/bin/ef.auth\" 2>&1'" >/dev/null 2>&1

  _result="$?"

  case "${_result}" in
    "0") echo "${efmsg_ok}" ;;
      *) echo "${efmsg_notok} \"${_result}\"" ;;
  esac

}

#-------------------------------------------------------------------------------
# Get the EF_ROOT inferred from the invocation of the script
# Assumption: this script is under ${EF_ROOT}/tools
#-------------------------------------------------------------------------------
get_ef_root(){
  pushd "$(dirname $0)/.." > /dev/null
  _ef_root=$(pwd)
  popd > /dev/null

  echo "${_ef_root}"
}

#-------------------------------------------------------------------------------
# Get the EF_CONF_ROOT inferred from the value of EF_ROOT
# Assumption: EF_CONF_ROOT=${EF_ROOT}/../../conf  ==>  ${EF_TOP}/conf
#-------------------------------------------------------------------------------
get_ef_conf_root(){
  [ -z "${EF_ROOT}" ] && return 1
  pushd "${EF_ROOT}/../../conf"  > /dev/null
  _ef_conf_root=$(pwd)
  popd > /dev/null

  echo "${_ef_conf_root}"
}

#-------------------------------------------------------------------------------
# check if EF_ROOT directory is OK
#-------------------------------------------------------------------------------
ef_check_nice(){


  # check empty string
  [ -z "${EF_ROOT}" ] && return 1

  # check if directory exists
  if [ ! -d "${EF_ROOT}" ] ; then
    echo "${efmsg_invaliddirectory}"
    EF_ROOT=""
    return 1
  fi

  # find checkpassword for your architecture
  EF_CHECKPWD="${EF_ROOT}/plugins/pam/lib/checkpassword-pam.`uname`.`uname -m`"
  if [ ! -f "${EF_CHECKPWD}" ] ; then
    EF_CHECKPWD="${EF_ROOT}/plugins/pam/lib/checkpassword-pam.`uname`"
  fi

  # if there is no checkpassword for this architecture, exit
  if [ ! -f "${EF_CHECKPWD}" ] ; then
    echo "${efmsg_nocheckpassword}"
    EF_ROOT=""
    return 1
  fi

  # check file permissions
  _output="`ls -l \"${EF_CHECKPWD}\"`"
  _suid="`echo \"${_output}\" | awk '{print substr(\$0,4,1)}'`"
  if [ ! "${_suid}" = "s" ] ; then
    echo "${efmsg_nosuidbit}"
    EF_ROOT=""
    return 1
  fi

  return 0
}

#-------------------------------------------------------------------------------
# check if a user is valid
#-------------------------------------------------------------------------------
ef_check_user(){

  case "$1" in
    "-"*) _flag="$1"; shift ;;
  esac

  _user="$1"

  # check empty string
  [ -z "${_user}" ] && return 1

  # check if user is root
  if [ "${_user}" = "root" ] ; then
    echo "${efmsg_noroot}"
    return 1
  fi

  # check if user exists
  _output="`id \"${_user}\" 2>&1`"
  if [ ! "$?" = "0" ] ; then
    echo "ERROR: ${_output}"
    echo "${efmsg_invaliduser}"
    return 1
  fi

  # check if user can log into the system
  _output="`su - \"${_user}\" -c \"/bin/sh -c 'exit'\" </dev/null 2>&1`"
  if [ ! "$?" = "0" ] ; then
    echo "ERROR: ${_output}"
    echo "${efmsg_disableduser}"
    return 1
  fi
  if [ -n "${_output}" ] ; then
    echo "${efmsg_profileerrors}"
    echo "${_output}
"
  fi

  # check if user can execute checkpassword, if _flag="efauth"
  if [ "${_flag}" = "-auth" ] ; then
    su - "${_user}" -c "/bin/sh -c 'test -x \"${EF_CHECKPWD}\"'" \
      </dev/null >/dev/null 2>&1
    if [ ! "$?" = "0" ] ; then
      echo "${efmsg_usercannotexecute}"
      return 1
    fi
  fi

  return 0
}

#-------------------------------------------------------------------------------
# signal handler routine
#-------------------------------------------------------------------------------
ef_trap(){
  echo ""
  echo "${efmsg_trapped}"
  stty sane
  stty echo
  exit 1
}

#-------------------------------------------------------------------------------
# initialize the script
#-------------------------------------------------------------------------------
ef_init(){

  # welcome message
  efmsg_welcome="
#-------------------------------------------------------------------------------
# EnginFrame PAM Authentication Checker
#
# Note: you can stop this test by pressing CTRL+C
#-------------------------------------------------------------------------------"

  # trap message
  efmsg_trapped="
#-------------------------------------------------------------------------------
# This test has been interrupted
#-------------------------------------------------------------------------------"

  # the mother of all directories
  efmsg_pleasewait="
--- Checking password ----------------------------------------------------------"

  # ok message
  efmsg_ok="OK - EnginFrame can authenticate with these settings
"
  # error message
  efmsg_notok="ERROR - EnginFrame CANNOT authenticate with these settings.
Exit code: "

  # the mother of all directories
  efmsg_asknice="
--- Installation Directory -----------------------------------------------------
Please, specify the full path to the EnginFrame installation directory
"

  # the user who performs authentication
  efmsg_askefauth="
--- Authentication Account -----------------------------------------------------
Please, specify the user in charge of checking other users' credentials.
This username MUST be a valid user of your system.
This username MUST be enabled, MUST have a HOME and a valid login shell.
You CANNOT use root
"

  # the user to be tested
  efmsg_askaccount="
--- Operative System User ------------------------------------------------------
Please specify an existing account of your system that can be used to
check if EnginFrame is able to authenticate Operative System Users.
This username MUST be a valid user of your system.
This username MUST be enabled, MUST have a HOME and a valid login shell.
You CANNOT use root
"

  # you are not root
  efmsg_youarenotroot="
### WARNING ####################################################################
### You MUST be root in order to check user credentials
################################################################################"

  # directory does not exist
  efmsg_invaliddirectory="
### WARNING ####################################################################
### The directory you've specified does not exist
################################################################################"

  # exit if there is no valid checkpassword
  efmsg_nocheckpassword="
### WARNING ####################################################################
### The directory you've specified does not contain a checkpassword executable
### valid for your architecture: `uname` on `uname -m`
################################################################################"

  # checkpassword has no suid bit set
  efmsg_nosuidbit="
### WARNING ####################################################################
### Something is wrong with your installation of EnginFrame
### checkpassword file MUST have the suid bit set, but it is NOT
################################################################################"

  # you cannot use root
  efmsg_noroot="
### WARNING ####################################################################
### The user you've specified CANNOT be root
################################################################################"

  # user does not exist
  efmsg_invaliduser="
### WARNING ####################################################################
### The user you've specified does not exist
################################################################################"

  # user is disabled
  efmsg_disableduser="
### WARNING ####################################################################
### The user you've specified is disabled
################################################################################"

  # profile errors during user logon check
  efmsg_profileerrors="
WARNING: testing execution of an user shell produced messages on standard
  error, probably due to errors during user profile loading. Test checkpassword
  tool will continue, but these errors will probably lead to invalid results
  on Enginframe services that have to produce a valid XML output.
  Standard error messages follow: "

  # user cannot execute checkpassword
  efmsg_usercannotexecute="
### WARNING ####################################################################
### The user you've specified is not authorize to execute checkpassword
################################################################################"

  # Welcome message
  echo "${efmsg_welcome}"

  # reset variables
  EF_ROOT=""
  EF_AUTH=""
  EF_USER=""
  EF_USERNAME=""
  EF_PASSWORD=""
  EF_CHECKPWD=""

  # get actual username
  EF_USER="`id | sed 's/[^(]*(\([^)]*\)).*/\1/'`"

  # if you are not root, you cannot check passwords
  if [ ! "${EF_USER}" = "root" ] ; then
    echo "${efmsg_youarenotroot}"
    exit 1
  fi

}

#-------------------------------------------------------------------------------
# call the main routine
#-------------------------------------------------------------------------------
ef_main "$@"
