/*global jQuery, document, window, parent, setTimeout, clearTimeout, console, Tour */

////////////////////////////////////////////////////////////////////////////////
// Copyright 2023-2025 by NI SP Software GmbH, All rights reserved.
// Copyright 1999-2023 by Nice, srl., All rights reserved.
//
// This software includes confidential and proprietary information
// of NI SP Software GmbH ("Confidential Information").
// You shall not disclose such Confidential Information
// and shall use it only in accordance with the terms of
// the license agreement you entered into with NI SP Software.
////////////////////////////////////////////////////////////////////////////////

var ide = {
    debug: false,
    div: '#ef-srv-ide',
    contentDiv: "#ef-srv-ide-service-editor-content",
    plugin: '',
    sdf: '',
    adminUri: '',
    tour : '',
    serviceManagerUri: '//com.enginframe.service-manager/',

    service: {
        id: '',
        type: '',
        templateId: '',
        ideVersion: '',
        name: {
            value: 'New Service Name',
            set: function (value) {
                jQuery('.ef-service-title').text("Services: " + value).change();
                this.value = value;
                if (!value) {
                    ide.messages.alert(ide.messages.primary, "Service Name cannot be empty.");
                }
            }
        },
        rfbList: {
            value: {},
            update: function () {
                var list = {},
                    isSame;

                list = jQuery(ide.service.components.core.div).efserviceeditor('listOptions', 'rfb');
                isSame = (JSON.stringify(list) === JSON.stringify(this.value));

                this.value = list;
                return !isSame;
            },
            getOptionId: function (htmlId) {
                return jQuery(ide.service.components.core.div).efserviceeditor('getOptionId', htmlId);
            },
            getHtmlId: function (optionId) {
                return jQuery(ide.service.components.core.div).efserviceeditor('getOptionHtmlId', optionId);
            }
        },
        lock: {
            byAnother: false,
            isSet: false,
            timer: 0,
            set: function () {
                jQuery.hydrogen.invokeService({
                    sdf: ide.sdf,
                    uri: ide.serviceManagerUri + 'lock.service',
                    data: {
                        plugin: ide.plugin,
                        serviceId: ide.service.id,
                        editorId: ide.editorId
                    },
                    messagebox: ide.messages.getMessageBox(ide.messages.secondary),
                    success: function () {
                        ide.service.lock.byAnother = false;
                    },
                    error: function () {
                        ide.buttons.disable(ide.buttons.testService);
                        ide.buttons.disable(ide.buttons.restoreService);
                        ide.buttons.disable(ide.buttons.saveService);
                    }
                });
                clearTimeout(this.timer);
                this.timer = setTimeout(ide.service.lock.set, 30000);
                this.isSet = true;
            },
            unset: function (callback) {
                if (ide.service.id && this.isSet) {
                    clearTimeout(ide.service.lock.timer);
                    jQuery.hydrogen.invokeService({
                        sdf: ide.sdf,
                        uri: ide.serviceManagerUri + 'unlock.service',
                        data: {
                            plugin: ide.plugin,
                            serviceId: ide.service.id,
                            editorId: ide.editorId
                        },
                        success: function () {
                            if (callback && typeof callback === 'function') {
                                callback();
                            }
                        },
                        error: function () {
                            if (callback && typeof callback === 'function') {
                                callback();
                            }
                        }
                    });
                    this.isSet = false;
                }
            },
            check: function () {
                if (ide.service.id !== '') {
                    jQuery.hydrogen.invokeService({
                        sdf: ide.sdf,
                        uri: ide.serviceManagerUri + 'is.service.locked',
                        data: {
                            plugin: ide.plugin,
                            serviceType: ide.service.type,
                            serviceId: ide.service.id,
                            editorId: ide.editorId
                        },
                        success: function (xml) {
                            var isServiceLocked = jQuery(xml).find('ef\\:result, result').text();
                            if (isServiceLocked === "false") {
                                ide.service.lock.byAnother = false;
                                ide.service.lock.set();
                            }
                            else {
                                ide.service.lock.byAnother = true;
                                ide.messages.alert(ide.messages.secondary, "The Service is locked for editing by an Admin.");
                                ide.buttons.disable(ide.buttons.testService);
                                ide.buttons.disable(ide.buttons.restoreService);
                                ide.buttons.disable(ide.buttons.saveService);
                            }
                        }
                    });
                }
            }
        },
        components: {
            core: { name: 'Service Core', div: '#ef-srv-ide-service-editor-content', type: 'efserviceeditor', mode: 'all', plugin: 'all', savedValue: '' },
            conf: { name: 'Configuration Parameter', div: '#ef-srv-ide-conf-editor > .handsontable', type: 'conf', mode: 'all', plugin: 'all', savedValue: '' },
            hiddenAttr: { name: 'Hidden Attribute', div: '#ef-srv-ide-service-attributes-hidden', type: 'boolean', mode: 'all', plugin: 'all', savedValue: '' },
            embeddableAttr: { name: 'Embeddable Attribute', div: '#ef-srv-ide-service-attributes-embeddable', type: 'boolean', mode: 'batch', plugin: 'all', savedValue: '' },
            resubmitAttr: { name: 'Resubmit Attribute', div: '#ef-srv-ide-service-attributes-service-resubmit', type: 'boolean', mode: 'all', plugin: 'applications', savedValue: '' },
            reuseSpoolerAttr: { name: 'Reuse Spooler Attribute', div: '#ef-srv-ide-service-attributes-reuse-spooler', type: 'boolean', mode: 'all', plugin: 'applications', savedValue: '' },
            resetTTLAttr: { name: 'Reset TTL Attribute', div: '#ef-srv-ide-service-attributes-reset-ttl', type: 'boolean', mode: 'all', plugin: 'applications', savedValue: '' },
            profile: { name: 'Service Profile', div: '#ef-srv-ide-service-attributes-service-profile', type: 'boolean', mode: 'all', plugin: 'all', savedValue: '' },
            spoolerTTL: { name: 'Spooler TTL', div: '#ef-srv-ide-service-attributes-ttl-slider-value', type: 'ttl', mode: 'all', plugin: 'applications', savedValue: '' },
            sessionClass: { name: 'Session Class', div: '#ef-srv-ide-service-attributes-session-class', type: 'string', mode: 'interactive', plugin: 'all', savedValue: '',
                validator: { pattern: '^([\\w]+[\\w\\-\\.]*)?$', message: 'must start with alphanumeric character and must contain only others alphanumeric, dashes (-), periods (.) and underscores (_)' }
            },
            maxSessions: { name: 'Max Number of Sessions', div: '#ef-srv-ide-service-attributes-max-sessions', type: 'string', mode: 'interactive', plugin: 'all', savedValue: '',
                validator: { pattern: '^(?!0+$)\\d*$', message: 'must be a valid positive number' }
            },
            metadata: { name: 'Service Metadata', div: '#ef-srv-ide-attrs-metadata > .handsontable', type: 'metadata', mode: 'all', plugin: 'all', savedValue: '' },
            serviceJs: { name: 'Custom JavaScript', div: '#ef-srv-ide-service-js-editor-content', type: 'texteditor', mode: 'all', plugin: 'all', savedValue: '' },
            serviceCss: { name: 'Custom CSS', div: '#ef-srv-ide-service-css-editor-content', type: 'texteditor', mode: 'all', plugin: 'all', savedValue: '' },
            fileAction: {
                enabled: { name: "File Action Attribute", div: '#ef-srv-ide-file-action-enabled', type: 'boolean', mode: 'all', plugin: 'applications', savedValue: '' },
                label: { name: "File Action Label", div: '#ef-srv-ide-file-action-label', type: 'string', mode: 'all', plugin: 'applications', savedValue: '' },
                optionId: { name: "File Action RFB Option Id", div: '#ef-srv-ide-file-action-option-id', type: 'select', mode: 'all', plugin: 'applications', savedValue: '' },
                fileFilter: { name: "File Action File Filter", div: '#ef-srv-ide-file-action-file-filter', type: 'string', mode: 'all', plugin: 'applications', savedValue: '' },
                serviceExecution: { name: 'File Action Service Execution', div: 'input[name=ef-srv-ide-file-action-service-execution]', type: 'radio', mode: 'all', plugin: 'applications', savedValue: '' },
                showMenuOnly: { name: "File Action Show Menu Only", div: '#ef-srv-ide-file-action-show-menu-only', type: 'boolean', mode: 'all', plugin: 'applications', savedValue: '' }
            },
            action: {
                id: { name: 'Action Id', div: '#ef-srv-ide-action-id', type: 'string', mode: 'all', plugin: 'all', savedValue: '',
                    validator: { pattern: '^([\\w]+[\\w\\-\\.]*)?$', message: 'must start with alphanumeric character and must contain only others alphanumeric, dashes (-), periods (.) and underscores (_)' }
                },
                label: { name: 'Action Button Label', div: '#ef-srv-ide-action-button-label', type: 'string', mode: 'all', plugin: 'all', savedValue: '',
                    validator: { pattern: '[\\S ]+', message: 'cannot be empty' }
                },
                resultType: { name: 'Action Result Type', div: '#ef-srv-ide-action-result-type-select', type: 'select', mode: 'all', plugin: 'all', savedValue: '' },
                outputMode: { name: 'Action Output Mode', div: '#ef-srv-ide-action-output-mode-select', type: 'select', mode: 'all', plugin: 'all', savedValue: '' },
                actionScript: { name: 'Action Script', div: '#ef-srv-ide-action-script-editor-content', type: 'texteditor', mode: 'all', plugin: 'all', savedValue: '' },
                jobScript: { name: 'Job Script', div: '#ef-srv-ide-job-script-editor-content', type: 'texteditor', mode: 'all', plugin: 'all', savedValue: '' },
                os: { name: 'Operating System', div: '#ef-srv-ide-action-os', type: 'dynamic-select', mode: 'interactive', plugin: 'all', savedValue: '', initFunc: 'drawOSs' },
                sessionMode: { name: 'Session Mode', div: '#ef-srv-ide-action-session-mode', type: 'dynamic-select', mode: 'interactive', plugin: 'all', savedValue: '', initFunc: 'drawSessionModes' },
                cluster: { name: 'Cluster', div: '#ef-srv-ide-action-cluster', type: 'dynamic-select', mode: 'interactive', plugin: 'all', savedValue: '' },
                remote: { name: 'Remote Visualization Technology', div: '#ef-srv-ide-action-remote', type: 'dynamic-select', mode: 'interactive', plugin: 'all', savedValue: '' },
                desktopManager: { name: 'Desktop Manager', div: '#ef-srv-ide-action-desktop-manager', type: 'dynamic-select', mode: 'interactive', plugin: 'all', savedValue: '' },
                displayResolution: { name: 'Display Resolution', div: '#ef-srv-ide-action-display-resolution', type: 'resolution', mode: 'interactive', plugin: 'all', savedValue: '' }
            },
            getValue: function (component) {
                var value = '';
                switch (component.type) {
                case 'efserviceeditor':
                    if (jQuery(component.div).children().length > 0) {
                        value = jQuery(component.div).efserviceeditor('getContent');
                    }
                    else {
                        value = component.savedValue;
                    }
                    break;
                case 'string':
                    value = jQuery(component.div).val();
                    break;
                case 'select':
                case 'dynamic-select':
                    value = jQuery(component.div).val();
                    if (!value) {
                        value = component.savedValue;
                    }
                    break;
                case 'resolution':
                    if (jQuery("[for='" + component.div.substring(1) + "']") !== "none") {
                        value = jQuery(component.div).val();
                    }
                    else {
                        value = component.savedValue;
                    }
                    break;
                case 'texteditor':
                    value = ide.texteditors.getContent(component);
                    if (typeof(value) === 'undefined' || value === null) {
                        value = component.savedValue;
                    }
                    break;
                case 'conf':
                    jQuery.each(ide.handsontables.getContent(component), function (key, val) {
                        if (val[0]) {
                            if (val[2]) {
                                value += '#' + val[2];
                            }
                            value += '\n' + val[0] + '=' + val[1] + '\n';
                        }
                    });
                    if (typeof(value) === 'undefined' || value === null) {
                        value = component.savedValue;
                    }
                    break;
                case 'metadata':
                    jQuery.each(ide.handsontables.getContent(component), function (key, val) {
                        if (val[0]) {
                            value += val[0] + ',' + (val[1] ? val[1] : '') + ';';
                        }
                    });
                    if (typeof(value) === 'undefined' || value === null) {
                        value = component.savedValue;
                    }
                    break;
                case 'boolean':
                    if (jQuery(component.div).length > 0) {
                        // value = jQuery(component.div).attr("checked") ? "true" : "false";
			value = jQuery(component.div).is(":checked") ? "true" : "false";

                    }
                    else {
                        value = component.savedValue;
                    }
                    break;
                case 'radio':
                    if (jQuery(component.div).length > 0) {
                        value = jQuery(component.div + ':checked').val();
                    }
                    else {
                        value = component.savedValue;
                    }
                    break;
                case 'ttl':
                    var ttlLabel, indexes, ttlValueIndex;

                    if (jQuery(component.div).length > 0) {
                        value = '-1';
                        if (jQuery('#ef-srv-ide-service-attributes-spooler-presence').prop('checked')) {
                            // Convert slider Label text value to TTL value
                            ttlLabel = jQuery(component.div).text();
                            indexes = jQuery.map(ide.ttlValues, function (obj, index) {
                                if (obj.label === ttlLabel) {
                                    return index;
                                }
                            });
                            ttlValueIndex = typeof indexes[0] !== "undefined" ? indexes[0] : 8;
                            value = ide.ttlValues[ttlValueIndex].v;
                        }
                    }
                    else {
                        value = component.savedValue;
                    }
                    break;
                default:
                    break;
                }
                return value;
            },
            setValue: function (component, value) {
                var data;
                switch (component.type) {
                case 'efserviceeditor':
                    ide.triggerEvent(ide._events.LOAD, component.name);
                    jQuery(component.div).efserviceeditor('setContent', value);
                    break;
                case 'string':
                case 'resolution':
                    jQuery(component.div).val(value);
                    break;
                case 'select':
                    jQuery(component.div).val(value).change();
                    break;
                case 'dynamic-select':
                    if (jQuery(component.div + " option[value='" + value + "']").length > 0) {
                        jQuery(component.div).val(value).change();
                    }
                    else {
                        if (jQuery(component.div + " option[value='--']").length === 0) {
                            jQuery(component.div).append('<option value="--">--</option>');
                        }
                        jQuery(component.div).val("--").change();
                    }
                    break;
                case 'texteditor':
                    ide.texteditors.setContent(component, value);
                    break;
                case 'conf':
                    data = [];
                    jQuery.each(ide.handsontables.conf.fileToJson(value), function (arrayID, obj) {
                        data.push([ obj.property, obj.value, obj.comment ]);
                    });
                    if (data.length !== 0) {
                        ide.handsontables.setContent(component, data);
                    }
                    break;
                case 'metadata':
                    data = [];
                    value.each(function () {
                        data.push([ jQuery(this).attr("name"), jQuery(this).text() ]);
                    });
                    if (data.length !== 0) {
                        ide.handsontables.setContent(component, data);
                    }
                    break;
                case 'boolean':
                    // NSW jQuery(component.div).attr("checked", value === "true").change();
                    jQuery(component.div).prop("checked", value === "true").change();
                    break;
                case 'radio':
                    jQuery(component.div + '[value=' + value + ']').prop('checked', true);
                    break;
                case 'ttl':
                    if (value === '-1') {
                        jQuery('#ef-srv-ide-service-attributes-spooler-presence').prop('checked', false).change();
                    }
                    else {
                        var indexes;
                        jQuery('#ef-srv-ide-service-attributes-spooler-presence').prop('checked', true).change();
                        // Convert TTL to int for the slider
                        indexes = jQuery.map(ide.ttlValues, function (obj, index) {
                            if (obj.v === value) {
                                return index;
                            }
                        });
                        jQuery(".ef-srv-ide-service-attributes-ttl-slider").slider("value", typeof indexes[0] !== "undefined" ? indexes[0] : 8);
                    }
                    break;
                default:
                    break;
                }
            },
            setSavedValue: function (component, value) {
                component.savedValue = value;
                ide._log("[" + component.name + "] -> set saved value to (" + value + ")");
            },
            init: function (component, value) {
                switch (component.type) {
                case 'efserviceeditor':
                    ide.service.components.setValue(component, value);
                    // savedValue will be updated only when init, revert or save phase is completed
                    break;
                case 'string':
                case 'select':
                case 'texteditor':
                case 'conf':
                case 'metadata':
                case 'boolean':
                case 'radio':
                case 'ttl':
                    ide.service.components.setValue(component, value);
                    ide.service.components.setSavedValue(component, ide.service.components.getValue(component));
                    break;
                case 'dynamic-select':
                    ide.service.components.setSavedValue(component, value);
                    if (component.initFunc) {
                        ide[component.initFunc]();
                    }
                    break;
                case 'resolution':
                    ide.service.components.setValue(component, value);
                    ide.service.components.setSavedValue(component, value);
                    break;
                default:
                    break;
                }
            },
            validate: function (component, ideMessage) {
                var isValid, sortedIdList, jsonData;
                isValid = true;
                if ((component.mode === ide.service.type || component.mode === 'all') &&
                        (component.plugin === ide.plugin || component.plugin === 'all')) {
                    if (component.validator && component.validator.pattern) {
                        if (!new RegExp(component.validator.pattern).test(ide.service.components.getValue(component))) {
                            ide.messages.alert(ideMessage, component.name + ' ' + component.validator.message);
                            isValid = false;
                        }
                    }
                    else {
                        switch (component.type) {
                        case 'efserviceeditor':
                            // service name
                            if (!ide.service.name.value) {
                                ide.messages.alert(ide.messages.primary, "Service Name cannot be empty.");
                                isValid = false;
                                break;
                            }
                            // Check for duplicate ids
                            sortedIdList = jQuery(component.div).efserviceeditor('listId').sort();
                            for (var i = 0; i < sortedIdList.length - 1; i++) {
                                if (sortedIdList[i + 1] === sortedIdList[i]) {
                                    ide.messages.alert(ideMessage, "The option with id (" + efEncodeHtml(sortedIdList[i]) + ") is duplicated. Please fix it.");
                                    isValid = false;
                                    return false;
                                }
                            }
                            if (!isValid) {
                                break;
                            }
                            // validate embed and list-of-resolutions
                            jsonData = jQuery.parseJSON(ide.service.components.getValue(component));
                            if (jsonData.options) {
                                // service.options item
                                jQuery.each(jsonData.options, function (key, value) {
                                    if (!ide.validateEmbedOptions(jsonData.options[key], ideMessage)) {
                                        isValid = false;
                                        return false;
                                    }
                                    if (!ide.validateResolutionsOptions(jsonData.options[key], ideMessage)) {
                                        isValid = false;
                                        return false;
                                    }
                                });
                            }
                            if (!isValid) {
                                break;
                            }
                            // validate file action
                            isValid = ide.validateFileAction(ide.messages.primary);
                            break;
                        case 'dynamic-select':
                            if (ide.service.components.getValue(component) === '--') {
                                ide.messages.alert(ideMessage, component.name + " value is not valid.");
                                isValid = false;
                            }
                            break;
                        case 'conf':
                        case 'metadata':
                            isValid = ide.handsontables.validate(component.div, component.name, ideMessage);
                            break;
                        case 'ttl':
                            if (jQuery('.ef-srv-edt-option-file, .ef-srv-edt-option-sfu, .ef-srv-edt-option-mfu, .ef-srv-edt-option-textfile').length) {
                                if (ide.service.components.getValue(component) === '-1') {
                                    ide.messages.alert(ideMessage, "Spooler Area is mandatory when using File options");
                                    isValid = false;
                                }
                                if (ide.service.components.getValue(component) === '0') {
                                    ide.messages.alert(ideMessage, "Using File options, Spooler Area with immediate expiration is not allowed");
                                    isValid = false;
                                }
                            }
                            break;
                        default:
                            break;
                        }
                    }
                }
                return isValid;
            },
            enable: function (component) {
                switch (component.type) {
                case 'boolean':
                case 'radio':
                case 'string':
                case 'select':
                    jQuery(component.div).prop('disabled', false);
                    jQuery(component.div).parent().removeClass('ef-srv-ide-component-disabled');
                    ide._log("[" + component.name + "] -> Enabling");
                    break;
                case 'dynamic-select':
                case 'resolution':
                    jQuery(component.div).parent().show();
                    ide._log("[" + component.name + "] -> Enabling");
                    break;
                default:
                    break;
                }
            },
            disable: function (component, value) {
                let logMsg = "";
                switch (component.type) {
                case 'boolean':
                case 'radio':
                case 'string':
                case 'select':
                    logMsg = "";
                    if (value !== undefined) {
                        ide.service.components.setValue(component, value);
                        logMsg = " and setting value to: '" + value + "'";
                    }
                    jQuery(component.div).prop('disabled', true);
                    jQuery(component.div).parent().addClass('ef-srv-ide-component-disabled');
                    ide._log("[" + component.name + "] -> Disabling" + logMsg);
                    break;
                case 'dynamic-select':
                case 'resolution':
                    logMsg = "";
                    if (value !== undefined) {
                        jQuery(component.div).val(value).change();
                        logMsg = " and setting value to: '" + value + "'";
                    }
                    jQuery(component.div).parent().hide();
                    ide._log("[" + component.name + "] -> Disabling" + logMsg);
                    break;
                default:
                    break;
                }
            }
        },
        isModified: function () {
            var changed, tmpChanged, innerComponentObj, componentObj;

            changed = false;
            jQuery.each(this.components, function (component) {
                if (component === 'action' || component === 'fileAction') {
                    jQuery.each(ide.service.components[component], function (innerComponent) {
                        innerComponentObj = ide.service.components[component][innerComponent];
                        if ((innerComponentObj.mode === ide.service.type || innerComponentObj.mode === 'all') &&
                                (innerComponentObj.plugin === ide.plugin || innerComponentObj.plugin === 'all') &&
                                typeof innerComponentObj.savedValue !== "undefined" &&
                                typeof ide.service.components.getValue(innerComponentObj) !== "undefined") {
                            tmpChanged = (innerComponentObj.savedValue !== ide.service.components.getValue(innerComponentObj));
                            if (tmpChanged) {
                                ide._log("[" + innerComponentObj.name + "] -> Saved value (" + innerComponentObj.savedValue + ") is different from (" + ide.service.components.getValue(innerComponentObj) + ")");
                            }
                            changed = changed || tmpChanged;
                        }
                    });
                }
                else {
                    componentObj = ide.service.components[component];
                    if ((componentObj.mode === ide.service.type || componentObj.mode === 'all') &&
                            (componentObj.plugin === ide.plugin || componentObj.plugin === 'all') &&
                            typeof componentObj.savedValue !== "undefined" &&
                            typeof ide.service.components.getValue(componentObj) !== "undefined") {
                        tmpChanged = (componentObj.savedValue !== ide.service.components.getValue(componentObj));
                        if (tmpChanged) {
                            ide._log("[" + componentObj.name + "] -> Saved value (" + componentObj.savedValue + ") is different from (" + ide.service.components.getValue(componentObj) + ")");
                        }
                        changed = changed || tmpChanged;
                    }
                }
            });
            return changed;
        },
        save: function () {
            var isValid = true;
            jQuery.each(this.components, function (component) {
                if (component === 'action' || component === 'fileAction') {
                    jQuery.each(ide.service.components[component], function (innerComponent) {
                        if (!ide.service.components.validate(ide.service.components[component][innerComponent], ide.messages.primary)) {
                            isValid = false;
                            return false;
                        }
                    });
                }
                else if (!ide.service.components.validate(ide.service.components[component], ide.messages.primary)) {
                    isValid = false;
                    return false;
                }
            });
            if (!isValid) {
                return;
            }

            ide.saveFiles();
        }
    },

    buttons: {
        editSettings: '#ef-srv-ide-settings-button',
        testService: '#ef-srv-ide-test-run-button',
        restoreService: '#ef-srv-ide-restore-button',
        saveService: '#ef-srv-ide-save-service-button',
        publishService: '#ef-srv-ide-publish-service-button',
        closeIde: '#ef-srv-ide-close-button',
        draw: function () {
            jQuery(ide.buttons.testService).button().click(function (event) {
                var newWindow, jsonData;

                event.preventDefault();
                if (ide.service.id) {
                    if (ide.unsavedChanges.value) {
                        ide.messages.alert(ide.messages.primary, "There are unsaved changes. The Service '" + efEncodeHtml(ide.service.name.value) + "' must be saved prior to test it.", true);
                    }
                    else {
                        jsonData = jQuery.parseJSON(ide.service.components.getValue(ide.service.components.core));
                        newWindow = window.open("", ide.service.id);
                        if (jQuery(jsonData.options).length > 0) {
                            newWindow.location.href = ide.sdf + '?_service=' + encodeURIComponent(ide.service.id) + '&testRun=true&navigation=manage.services';
                        }
                        else {
                            newWindow.location.href = ide.sdf + '?_uri=' + ide.adminUri + encodeURIComponent(ide.service.id) + '&testRun=true&navigation=manage.services';
                        }
                    }
                }
                else {
                    ide.messages.alert(ide.messages.primary, "The Service '" + efEncodeHtml(ide.service.name.value) + "' must be saved prior to test it", true);
                }
                return false;
            });

            jQuery(ide.buttons.restoreService).button().click(function (event) {
                var dialog;

                event.preventDefault();
                if (ide.service.id) {
                    if (ide.unsavedChanges.value) {
                        dialog = jQuery('<div class="hy-simple-input-dialog"/>').appendTo(jQuery('body'));
                        jQuery('<div class="ef-srv-ide-confirmation-msg">Are you sure you want to discard changes?</div>').appendTo(dialog);

                        dialog.dialog({
                            title: "Revert to the last saved version",
                            resizable: false,
                            closeText: '',
                            buttons: {
                                Cancel: function () {
                                    jQuery(this).dialog("close");
                                },
                                Ok: function () {
                                    jQuery(this).dialog("close");
                                    ide.revertServiceEditor();
                                }
                            },
                            modal: true
                        });
                    }
                    else {
                        ide.revertServiceEditor();
                    }
                }
                else {
                    ide.messages.alert(ide.messages.primary, "There are no backup of '" + efEncodeHtml(ide.service.name.value) + "' Service available", true);
                }
            });

            jQuery(ide.buttons.saveService).button().click(function (event) {
                event.preventDefault();
                ide.service.save();
            });
            jQuery('<span class="ef-srv-ide-unsaved-changes"><i class="fa fa-pencil-square-o" /></span>').appendTo(ide.buttons.saveService).hide();

            // Update and publish an already published service
            jQuery(ide.buttons.publishService).button().click(function (event) {
                event.preventDefault();

                console.log("Saving service before publishing.");
                ide.service.save();

                console.log("Updating service: Unpublishing and then publishing " + ide.service.id);
                ide.triggerEvent(ide._events.PUBLISH, 'Service Editor');

                // Already published services will not be overwritten, so we unpublish first and then publish.
                // Service is marked as locked in the editor, so we need to force unlock it to unpublish here
                // Unpublish first...
                jQuery.hydrogen.invokeService({
                    sdf: ide.sdf,
                    uri: ide.serviceManagerUri + 'unpublish.service',
                    data: {
                        serviceIds: ide.service.id,
                        plugin:     ide.plugin,
                        unlock:     true
                    },
                    success: function (xml) {
                        console.log("Unpublish successful for service " + ide.service.id);
                        // ...then publish
                        // Service is marked as locked in the editor, so we need to force unlock it to publish here
                        jQuery.hydrogen.invokeService({
                            sdf: ide.sdf,
                            uri: ide.serviceManagerUri + 'publish.service',
                            data: {
                                serviceIds:       ide.service.id,
                                userGroupList:    ide.service.service_detail_json.groups,
                                plugin:           ide.plugin,
                                targetFolderName: ide.service.service_detail_json.targetFolder,
                                unlock:           true
                            },
                            success: function (xml) {
                                // do not trigger LOAD_DONE event for the 'Service Editor' component
                                // to keep the current PUBLISHING state before re-init
                                ide.initServiceEditor();
                                console.log("Successfully published " + ide.service.id);
                            },
                            error: function () {
                                ide.triggerEvent(ide._events.LOAD_ERROR, 'Service Editor');
                            },
                            messagebox: "Published"
                        });
                    },
                    error: function () {
                        ide.triggerEvent(ide._events.LOAD_ERROR, 'Service Editor');
                    },
                    messagebox: "Published"
                });
            });

            jQuery(ide.buttons.closeIde).button().click(function (event) {
                event.preventDefault();
                if (ide.unsavedChanges.value) {
                    if (confirm("New application settings not saved. Do you wish to leave the page?")) {
                        ide.redirectToManager();
                    }
                } else {
                    ide.redirectToManager();
                }
            });

            jQuery(ide.buttons.editSettings).button().click(function (event) {
                event.preventDefault();
                jQuery("#ef-srv-ide-service-settings-editor").dialog('open');
                // render handsontable
                jQuery(ide.service.components.metadata.div).handsontable('render');
                // re-render first tab editor
                jQuery('[for=ef-srv-ide-attributes-button]').click().change();
                return false;
            });
        },
        disableAll: function () {
            this.disable(ide.buttons.editSettings);
            this.disable(ide.buttons.testService);
            this.disable(ide.buttons.restoreService);
            this.disable(ide.buttons.saveService);
            this.disable(ide.buttons.publishService);
            this.disable(ide.buttons.closeIde);
        },
        enableAll: function () {
            this.enable(ide.buttons.editSettings);
            this.enable(ide.buttons.testService);
            this.enable(ide.buttons.restoreService);
            this.enable(ide.buttons.saveService);
            this.enable(ide.buttons.publishService);
            this.enable(ide.buttons.closeIde);
        },
        disable: function (buttonDiv) {
            jQuery(buttonDiv).button('disable');
        },
        enable: function (buttonDiv) {
            jQuery(buttonDiv).button('enable');
        }
    },

    messages: {
        primary: '#ef-srv-ide-message-wrapper',
        secondary: '#ef-srv-ide-secondary-message-wrapper',
        settings: '#ef-srv-ide-settings-editor-message',
        actionEditor: '#ef-srv-ide-action-editor-message',
        info: function (div, text, timeoutEnabled) {
            ide.messages._show(div, 'info', text, timeoutEnabled);
        },
        alert: function (div, text, timeoutEnabled) {
            ide.messages._show(div, 'alert', text, timeoutEnabled);
        },
        clear: function (div) {
            jQuery(div).hymessage().hymessage('clear');
        },
        getMessageBox: function (div) {
            return jQuery(div).hymessage();
        },
        _show: function (div, mode, text, timeoutEnabled) {
            if (timeoutEnabled) {
                jQuery(div).hymessage().hymessage(mode, text, this._timeout);
            }
            else {
                jQuery(div).hymessage().hymessage(mode, text);
            }
        },
        _timeout: 4000
    },

    _log: function () {
        if (this.debug) {
            console.log.apply(console, arguments);
        }
    },

    // ------ IDE status management ------ //

    _events: Object.freeze({
        // Events related to asynch actions
        LOAD: "load",             // get service content (AJAX), update options or dynamic select (AJAX): os, clusters, sessionModes, remotes, desktop-managers, resolutions, embeddable-services, user-groups
        REVERT: "revert",         // reload service content (AJAX)
        SAVE: "save",             // save service (AJAX)
        PUBLISH: "publish",       // update published service (AJAX)
        LOAD_DONE: "load-done",   // async action terminated correctly
        LOAD_ERROR: "load-error", // async action failed

        // Event related to synch action
        CHANGE: "change"          // simple UI interaction
    }),

    triggerEvent: function (ideEvent, component) {
        ide.status.evaluateEvent(ideEvent, component);
    },

    _status: Object.freeze({
        INITIALIZING: "initializing", // getting service content for the first time
        READY: "ready",               // no actions in progress
        REVERTING: "reverting",       // getting service content
        SAVING: "saving",             // saving service
        PUBLISHING: "publishing",     // publishing service
        UPDATING: "updating"          // updating efserviceeditor options or dynamic select according to UI changes
    }),

    status: {
        _value: '',
        _components: {
            _loading: {},
            _failed: {},
            reset: function () {
                this._loading = {};
                this._failed = {};
            },
            setLoading: function (component) {
                this._loading[component] = true;
            },
            setFinished: function (component) {
                delete this._loading[component];
                if (this._failed[component]) {
                    delete this._failed[component];
                }
            },
            setFailed: function (component) {
                delete this._loading[component];
                this._failed[component] = true;
            }
        },
        init: function () {
            this._components.reset();
            this.set(ide._status.INITIALIZING);
            ide._log("Status (" + this._value + ")");
        },
        set: function (newStatus) {
            this._value = newStatus;
        },
        isInitState: function () {
            var isInit;

            switch (this._value) {
            case ide._status.INITIALIZING:
            case ide._status.REVERTING:
            case ide._status.SAVING:
            case ide._status.PUBLISHING:
                isInit = true;
                break;
            case ide._status.UPDATING:
            case ide._status.READY:
                isInit = false;
                break;
            default:
                isInit = false;
                break;
            }
            return isInit;
        },
        getLoadingComponents: function () {
            return this._components._loading;
        },
        hasLoadingComponents: function () {
            return jQuery.isEmptyObject(this._components._loading) ? false : true;
        },
        getFailedComponents: function () {
            return this._components._failed;
        },
        hasFailedComponents: function () {
            return jQuery.isEmptyObject(this._components._failed) ? false : true;
        },
        evaluateEvent: function (ideEvent, component) {
            var oldStatus, newStatus, oldLoadingListSize, newLoadingList, debugMessage;

            oldStatus = this._value;
            oldLoadingListSize = Object.keys(this.getLoadingComponents()).length;

            switch (oldStatus) {
            case ide._status.READY:
                switch (ideEvent) {
                case ide._events.REVERT:
                    this._components.reset();
                    this.set(ide._status.REVERTING);
                    break;
                case ide._events.SAVE:
                    this._components.reset();
                    this._components.setLoading(component);
                    this.set(ide._status.SAVING);
                    break;
                case ide._events.PUBLISH:
                case ide._events.LOAD_DONE:
                case ide._events.LOAD_ERROR:
                    // should not occur
                    break;
                case ide._events.LOAD:
                    this._components.setLoading(component);
                    this.set(ide._status.UPDATING);
                    break;
                case ide._events.CHANGE:
                    // do nothing, just perform an UI update for unsavedChanges flag
                    break;
                default:
                    break;
                }
                break;
            case ide._status.INITIALIZING:
            case ide._status.SAVING:
            case ide._status.REVERTING:
                switch (ideEvent) {
                case ide._events.REVERT:
                case ide._events.SAVE:
                    // cannot occur
                    break;
                case ide._events.PUBLISH:
                    this._components.reset();
                    this._components.setLoading(component);
                    this.set(ide._status.PUBLISHING);
                    break;
                case ide._events.LOAD:
                    this._components.setLoading(component);
                    break;
                case ide._events.LOAD_DONE:
                    this._components.setFinished(component);
                    if (!this.hasLoadingComponents()) {
                        this.set(ide._status.READY);
                    }
                    break;
                case ide._events.LOAD_ERROR:
                    this._components.setFailed(component);
                    if (!this.hasLoadingComponents()) {
                        this.set(ide._status.READY);
                    }
                    break;
                case ide._events.CHANGE:
                    // do nothing, just perform an UI update for unsavedChanges flag
                    break;
                default:
                    break;
                }
                break;
            case ide._status.PUBLISHING:
                switch (ideEvent) {
                case ide._events.REVERT:
                case ide._events.SAVE:
                case ide._events.PUBLISH:
                    // cannot occur
                    break;
                case ide._events.LOAD:
                    this._components.setLoading(component);
                    break;
                case ide._events.LOAD_DONE:
                    this._components.setFinished(component);
                    if (!this.hasLoadingComponents()) {
                        this.set(ide._status.READY);
                    }
                    break;
                case ide._events.LOAD_ERROR:
                    this._components.setFailed(component);
                    if (!this.hasLoadingComponents()) {
                        this.set(ide._status.READY);
                    }
                    break;
                }
                break;
            case ide._status.UPDATING:
                switch (ideEvent) {
                case ide._events.REVERT:
                case ide._events.SAVE:
                case ide._events.PUBLISH:
                    // cannot occur
                    break;
                case ide._events.LOAD:
                    this._components.setLoading(component);
                    break;
                case ide._events.LOAD_DONE:
                    this._components.setFinished(component);
                    if (!this.hasLoadingComponents()) {
                        this.set(ide._status.READY);
                    }
                    break;
                case ide._events.LOAD_ERROR:
                    this._components.setFailed(component);
                    if (!this.hasLoadingComponents()) {
                        this.set(ide._status.READY);
                    }
                    break;
                case ide._events.CHANGE:
                    // do nothing, just perform an UI update to check unsaved changes flag
                    break;
                default:
                    break;
                }
                break;
            default:
                break;
            }

            newStatus = this._value;
            newLoadingList = this.getLoadingComponents();

            debugMessage = (oldLoadingListSize !== Object.keys(newLoadingList).length) ? "In progress: (" + JSON.stringify(newLoadingList) + "). Failed: (" + JSON.stringify(this.getFailedComponents()) + ") -> " : "";
            debugMessage += (oldStatus === newStatus) ? "No status change" : "Status changes from (" + oldStatus + ") to (" + newStatus + ")";
            ide._log("[" + component + "] -> Event (" + ideEvent + ") ->", debugMessage);
            ide.updateUI(oldStatus, newStatus);
        }
    },

    updateUI: function (oldStatus, newStatus) {
        var component, serviceCore, errorMsg = '';

        switch (newStatus) {
        case ide._status.INITIALIZING:
            ide.loadingBox.set("Loading ...");
            ide.buttons.disableAll();
            break;
        case ide._status.REVERTING:
            ide.loadingBox.set("Loading ...");
            ide.buttons.disableAll();
            break;
        case ide._status.UPDATING:
            ide.loadingBox.set("Loading ...");
            break;
        case ide._status.SAVING:
            ide.loadingBox.set("Saving ...");
            ide.buttons.disableAll();
            break;
        case ide._status.PUBLISHING:
            ide.loadingBox.set("Publishing ...");
            ide.buttons.disableAll();
            break;
        case ide._status.READY:
            ide.loadingBox.clean();

            switch (oldStatus) {
            case ide._status.INITIALIZING:
                // init ide layout
                jQuery('.ef-srv-edt-col-center').addClass('hy-left-column-wrapper');
                jQuery('.ef-srv-edt-col-right-wrapper').addClass('hy-right-column');
                jQuery('.ef-srv-edt-col-right').addClass('hy-innertube-right');
                jQuery(ide.contentDiv).show();

                // buttons and error messages
                jQuery(ide.messages.secondary).insertAfter(jQuery(".ef-srv-edt-toolbars"));
                jQuery(ide.messages.primary).insertAfter(jQuery(".ef-srv-edt-toolbars"));
                if (ide.service.lock.byAnother) {
                    // service locked
                    ide.buttons.enable(ide.buttons.editSettings);
                    ide.buttons.enable(ide.buttons.closeIde);
                }
                else if (ide.status.hasFailedComponents()) {
                    // some content loading fails
                    for (component in ide.status.getFailedComponents()) {
                        errorMsg += (errorMsg === '') ? component : ', ' + component;
                    }
                    ide.messages.alert(ide.messages.primary, 'An error occurs initializing (' + errorMsg + ') values. Please try to reload the page or contact the administrator.');
                    ide.buttons.enable(ide.buttons.editSettings);
                    ide.buttons.enable(ide.buttons.closeIde);
                    // do not enable revert action for template services
                    if (ide.service.id) {
                        ide.buttons.enable(ide.buttons.restoreService);
                    }
                }
                else {
                    // successful initialization
                    ide.messages.clear(ide.messages.primary);
                    ide.buttons.enableAll();

                    // init unsaved changes flag and additional messages
                    if (ide.service.id) {
                        // set service core saved value
                        ide.service.components.setSavedValue(ide.service.components.core, ide.service.components.getValue(ide.service.components.core));

                        // init from catalog
                        ide.unsavedChanges.set(false);

                        if (!ide.service.ideVersion) {
                            // editor version is empty in 2015.0 services
                            ide.messages.alert(ide.messages.primary, "The service has been imported from previous 2015.0 format.<br/>" +
                                "Please be aware the Session Properties are overridden by settings in the Action Script");
                        }
                    }
                    else {
                        // init from template
                        ide.unsavedChanges.set(true);

                        // append info box if there are no options
                        serviceCore = jQuery.parseJSON(ide.service.components.getValue(ide.service.components.core));
                        if (jQuery(serviceCore.options).length === 0) {
                            jQuery('.ef-srv-edt-toolbars').after('<div class="ef-srv-edt-initial-message ui-state-highlight ui-corner-all ef-srv-ide-warning">' +
                                    '<p><span class="ui-icon ui-icon-info"></span>' +
                                    'If no options are added to the service, no service form will be displayed and the service will run at the mouse click.</p>' +
                                    '<div class="ui-icon ui-icon-close"></div>' +
                                '</div>');
                            jQuery('.ef-srv-edt-initial-message .ui-icon-close').click(function () {
                                jQuery('.ef-srv-edt-initial-message').hide();
                            });
                        }

                        // show demo tour button
                        ide.tour.init();
                        jQuery('#ef-srv-ide-demo-tour-button').show().button().click(function () {
                            ide.tour.start(true);
                        });
                    }
                }
                break;
            case ide._status.REVERTING:
                // buttons, confirmation and error messages
                if (ide.status.hasFailedComponents()) {
                    for (component in ide.status.getFailedComponents()) {
                        errorMsg += (errorMsg === '') ? component : ', ' + component;
                    }
                    ide.messages.alert(ide.messages.primary, 'An error occurs retrieving (' + errorMsg + ') values. Please retry the Revert action or reload the page.');
                    ide.buttons.enable(ide.buttons.editSettings);
                    ide.buttons.enable(ide.buttons.restoreService);
                    ide.buttons.enable(ide.buttons.closeIde);
                }
                else {
                    ide.messages.info(ide.messages.primary, "Service (" + efEncodeHtml(ide.service.name.value) + ") correctly restored from latest backup.", true);
                    ide.buttons.enableAll();

                    // set service core saved value
                    ide.service.components.setSavedValue(ide.service.components.core, ide.service.components.getValue(ide.service.components.core));
                }

                // check for changes
                ide.unsavedChanges.check();
                break;
            case ide._status.SAVING:
                // confirmation and error messages
                if (ide.status.hasFailedComponents()) {
                    for (component in ide.status.getFailedComponents()) {
                        errorMsg += (errorMsg === '') ? component : ', ' + component;
                    }
                    ide.messages.alert(ide.messages.primary, 'An error occurs saving (' + errorMsg + ') values. Please retry the Save action or contact the administrator.');
                }
                else {
                    ide.messages.info(ide.messages.primary, 'Service (' + efEncodeHtml(ide.service.name.value) + ') saved successfully', true);

                    // set service core saved value
                    ide.service.components.setSavedValue(ide.service.components.core, ide.service.components.getValue(ide.service.components.core));
                }
                // buttons
                ide.buttons.enableAll();

                // check for changes
                ide.unsavedChanges.check();
                break;
            case ide._status.PUBLISHING:
                // confirmation and error messages
                if (ide.status.hasFailedComponents()) {
                    for (component in ide.status.getFailedComponents()) {
                        errorMsg += (errorMsg === '') ? component : ', ' + component;
                    }
                    ide.messages.alert(ide.messages.primary, 'An error occurs publishing (' + errorMsg + '). Please retry the Save action or contact the administrator.');
                }
                else {
                    ide.messages.info(ide.messages.primary, 'Service (' + efEncodeHtml(ide.service.name.value) + ') published successfully', true);

                    // set service core saved value
                    ide.service.components.setSavedValue(ide.service.components.core, ide.service.components.getValue(ide.service.components.core));
                }
                // buttons
                ide.buttons.enableAll();

                // check for changes
                ide.unsavedChanges.check();
                break;
            case ide._status.UPDATING:
                // buttons, confirmation and error messages
                if (ide.status.hasFailedComponents()) {
                    for (component in ide.status.getFailedComponents()) {
                        errorMsg += (errorMsg === '') ? component : ', ' + component;
                    }
                    ide.messages.alert(ide.messages.primary, 'An error occurs retrieving (' + errorMsg + ') values. Please check the correctness or Revert the Service from a safe state.');
                }
                else {
                    ide.messages.clear(ide.messages.primary);
                    ide.buttons.enableAll();
                }

                // check for changes
                ide.unsavedChanges.check();
                break;
            case ide._status.READY:
                // check for changes
                ide.unsavedChanges.check();
                break;
            default:
                break;
            }
            break;
        default:
            break;
        }
    },

    loadingBox: {
        _div: '.ef-srv-ide-loading-wrapper .loading',
        set: function (content) {
            jQuery(this._div).text(content).show();
        },
        clean: function () {
            jQuery(this._div).hide();
        }
    },

    init: function (plugin, serviceId, serviceType, templateId, editorId, debug) {
        var encPlugin;

        encPlugin = encodeURIComponent(plugin);
        ide.plugin = plugin;
        ide.sdf = '/' + jQuery.enginframe.rootContext + '/' + encPlugin + '/' + encPlugin + '.admin.xml';
        ide.adminUri = '//' + encPlugin + '.admin/';

        ide.service.id = (serviceId !== 'void') ? serviceId : '';
        ide.service.type = (serviceType !== 'void') ? serviceType : '';
        ide.service.templateId = (templateId !== 'void') ? templateId : '';
        ide.editorId = (editorId !== 'void') ? editorId : '';
        ide.debug = debug;

        ide.service.lock.check();
        ide.status.init();
        ide.drawServiceEditor();
        ide.updateForPublish();
        ide.initServiceEditor();
        jQuery(ide.div).show();
    },

    updateForPublish: function() {
        jQuery.hydrogen.invokeService({
            sdf: ide.sdf, 
            uri: ide.serviceManagerUri + 'get.service.details',
            data: {
                serviceId: ide.service.id,
                plugin:     ide.plugin
            },
            success: function (data) {
                ide.service.service_detail_json = {};
                try {
                    ide.service.service_detail_json = JSON.parse(data);
                    console.log(ide.service.service_detail_json);
                    if (ide.service.service_detail_json.published) {
                        $('<div class="ef-srv-ide-edit-published-message">Please be careful - you are editing a published service!</div>')
                            .insertAfter('.ef-srv-ide-header');
                        $('#ef-srv-ide-publish-service-button .ui-label').text('Update Published');
                    } else {
                        $('#ef-srv-ide-publish-service-button').hide();
                        $('#ef-srv-ide-publish-service-button .ui-label').text('Publish Service');
                    }
                } catch (err) {
                    console.log("Error retrieving service_detail_json");
                    console.log(err);
                }
            }
        });
    },
    
    redirectToManager: function (timeout) {
        ide.service.lock.unset(function () {
            if (timeout) {
                setTimeout(function () {
                    window.location.href = ide.sdf + '?_uri=' + ide.adminUri + 'manage.services';
                }, timeout);
            } else {
                window.location.href = ide.sdf + '?_uri=' + ide.adminUri + 'manage.services';
            }
        });
    },

    // ------ Unsaved Changes ------ //

    unsavedChanges: {
        div: '.ef-srv-ide-unsaved-changes',
        value: false,
        set: function (value) {
            if (value) {
                jQuery(this.div).show();
            }
            else {
                jQuery(this.div).hide();
            }
            this.value = value;
        },
        check: function () {
            var changed = ide.service.isModified();
            this.set(changed);
            return changed;
        }
    },

    // ------ Service editor ------ //

    drawServiceEditor: function () {
        var editor, optionsFilterTimeout, optionsFilterDiv, optionsFilterId, optionHeaderDiv, unusedEfOptions;

        // detect EF options to disable from the editor
        unusedEfOptions = '';
        if (ide.plugin === 'vdi') {
            unusedEfOptions = 'dynamic-list,mfu,sfu,file,textfile,list-of-clusters';
        }
        else {
            if (ide.service.type === 'batch') {
                unusedEfOptions = 'list-of-resolutions,list-of-desktop-managers';
            }
            else {
                unusedEfOptions = 'list-of-clusters';
            }
        }
        // Init efserviceeditor and bind its actions
        editor = jQuery(ide.service.components.core.div).efserviceeditor({ removeEfOptions: unusedEfOptions, debug: ide.debug });
        editor.bind('editAction', function (e, actionId) {
            jQuery("#ef-srv-ide-action-editor").dialog('open');
            // hide default Loading box
            jQuery('#ef-srv-ide-service-editor .ef-srv-ide-loading-wrapper').hide();
            if (ide.service.type === 'interactive') {
                ide.drawOSs();
                ide.drawSessionModes();
            }
            // re-render first tab editor
            jQuery('[for=ef-srv-ide-action-properties-button]').click().change();
        });
        editor.bind('updateTitle', function (e, title) {
            ide.service.name.set(title);
        });
        editor.bind('listServices', function (e) {
            ide.embeddableServices.update(true);
        });
        editor.bind('listUserGroups', function (e) {
            ide.userGroups.update(true);
        });
        editor.bind('addPluginOption', function (e) {
            // Populate plugin property item and hide it
            jQuery('input[name="plugin"]', '.ef-srv-edt-form').val(ide.plugin).change().parent().hide();
        });
        editor.bind('addRemoteOption', function (e) {
            // Populate remote property item
            jQuery('input[name="remote"]', '.ef-srv-edt-form').val(ide.service.components.getValue(ide.service.components.action.remote)).change().parent().hide();
        });
        editor.bind('importCompleted', function (e) {
            ide.triggerEvent(ide._events.LOAD_DONE, ide.service.components.core.name);
        });
        editor.bind('updateCompleted', function (e) {
            ide.triggerEvent(ide._events.LOAD_DONE, ide.service.components.core.name);
        });

        // Bind page leaving
        window.onpagehide = function (e) {
            ide.service.lock.unset();
        };

        // Add Quick search for Options
        optionsFilterDiv = "ef-srv-edt-options-filter";
        optionsFilterId = optionsFilterDiv + "-input";
        jQuery(".ef-srv-edt-col.ef-srv-edt-col-services").prepend("<div id='" + optionsFilterDiv + "' />");
        jQuery("#" + optionsFilterDiv).append("<label for='" + optionsFilterId + "'>Search:</label>" +
                "<input id='" + optionsFilterId + "' type='text'/>");
        jQuery("#" + optionsFilterId).keydown(function () {
            if (optionsFilterTimeout) {
                clearTimeout(optionsFilterTimeout);
            }
        });
        jQuery("#" + optionsFilterId).keyup(function () {
            optionsFilterTimeout = setTimeout(function () {
                editor.efserviceeditor('filter', jQuery('#' + optionsFilterId).val());
            }, 300);
        });
        jQuery('#' + optionsFilterId + '-button').button().click(function (e) {
            e.preventDefault();
        });
        optionHeaderDiv = "ef-srv-edt-col-services-header";
        jQuery("#nj-content-header").append("<div id='" + optionHeaderDiv + "' />");
        jQuery("#" + optionHeaderDiv).append("<div id='" + optionHeaderDiv + "-title'>Options</div>");

        // Move ide-header inside the content
        jQuery('.ef-srv-edt-col-center').prepend(jQuery('.ef-srv-ide-header').detach());

        // Draw editor buttons
        ide.buttons.draw();

        // Bind editor changes
        editor.bind('change', function () {
            if (ide.service.rfbList.update()) {
                ide.drawFileAction();
            }
            ide.triggerEvent(ide._events.CHANGE, ide.service.components.core.name);
        });

        // Draw loading dialog placeholders
        jQuery('#ef-srv-ide-service-editor').append('<div class="ef-srv-ide-loading-wrapper"><div class="loading ui-state-default ui-state-active" /></div>');
        jQuery('#ef-srv-ide-action-editor').append('<div class="ef-srv-ide-loading-wrapper"><div class="loading ui-state-default ui-state-active" /></div>');

        // Draw attribute and action dialogs
        ide.drawSettingsDialog();
        ide.drawActionDialog();

        // Define demo tour
        ide.tour = new Tour({
            steps: [
                {
                    element: ".ef-srv-title",
                    title: "Service Name",
                    content: "Set the name of the Service.",
                    placement: "right"
                },
                {
                    element: ".ef-srv-edt-services-group-0",
                    title: "Options",
                    content: "Drag and Drop options in the form, or click on them and set their properties.",
                    placement: "left"
                },
                {
                    element: ".ef-srv-edt-option-action input",
                    title: "Edit Action",
                    content: "Click in the submit button to edit Service Action.",
                    placement: "top"
                },
                {
                    element: ide.buttons.editSettings,
                    title: "Settings",
                    content: "Configure attributes, CSS and JavaScript for your Service.",
                    placement: "bottom"
                },
                {
                    element: ide.buttons.saveService,
                    title: "Save and Test Run",
                    content: "Save your Service and then test it.",
                    placement: "bottom"
                },
                {
                    element: ide.buttons.publishService,
                    title: "Publish Service",
                    content: "Publish the Service for users.",
                    placement: "bottom"
                }
            ]
        });
    },

    revertServiceEditor: function () {
        ide.triggerEvent(ide._events.REVERT, 'Service Editor');
        ide.initServiceEditor();
    },

    initServiceEditor: function () {
        ide.triggerEvent(ide._events.LOAD, 'Service Editor');
        jQuery.hydrogen.invokeService({
            sdf: ide.sdf,
            uri: ide.serviceManagerUri + 'get.service',
            data: {
                serviceId: ide.service.id,
                templateId: ide.service.templateId,
                serviceType: ide.service.type,
                plugin: ide.plugin
            },
            complete: function () {
            },
            success: function (xml) {
                var xmlObj, jsonObj;

                xmlObj = jQuery(xml);
                jsonObj = jQuery.parseJSON(xmlObj.find('item[name="service-json"]').text());

                ide.service.components.init(ide.service.components.core, xmlObj.find('item[name="service-json"]').text());
                ide.service.name.set(jsonObj.name);
                // service-ef-version is empty in 2015.0 services and templates
                ide.service.ideVersion =  xmlObj.find('item[name="service-ef-version"]').text() ? xmlObj.find('item[name="service-ef-version"]').text() : '';

                // service action
                ide.service.components.init(ide.service.components.action.id, jsonObj.actions[0].id);
                ide.service.components.init(ide.service.components.action.label, jsonObj.actions[0].label);
                ide.service.components.init(ide.service.components.action.resultType, jsonObj.actions[0]['result-type']);
                ide.service.components.init(ide.service.components.action.outputMode, jsonObj.actions[0]['output-mode']);

                // scripts and conf
                ide.service.components.init(ide.service.components.action.actionScript, xmlObj.find('item[name="action-script"]').text());
                ide.service.components.init(ide.service.components.action.jobScript, xmlObj.find('item[name="job-script"]').text());
                ide.service.components.init(ide.service.components.conf, xmlObj.find('item[name="conf"]').text());
                ide.service.components.init(ide.service.components.serviceJs, xmlObj.find('item[name="service-js"]').text());
                ide.service.components.init(ide.service.components.serviceCss, xmlObj.find('item[name="service-css"]').text());

                if (ide.service.type === 'interactive') {
                    ide.service.components.init(ide.service.components.sessionClass, xmlObj.find('item[name="session-class"]').text());
                    ide.service.components.init(ide.service.components.maxSessions, xmlObj.find('item[name="max-sessions"]').text());
                    ide.service.components.init(ide.service.components.action.os, xmlObj.find('item[name="action-os"]').text());
                    ide.service.components.init(ide.service.components.action.sessionMode, xmlObj.find('item[name="action-session-mode"]').text());
                    ide.service.components.init(ide.service.components.action.cluster, xmlObj.find('item[name="action-cluster"]').text());
                    ide.service.components.init(ide.service.components.action.remote, xmlObj.find('item[name="action-remote"]').text());
                    ide.service.components.init(ide.service.components.action.desktopManager, xmlObj.find('item[name="action-desktop-manager"]').text());
                    ide.service.components.init(ide.service.components.action.displayResolution, xmlObj.find('item[name="action-display-resolution"]').text());
                }
                else {
                    // batch attributes
                    ide.service.components.init(ide.service.components.embeddableAttr, xmlObj.find('item[name="service-embeddable"]').text());
                }
                if (ide.plugin === 'applications') {
                    ide.service.components.init(ide.service.components.profile, xmlObj.find('item[name="service-profile"]').text());
                    ide.service.components.init(ide.service.components.resubmitAttr, xmlObj.find('item[name="service-resubmit"]').text());
                    ide.service.components.init(ide.service.components.reuseSpoolerAttr, xmlObj.find('item[name="resubmit-reuse-spooler"]').text());
                    ide.service.components.init(ide.service.components.resetTTLAttr, xmlObj.find('item[name="resubmit-reset-ttl"]').text());
                    ide.service.components.init(ide.service.components.spoolerTTL, xmlObj.find('item[name="spooler-ttl"]').text());
                    ide.service.components.init(ide.service.components.hiddenAttr, xmlObj.find('item[name="service-hidden"]').text());
                    ide.service.components.init(ide.service.components.fileAction.enabled, xmlObj.find('item[name="file-action-enabled"]').text());
                    ide.service.components.init(ide.service.components.fileAction.label, xmlObj.find('item[name="file-action-label"]').text());
                    ide.service.components.init(ide.service.components.fileAction.serviceExecution, xmlObj.find('item[name="file-action-service-execution"]').text());
                    ide.service.components.init(ide.service.components.fileAction.showMenuOnly,
                            (ide.service.components.getValue(ide.service.components.fileAction.enabled) === "true" && ide.service.components.getValue(ide.service.components.hiddenAttr) === "true").toString());
                    ide.service.components.init(ide.service.components.fileAction.optionId, ide.service.rfbList.getHtmlId(xmlObj.find('item[name="file-action-option-id"]').text()));
                    ide.service.components.init(ide.service.components.fileAction.fileFilter, xmlObj.find('item[name="file-action-file-filter"]').text());
                }
                ide.service.components.init(ide.service.components.metadata, xmlObj.find('item[type="metadata"]'));

                ide.embeddableServices.init();
                ide.userGroups.init();
                ide.triggerEvent(ide._events.LOAD_DONE, 'Service Editor');
            },
            error: function () {
                ide.triggerEvent(ide._events.LOAD_ERROR, 'Service Editor');
            }
        });
    },

    // ------ Embeddable services ------ //

    embeddableServices: {
        list: {},
        init: function () {
            this.update(false);
        },
        update: function (updateOptions) {
            ide.triggerEvent(ide._events.LOAD, "Embeddable Services");
            jQuery.hydrogen.invokeService({
                sdf: ide.sdf,
                uri: ide.serviceManagerUri + "list.published.embeddable",
                data: {
                    plugin: ide.plugin
                },
                dataType: "xml",
                success: function (xml) {
                    ide.embeddableServices.list = {};
                    jQuery(xml).find('ef\\:option, option').each(function (index) {
                        var currentId = jQuery(this).attr("id") + ".published";
                        ide.embeddableServices.list[currentId] = jQuery(this).text();
                    });
                    ide.triggerEvent(ide._events.LOAD_DONE, "Embeddable Services");
                    if (updateOptions) {
                        ide.updateEmbedOptions();
                    }
                },
                error: function () {
                    ide.triggerEvent(ide._events.LOAD_ERROR, "Embeddable Services");
                }
            });
        }
    },

    updateEmbedOptions: function () {
        if (jQuery.isEmptyObject(ide.embeddableServices.list)) {
            ide.messages.alert(ide.messages.primary, "You cannot use Dynamic List because there are no Embeddable Services published.");
        } else {
            jQuery('select[name="embedded-service"]').each(function (index, embedElem) {
                var selected, foundId, optionId, selectedServiceId;

                foundId = false;
                selected = jQuery(embedElem).val();
                jQuery(embedElem).empty();
                jQuery.each(ide.embeddableServices.list, function (key, value) {
                    jQuery(embedElem).append('<option value="' + key + '">' + value + '</option>');
                    if (selected === key || selected === '') {
                        jQuery(embedElem).val(key).change();
                        foundId = true;
                    }
                });
                if (!foundId) {
                    optionId = jQuery(embedElem).parent().parent().find('input[name="id"]').first().attr('value');
                    selectedServiceId = selected.substr(selected.lastIndexOf('/') + 1, selected.length - selected.lastIndexOf('/') - ('.published').length - 1);
                    ide.messages.alert(ide.messages.primary, "The option with id (" + optionId + ") has a property (Embedded Service) with inconsistent value:" +
                            "the Service with id (" + selectedServiceId + ") is not published. Select a new Embeddable service or re-publish the selected service.");
                    jQuery(embedElem).click(function () {
                        jQuery(embedElem).change();
                        ide.triggerEvent(ide._events.CHANGE, "Embeddable Services");
                        ide.messages.clear(ide.messages.primary);
                    });
                    return false;
                }
            });
        }
    },

    validateEmbedOptions: function (optionList, ideMessage) {
        var optionId, isValid = true;

        // find and check embed a this level
        jQuery.each(optionList, function (key, value) {
            var foundId, message, selectedServiceId;

            if (key === 'id') {
                optionId = value;
            }
            if (key === 'embedded-service') {
                foundId = false;
                // detect if embed is empty
                if (value === '') {
                    message = "The option with id (" + optionId + ") could not have property (Embedded Service) empty.";
                    isValid = false;
                    if (jQuery.isEmptyObject(ide.embeddableServices.list)) {
                        message += " You cannot use Dynamic List because there are no Embeddable Services published.";
                    }
                    ide.messages.alert(ideMessage, message);
                    return false;
                }

                // detect if the selected embed is a valid published service
                jQuery.each(ide.embeddableServices.list, function (embedId, embedLabel) {
                    if (embedId === value) {
                        foundId = true;
                        return false;
                    }
                });
                if (!foundId) {
                    isValid = false;
                    selectedServiceId = value.substr(value.lastIndexOf('/') + 1, value.length - value.lastIndexOf('/') - ('.published').length - 1);
                    ide.messages.alert(ideMessage,
                            "The option with id (" + optionId + ") has a property Embedded Service with inconsistent value: " +
                            "the Service with id (" + selectedServiceId + ") is not published. " +
                            "Select a new Embeddable service or re-publish the selected service.");
                    return false;
                }
            }
        });
        if (!isValid) {
            // if an embed is not valid, exit
            return false;
        }
        if (optionList.options) {
            // a "group" option has other options inside
            jQuery.each(optionList.options, function (optKey, optValue) {
                if (!ide.validateEmbedOptions(optionList.options[optKey], ideMessage)) {
                    isValid = false;
                    return false;
                }
            });
        }
        return isValid;
    },

    validateResolutionsOptions: function (optionList, ideMessage) {
        var isValid = true;

        if (optionList.options) {
            // a "group" option has other options inside
            jQuery.each(optionList.options, function (optKey, optValue) {
                if (!ide.validateResolutionsOptions(optionList.options[optKey], ideMessage)) {
                    isValid = false;
                    return false;
                }
            });
        }
        return isValid;
    },

    // ------ User Groups ------ //

    userGroups: {
        list: {},
        init: function () {
            this.update(false);
        },
        update: function (updateOptions) {
            ide.triggerEvent(ide._events.LOAD, "User Groups");
            jQuery.hydrogen.invokeService({
                sdf: ide.sdf,
                uri: '//com.enginframe.user-group-manager/list.groups',
                dataType: 'xml',
                data: {
                    namespace: ide.plugin
                },
                success: function (xml) {
                    ide.userGroups.list = {};
                    jQuery(xml).find('ugm\\:group, group').each(function () {
                        var userGroup = jQuery(this).attr("name");
                        ide.userGroups.list[userGroup] = jQuery(this).attr("name");
                    });
                    ide.triggerEvent(ide._events.LOAD_DONE, "User Groups");
                    if (updateOptions) {
                        ide.updateAccessModifierOptions();
                    }
                },
                error: function () {
                    ide.triggerEvent(ide._events.LOAD_ERROR, "User Groups");
                }
            });
        }
    },

    updateAccessModifierOptions: function () {
        if (jQuery.isEmptyObject(ide.userGroups.list)) {
            ide.messages.alert(ide.messages.primary, "Error getting User Groups.");
        } else {
            jQuery('input[name="user-groups"]').each(function (index, userGroup) {
                var priority, aclPrefix, multiselectId, clearElem, selected;

                priority = jQuery(this).parent().parent().find('select[name="priority"]').val();
                aclPrefix = "deny-to";
                if (priority.indexOf('allow') > -1) {
                    aclPrefix = "allow-to";
                }

                multiselectId = 'ef-ugm-multiselect-' + index;
                if (jQuery('.ef-ugm-multiselect').length === 0) {
                    jQuery(userGroup).hide().parent().find('.ef-srv-edt-option-clear').before('<div class="ef-ugm-multiselect" id="' + multiselectId + '"></div>');
                    jQuery.each(ide.userGroups.list, function (key, value) {
                        jQuery('#' + multiselectId).append('<label><input type="checkbox" name="user-group-multiselect" value="' + aclPrefix + '[' + ide.plugin + ':' + value + ']" />' + key + '</label>');
                    });

                    selected = jQuery(userGroup).val().split(',');
                    jQuery.each(selected, function (key, value) {
                        jQuery('#' + multiselectId + ' input[value="' + value + '"]').prop("checked", true);
                    });

                    jQuery('#' + multiselectId + ' input').change(function () {
                        var userGroupList = [];
                        jQuery('#' + multiselectId + ' input:checkbox[name=user-group-multiselect]:checked').each(function (index) {
                            userGroupList[index] = jQuery(this).val();
                        });
                        jQuery(userGroup).val(userGroupList.join(',')).change();
                        ide.triggerEvent(ide._events.CHANGE, "User Groups");
                    });
                }
            });
        }

        // replace acl value according to priority value
        jQuery('select[name="priority"]').change(function () {
            var priority = jQuery(this).val();
            jQuery(this).parent().parent().find('.ef-ugm-multiselect input').each(function (index, userGroup) {
                var currentVal = jQuery(this).val();
                if (priority.indexOf('allow') > -1) {
                    jQuery(this).val(currentVal.replace('deny-to', 'allow-to')).change();
                } else {
                    jQuery(this).val(currentVal.replace('allow-to', 'deny-to')).change();
                }
            });
        });
    },

    // ------  Action Dialog ------ //

    drawActionDialog: function () {
        var dialog, button, dialogButtons;

        // Draw Dialog
        dialog = jQuery("#ef-srv-ide-action-editor");
        dialogButtons = {
            Close: function () {
                var isValid = true;
                // validate
                ide.messages.clear(ide.messages.actionEditor);
                jQuery.each(ide.service.components.action, function (innerComponent) {
                    if (!ide.service.components.validate(ide.service.components.action[innerComponent], ide.messages.actionEditor)) {
                        isValid = false;
                        return;
                    }
                });
                if (!isValid || !ide.service.components.validate(ide.service.components.conf, ide.messages.actionEditor)) {
                    return;
                }
                jQuery(this).dialog("close");
            }
        };
        dialog.dialog({
            title: "Edit Action",
            resizable: true,
            buttons: dialogButtons,
            modal: true,
            autoOpen: false,
            width: "900",
            height: "520",
            closeText: '',
            beforeClose: function () {
                // cleanup
                ide.texteditors.cleanup(ide.service.components.action.actionScript);
                ide.texteditors.cleanup(ide.service.components.action.jobScript);
                // show default Loading box
                jQuery('#ef-srv-ide-service-editor .ef-srv-ide-loading-wrapper').show();
            },
            resizeStop: function () {
                var newEditorHeight = dialog.innerHeight() - jQuery('.ef-srv-ide-action-editor-buttons').outerHeight() - 25;
                ide.texteditors.resize(ide.service.components.action.jobScript, { height: newEditorHeight });
                ide.texteditors.resize(ide.service.components.action.actionScript, { height: newEditorHeight });
                jQuery('#ef-srv-ide-conf-editor-content').width(dialog.innerWidth() - 40).handsontable('render');
            }
        });
        button = jQuery('button:contains(Close)', dialog.parent('div.ui-dialog'));
        button.addClass('ui-priority-primary');

        // Draw tabs buttonset
        jQuery('.ef-srv-ide-action-editor-buttons').controlgroup({
            items: {
                "button": "input[type=button], input[type=submit], input[type=reset], input[type='radio'], button, a",
                "controlgroupLabel": ".ui-controlgroup-label, input[type='text']",
                "checkboxradio": "input[type='checkbox']",
                "selectmenu": "select",
                "spinner": ".ui-spinner-input"
            }
        });
        jQuery('#ef-srv-ide-action-editor > div[class$="editor"]').hide();

        jQuery('.ef-srv-ide-action-editor-buttons').change(function () {
            jQuery('#ef-srv-ide-action-editor > div[class$="editor"]').hide();

            var mode = jQuery(this).children(':checked').val();
            jQuery("#ef-srv-ide-" + mode + "-editor").show();
            switch (mode) {
            case 'conf':
                jQuery('#ef-srv-ide-' + mode + '-editor-content').handsontable('render');
                break;
            case 'action-script':
                ide.texteditors.resize(ide.service.components.action.actionScript);
                break;
            case 'job-script':
                ide.texteditors.resize(ide.service.components.action.jobScript);
                break;
            case 'action-properties':
                jQuery('#ef-srv-ide-' + mode + '-editor-content').height(jQuery('#ef-srv-ide-action-editor').innerHeight() - jQuery('.ef-srv-ide-action-editor-buttons').outerHeight() - 20);
                break;
            default:
                break;
            }
        });
        jQuery('.ef-srv-ide-action-editor-buttons').click(function () {
            ide.texteditors.cleanup(ide.service.components.action.actionScript);
            ide.texteditors.cleanup(ide.service.components.action.jobScript);
        });

        // Draw action script editor
        ide.texteditors.draw(ide.service.components.action.actionScript, 'sh');

        // Draw job script editor
        ide.texteditors.draw(ide.service.components.action.jobScript, 'sh');

        // Draw configuration table
        ide.handsontables.conf.draw();

        // Bind Action attributes
        jQuery(ide.service.components.action.id.div).bind("keyup change", function () {
            jQuery('.ef-srv-edt-option-action').attr('opt-id', ide.service.components.getValue(ide.service.components.action.id));
            ide.service.components.validate(ide.service.components.action.id, ide.messages.actionEditor);
            ide.triggerEvent(ide._events.CHANGE, ide.service.components.action.id.name);
        });
        jQuery(ide.service.components.action.label.div).bind("keyup change", function () {
            jQuery('.ef-srv-edt-option-action').attr('opt-label', ide.service.components.getValue(ide.service.components.action.label));
            jQuery('.ef-srv-edt-option-action input').val(ide.service.components.getValue(ide.service.components.action.label));
            ide.triggerEvent(ide._events.CHANGE, ide.service.components.action.label.name);
        });
        if (ide.service.type === 'batch') {
            jQuery(ide.service.components.action.outputMode.div).change(function () {
                jQuery('.ef-srv-edt-option-action').attr('opt-output-mode', ide.service.components.getValue(ide.service.components.action.outputMode));
                ide.triggerEvent(ide._events.CHANGE, ide.service.components.action.outputMode.name);
            });
            jQuery(ide.service.components.action.resultType.div).change(function () {
                jQuery('.ef-srv-edt-option-action').attr('opt-result-type', ide.service.components.getValue(ide.service.components.action.resultType));
                ide.triggerEvent(ide._events.CHANGE, ide.service.components.action.resultType.name);
            });
        }
        jQuery(ide.service.components.action.os.div).change(function () {
            ide.drawClusters();
            ide.drawSessionModes();
            if (ide.service.components.getValue(ide.service.components.action.os) === "windows") {
                ide.texteditors.changeMode(ide.service.components.action.jobScript, "batchfile");
                ide.texteditors.enableAutocompletion(ide.service.components.action.jobScript, false);
            }
            else  {
                ide.texteditors.changeMode(ide.service.components.action.jobScript, "sh");
                ide.texteditors.enableAutocompletion(ide.service.components.action.jobScript, true);
            }
            ide.triggerEvent(ide._events.CHANGE, ide.service.components.action.os.name);
        });
        jQuery(ide.service.components.action.remote.div).change(function () {
            ide.drawDesktopManagers();
            ide.drawDisplayResolutions();
            // update remote in the list-of-resolutions widgets
            ide.triggerEvent(ide._events.LOAD, ide.service.components.core.name);
            jQuery(ide.service.components.core.div).trigger('updateOptions', ['remote', ide.service.components.getValue(ide.service.components.action.remote)]);
            ide.triggerEvent(ide._events.CHANGE, ide.service.components.action.remote.name);
        });
        jQuery(ide.service.components.action.cluster.div).change(function () {
            ide.drawRemotes();
            ide.drawSessionModes();
            ide.triggerEvent(ide._events.CHANGE, ide.service.components.action.cluster.name);
        });
        jQuery(ide.service.components.action.desktopManager.div).change(function () {
            ide.triggerEvent(ide._events.CHANGE, ide.service.components.action.desktopManager.name);
        });
        jQuery(ide.service.components.action.displayResolution.div).change(function () {
            ide.triggerEvent(ide._events.CHANGE, ide.service.components.action.displayResolution.name);
        });
    },

    texteditors: {
        draw: function (component, mode) {
            jQuery(component.div).texteditor({mode : mode, height: "375", width: "100%", autocompleter: this._autocompleter, toolbar: true, rowNumberEnabled: true});
            jQuery(component.div).bind('keyup', function () {
                ide.triggerEvent(ide._events.CHANGE, component.name);
            });
        },
        getContent: function (component) {
            var value = '';
            if (jQuery(component.div).children().length > 0) {
                value = jQuery(component.div).texteditor('getContent');
            }
            return value;
        },
        setContent: function (component, value) {
            jQuery(component.div).texteditor('setContent', value);
        },
        changeMode: function (component, mode) {
            jQuery(component.div).texteditor('changeMode', mode);
        },
        enableAutocompletion: function (component, value) {
            jQuery(component.div).texteditor('enableAutocompletion', value);
        },
        cleanup: function (component) {
            jQuery(component.div).texteditor('cleanup');
        },
        resize: function (component, size) {
            if (size) {
                jQuery(component.div).texteditor('resize', size);
            }
            else {
                jQuery(component.div).texteditor('resize');
            }
        },
        _autocompleter: {
            getCompletions: function (editor, session, pos, prefix, callback) {
                var outputList = [];
                if (prefix.length === 0) {
                    callback(null, []);
                    return;
                }
                // System variables
                var systemVars = ["EF_AGENT", "EF_BROWSER_NAME", "EF_BROWSER_VERSION", "EF_CONF_ROOT",
                                  "EF_DATA_ROOT", "EF_DOWNLOAD_URL", "EF_LOGIN_NAME", "EF_OUTPUT_MODE",
                                  "EF_ROOT", "EF_SERVICE_ID", "EF_SERVICE_NAME", "EF_SESSION",
                                  "EF_SPOOLER_NAME", "EF_SPOOLER", "EF_SPOOLER_OWNER", "EF_SPOOLER_PROJECT",
                                  "EF_SPOOLER_TYPE", "EF_SPOOLER_URI", "EF_USER", "QUERY_STRING",
                                  "REMOTE_ADDR", "REMOTE_HOST", "REQUEST_URL", "EF_CALLSERVICE_REQUEST",
                                  "EF_SCHEDULED_REQUEST", "EF_WEBSERVICE_REQUEST", "EF_TRIGGER_ID",
                                  "EF_TRIGGER_GROUP", "EF_TRIGGER_TIMES_FIRED", "EF_TRIGGER_IS_SYSTEM",
                                  "EF_TRIGGER_PREVIOUS_RUN_TIME", "EF_TRIGGER_CURRENT_RUN_TIME", "EF_ENTERPRISE_SERVERS"
                                  ];
                if (ide.service.type === 'interactive') {
                    systemVars = systemVars.concat(["VDI_OS", "VDI_CLUSTER", "VDI_REMOTE",
                                                    "VDI_DESKTOP_MANAGER", "VDI_GEOMETRY"]);
                }
                jQuery.each(systemVars, function (index, value) {
                    var bashValue = '${' + value + '}';
                    outputList.push({name: bashValue, value: bashValue, meta: "system-var"});
                });
                // options Ids
                jQuery.each(jQuery(ide.service.components.core.div).efserviceeditor('listId'), function (index, value) {
                    var bashValue = '${' + value + '}';
                    outputList.push({name: bashValue, value: bashValue, meta: "option-id"});
                });
                // configuration parameters
                jQuery.each(ide.handsontables.getContent(ide.service.components.conf), function (key, value) {
                    if (value[0]) {
                        var bashValue = '${' + value[0] + '}';
                        outputList.push({name: bashValue, value: bashValue, meta: "conf-parameter"});
                    }
                });
                // metadata
                jQuery.each(ide.handsontables.getContent(ide.service.components.metadata), function (key, value) {
                    if (value[0]) {
                        var bashValue = '${' + value[0] + '}';
                        outputList.push({name: bashValue, value: bashValue, meta: "metadata"});
                    }
                });
                callback(null, outputList);
            },
            getName: function () {
                return "efcompleter";
            }
        }
    },

    // ------ Settings Dialog ------ //

    drawSettingsDialog: function () {
        var dialog, button, dialogButtons, dialogWidth;

        // Draw Dialog
        dialog = jQuery("#ef-srv-ide-service-settings-editor");
        dialogButtons = {
            Close: function () {
                ide.messages.clear(ide.messages.settings);
                if (!ide.service.components.validate(ide.service.components.metadata, ide.messages.settings) ||
                        !ide.service.components.validate(ide.service.components.spoolerTTL, ide.messages.settings) ||
                        !ide.service.components.validate(ide.service.components.sessionClass, ide.messages.settings) ||
                        !ide.service.components.validate(ide.service.components.maxSessions, ide.messages.settings) ||
                        !ide.validateFileAction(ide.messages.settings)) {
                    return;
                }
                // cleanup
                ide.texteditors.cleanup(ide.service.components.serviceJs);
                ide.texteditors.cleanup(ide.service.components.serviceCss);
                jQuery(this).dialog("close");
            }
        };
        dialog.dialog({
            title: "Service Settings",
            resizable: false,
            buttons: dialogButtons,
            modal: true,
            autoOpen: false,
            width: ide.plugin === 'applications' ? '770' : '520',
            closeText: '',
            beforeClose: function () {
                ide.texteditors.cleanup(ide.service.components.serviceJs);
                ide.texteditors.cleanup(ide.service.components.serviceCss);
            },
            resizeStop: function () {
                var newEditorHeight = dialog.innerHeight() - jQuery('.ef-srv-ide-action-editor-buttons').outerHeight() - 25;
                ide.texteditors.resize(ide.service.components.serviceJs, { height: newEditorHeight });
                ide.texteditors.resize(ide.service.components.serviceCss, { height: newEditorHeight });
                jQuery(ide.service.components.metadata.div).handsontable('render');
            }
        });
        button = jQuery('button:contains(Close)', dialog.parent('div.ui-dialog'));
        button.addClass('ui-priority-primary');

        // Draw tabs controlgroup
        jQuery('.ef-srv-ide-service-settings-editor-buttons').controlgroup({
            items: {
                "button": "input[type=button], input[type=submit], input[type=reset], input[type='radio'], button, a",
                "controlgroupLabel": ".ui-controlgroup-label, input[type='text']",
                "checkboxradio": "input[type='checkbox']",
                "selectmenu": "select",
                "spinner": ".ui-spinner-input"
            }
        });
        jQuery('#ef-srv-ide-service-settings-editor > div[class$="editor"]').hide();

        jQuery('.ef-srv-ide-service-settings-editor-buttons').change(function () {
            jQuery('#ef-srv-ide-service-settings-editor > div[class$="editor"]').hide();

            var mode = jQuery(this).children(':checked').val();
            jQuery("#ef-srv-ide-" + mode + "-editor").show();
            switch (mode) {
            case 'service-js':
                ide.texteditors.resize(ide.service.components.serviceJs);
                break;
            case 'service-css':
                ide.texteditors.resize(ide.service.components.serviceCss);
                break;
            case 'attributes':
                jQuery(ide.service.components.metadata.div).handsontable('render');
                break;
            case 'file-action':
                ide.validateFileAction(ide.messages.settings);
                break;
            default:
                break;
            }
        });
        jQuery('.ef-srv-ide-service-settings-editor-buttons').click(function () {
            ide.texteditors.cleanup(ide.service.components.serviceJs);
            ide.texteditors.cleanup(ide.service.components.serviceCss);
        });

        // Draw service javascript editor
        ide.texteditors.draw(ide.service.components.serviceJs, 'javascript');

        // Draw service css editor
        ide.texteditors.draw(ide.service.components.serviceCss, 'css');

        // Draw service metadata editor
        ide.handsontables.metadata.draw();

        // Draw file action
        if (ide.plugin === 'applications') {
            jQuery(ide.service.components.fileAction.enabled.div).change(function () {
                if (ide.service.components.getValue(ide.service.components.fileAction.enabled) === "false") {
                    ide.service.components.setValue(ide.service.components.fileAction.showMenuOnly, "false");
                }
                ide.drawFileAction();
                ide.drawEmbeddableAttr();
                ide.triggerEvent(ide._events.CHANGE, ide.service.components.fileAction.enabled.name);
            });
            jQuery(ide.service.components.fileAction.label.div).bind("keyup change", function () {
                ide.triggerEvent(ide._events.CHANGE, ide.service.components.fileAction.label.name);
            });
            jQuery(ide.service.components.fileAction.serviceExecution.div).change(function () {
                ide.triggerEvent(ide._events.CHANGE, ide.service.components.fileAction.serviceExecution.name);
            });
            jQuery(ide.service.components.fileAction.showMenuOnly.div).change(function () {
                ide.drawHiddenAttr();
                ide.triggerEvent(ide._events.CHANGE, ide.service.components.fileAction.showMenuOnly.name);
            });
            jQuery(ide.service.components.fileAction.optionId.div).change(function () {
                ide.triggerEvent(ide._events.CHANGE, ide.service.components.fileAction.optionId.name);
            });
            jQuery(ide.service.components.fileAction.fileFilter.div).bind("keyup change", function () {
                ide.triggerEvent(ide._events.CHANGE, ide.service.components.fileAction.fileFilter.name);
            });
        }

        // Bind attributes
        if (ide.plugin === 'applications') {
            if (ide.service.type === 'batch') {
                jQuery(ide.service.components.embeddableAttr.div).change(function () {
                    if (ide.service.components.getValue(ide.service.components.embeddableAttr) === "true") {
                        jQuery(".ef-srv-ide-attrs-embeddable-service-warning").removeClass("ui-helper-hidden");
                    } else {
                        jQuery(".ef-srv-ide-attrs-embeddable-service-warning").addClass("ui-helper-hidden");
                    }
                    ide.drawFileAction();
                    ide.drawHiddenAttr();
                    ide.drawResubmitAttr();
                    ide.drawReuseSpoolerAttr();
                    ide.drawResetTTLAttr();
                    ide.drawProfileAttr();
                    ide.triggerEvent(ide._events.CHANGE, ide.service.components.embeddableAttr.name);
                });
            }

            jQuery(ide.service.components.resubmitAttr.div).change(function () {
                ide.drawReuseSpoolerAttr();
                ide.drawResetTTLAttr();
                ide.triggerEvent(ide._events.CHANGE, ide.service.components.resubmitAttr.name);
            });
            jQuery(ide.service.components.reuseSpoolerAttr.div).change(function () {
                ide.drawResetTTLAttr();
                ide.triggerEvent(ide._events.CHANGE, ide.service.components.reuseSpoolerAttr.name);
            });
            jQuery(ide.service.components.resetTTLAttr.div).change(function () {
                ide.triggerEvent(ide._events.CHANGE, ide.service.components.resetTTLAttr.name);
            });
            jQuery(ide.service.components.profile.div).change(function () {
                ide.triggerEvent(ide._events.CHANGE, ide.service.components.profile.name);
            });

            ide.drawSpoolerTTL();
        }
        jQuery(ide.service.components.sessionClass.div).bind("keyup change", function () {
            ide.triggerEvent(ide._events.CHANGE, ide.service.components.sessionClass.name);
        });
        jQuery(ide.service.components.maxSessions.div).bind("keyup change", function () {
            ide.triggerEvent(ide._events.CHANGE, ide.service.components.maxSessions.name);
        });
    },

    drawEmbeddableAttr: function () {
        if (ide.service.components.getValue(ide.service.components.fileAction.enabled) === "true") {
            ide.service.components.disable(ide.service.components.embeddableAttr);
            jQuery(".ef-srv-ide-attrs-embeddable-service-disabled-warning").removeClass("ui-helper-hidden");
        }
        else {
            ide.service.components.enable(ide.service.components.embeddableAttr);
            jQuery(".ef-srv-ide-attrs-embeddable-service-disabled-warning").addClass("ui-helper-hidden");
        }
    },

    drawHiddenAttr: function () {
        if (ide.service.components.getValue(ide.service.components.embeddableAttr) === "true" || ide.service.components.getValue(ide.service.components.fileAction.showMenuOnly) === "true") {
            ide.service.components.setValue(ide.service.components.hiddenAttr, "true");
        } else {
            ide.service.components.setValue(ide.service.components.hiddenAttr, "false");
        }
    },

    drawResubmitAttr: function () {
        if (ide.service.components.getValue(ide.service.components.spoolerTTL) === '-1') {
            ide.service.components.disable(ide.service.components.resubmitAttr, false);
        }
        else {
            if (ide.service.components.getValue(ide.service.components.embeddableAttr) === "true") {
                ide.service.components.disable(ide.service.components.resubmitAttr, false);
            } else {
                ide.service.components.enable(ide.service.components.resubmitAttr);
            }
        }
    },

    drawReuseSpoolerAttr: function () {
        if (ide.service.components.getValue(ide.service.components.resubmitAttr) === "true") {
            ide.service.components.enable(ide.service.components.reuseSpoolerAttr);
        } else {
            ide.service.components.disable(ide.service.components.reuseSpoolerAttr, false);
        }
    },

    drawResetTTLAttr: function () {
        if (ide.service.components.getValue(ide.service.components.reuseSpoolerAttr) === "true") {
            ide.service.components.enable(ide.service.components.resetTTLAttr);
        } else {
            ide.service.components.disable(ide.service.components.resetTTLAttr, false);
        }
    },

    drawProfileAttr: function () {
        if (ide.service.components.getValue(ide.service.components.embeddableAttr) === "true") {
            ide.service.components.disable(ide.service.components.profile, false);
        } else {
            ide.service.components.enable(ide.service.components.profile);
        }
    },

    // Spooler TTL

    ttlValues: [
         // { v: "-1", label: "No spooler" },
        { v: "0", label: "Immediately" },
        { v: "1s", label: "1 second" },
        { v: "1m", label: "1 minute" },
        { v: "30m", label: "30 minutes" },
        { v: "1h", label: "1 hour" },
        { v: "6h", label: "6 hours" },
        { v: "12h", label: "12 hours" },
        { v: "1d", label: "1 day" },
        { v: "7d", label: "7 days" },
        { v: "15d", label: "15 days" },
        { v: "30d", label: "30 days" },
        { v: "180d", label: "180 days" },
        { v: "365d", label: "365 days" },
        { v: "forever", label: "Does not expire" }
    ],

    drawSpoolerTTL: function () {
        // Slider
        jQuery(".ef-srv-ide-service-attributes-ttl-slider").slider({
            range: "min",
            min: 0,
            max: 13,
            value: 0,
            slide: function (event, ui) {
                jQuery(ide.service.components.spoolerTTL.div).text(ide.ttlValues[ui.value].label);
            },
            change: function (event, ui) {
                jQuery(ide.service.components.spoolerTTL.div).text(ide.ttlValues[ui.value].label);
                ide.messages.clear(ide.messages.settings);
                ide.service.components.validate(ide.service.components.spoolerTTL, ide.messages.settings);
                ide.triggerEvent(ide._events.CHANGE, ide.service.components.spoolerTTL.name);
            }
        });
        ide.service.components.setValue(ide.service.components.spoolerTTL, '0');

        // Spooler presence checkbox
        jQuery('#ef-srv-ide-service-attributes-spooler-presence').change(function () {
            if (jQuery(this).prop('checked')) {
                jQuery(".ef-srv-ide-service-attributes-ttl-slider").slider("enable");
                jQuery(".ef-srv-ide-service-attributes-ttl-slider-table").parent().removeClass("ef-srv-ide-text-disabled");
                jQuery(".ef-srv-ide-service-spooler-presence-warning").addClass("ui-helper-hidden");
            }
            else {
                jQuery(".ef-srv-ide-service-attributes-ttl-slider").slider("disable");
                jQuery(".ef-srv-ide-service-attributes-ttl-slider-table").parent().addClass("ef-srv-ide-text-disabled");
                jQuery(".ef-srv-ide-service-spooler-presence-warning").removeClass("ui-helper-hidden");
            }
            ide.messages.clear(ide.messages.settings);
            ide.service.components.validate(ide.service.components.spoolerTTL, ide.messages.settings);
            ide.drawResubmitAttr();
            ide.triggerEvent(ide._events.CHANGE, ide.service.components.spoolerTTL.name);
        });
    },

    // ------ File/spooler actions ----- //

    drawFileAction: function () {
        if (ide.service.components.getValue(ide.service.components.embeddableAttr) === "true") {
            jQuery(".ef-srv-ide-file-action-embeddable-warning").removeClass("ui-helper-hidden");
            jQuery(".ef-srv-ide-file-action-rfb-warning").addClass("ui-helper-hidden");
            ide.service.components.disable(ide.service.components.fileAction.enabled);
        }
        else {
            jQuery(".ef-srv-ide-file-action-embeddable-warning").addClass("ui-helper-hidden");
            if (!jQuery.isEmptyObject(ide.service.rfbList.value)) {
                jQuery(".ef-srv-ide-file-action-rfb-warning").addClass("ui-helper-hidden");
                ide.service.components.enable(ide.service.components.fileAction.enabled);
            }
            else {
                jQuery(".ef-srv-ide-file-action-rfb-warning").removeClass("ui-helper-hidden");
                // Do not disable file action enabled component if it is checked
                if (ide.service.components.getValue(ide.service.components.fileAction.enabled) !== "true") {
                    ide.service.components.disable(ide.service.components.fileAction.enabled);
                }
            }
        }

        if (ide.service.components.getValue(ide.service.components.fileAction.enabled) === "true") {
            ide.service.components.enable(ide.service.components.fileAction.label);
            ide.service.components.enable(ide.service.components.fileAction.showMenuOnly);
            ide.service.components.enable(ide.service.components.fileAction.optionId);
            ide.service.components.enable(ide.service.components.fileAction.fileFilter);
            ide.service.components.enable(ide.service.components.fileAction.serviceExecution);
            ide.drawFileActionOptionId();
        } else {
            ide.service.components.disable(ide.service.components.fileAction.label);
            ide.service.components.disable(ide.service.components.fileAction.showMenuOnly);
            ide.service.components.disable(ide.service.components.fileAction.optionId);
            ide.service.components.disable(ide.service.components.fileAction.fileFilter);
            ide.service.components.disable(ide.service.components.fileAction.serviceExecution);
        }
    },

    drawFileActionOptionId: function () {
        var fileActionHtmlId, foundId, selectDiv;

        fileActionHtmlId = ide.service.components.getValue(ide.service.components.fileAction.optionId);
        selectDiv = ide.service.components.fileAction.optionId.div;
        foundId = false;
        jQuery(selectDiv).empty();

        // populate select
        jQuery.each(ide.service.rfbList.value, function (key, value) {
            jQuery(selectDiv).append('<option value="' + key + '">' + value + '</option>');
            if (fileActionHtmlId === key || fileActionHtmlId === '') {
                jQuery(selectDiv).val(key).change();
                foundId = true;
                ide.messages.clear(ide.messages.primary);
            }
        });

        if (!foundId && fileActionHtmlId && !ide.status.isInitState()) {
            if (jQuery(selectDiv).val()) {
                ide.messages.alert(ide.messages.primary, "File Action Option Id has been set to (" + efEncodeHtml(ide.service.rfbList.getOptionId(jQuery(selectDiv).val())) + ")");
            }
            else {
                ide.messages.alert(ide.messages.primary, "You must have at least one RFB option to use File Action feature. Add it or disable this feature." );
            }
            jQuery(selectDiv).click(function () {
                jQuery(selectDiv).change();
                ide.messages.clear(ide.messages.primary);
            });
        }
    },

    validateFileAction: function (ideMessage) {
        var isValid = true;

        if (ide.service.components.getValue(ide.service.components.fileAction.enabled) === 'true') {
            if (jQuery.isEmptyObject(ide.service.rfbList.value)) {
                ide.messages.alert(ideMessage, "You must have at least one RFB option to use File Action feature. Add it or disable this feature." );
                isValid = false;
            }
        }
        return isValid;
    },

    // ------ handson tables ------ //

    handsontables: {
        metadata: {
            draw: function () {
                jQuery(ide.service.components.metadata.div).handsontable({
                    contextMenu : [ 'row_above', 'row_below', 'hsep1', 'remove_row' ],
                    columns : [ {
                            title : 'Name',
                            validator: ide.handsontables._nameValidator,
                            allowInvalid: true
                        }, {
                            title : 'Value',
                            validator: ide.handsontables._valueValidator,
                            allowInvalid: true
                        }
                    ],
                    stretchH : 'all',
                    minSpareRows : 1,
                    minRows : 5,
                    fillHandle : 'vertical',
                    colWidths : [ 100, 140 ],
                    width: 240,
                    currentRowClassName : 'currentRow',
                    beforeChange: ide.handsontables._uppercaseValue,
                    afterChange: ide.handsontables.metadata._validate,
                    afterRemoveRow: ide.handsontables.metadata._validate,
                    afterValidate: ide.handsontables._render
                });
            },
            _validate: function (changes, source) {
                this.validateCells(function () {});
                ide.messages.clear(ide.messages.settings);
                ide.service.components.validate(ide.service.components.metadata, ide.messages.settings);
                ide.triggerEvent(ide._events.CHANGE, ide.service.components.metadata.name);
            }
        },
        conf: {
            draw: function () {
                jQuery('#ef-srv-ide-conf-editor-content').handsontable({
                    contextMenu: ["row_above", "row_below", "hsep1", "remove_row"],
                    columns: [ {
                            title: "Name",
                            validator: ide.handsontables._nameValidator,
                            allowInvalid: true
                        }, {
                            title: "Value",
                            validator: ide.handsontables._valueValidator,
                            allowInvalid: true
                        }, {
                            title: "Comments"
                        }
                    ],
                    stretchH: 'all',
                    minSpareRows: 1,
                    minRows: 15,
                    fillHandle: "vertical",
                    colWidths: [25, 42, 42],
                    width: 860,
                    beforeChange: ide.handsontables._uppercaseValue,
                    afterChange: ide.handsontables.conf._validate,
                    afterRemoveRow: ide.handsontables.conf._validate,
                    afterValidate: ide.handsontables._render
                });
            },
            fileToJson: function (confFile) { // Convert properties file to json object
                var comment, property, value, lines, jsonData, commentIndex;

                jsonData = [];
                lines = confFile.split('\n');
                commentIndex = 0;
                jQuery.each(lines, function (index, row) {

                    if (row.match('^#')) {
                        comment = row.replace(/#/g, '');
                        commentIndex = index;
                    }
                    if (row.match('^[A-Za-z0-9-_=]')) {
                        property = row.substring(0, row.indexOf('=')).replace(/\s/g, '');
                        value = row.substring(row.indexOf('=') + 1);
                        if (commentIndex !== index - 1) {
                            comment = '';
                        }
                        if (!value) {
                            value = '';
                        }
                        jsonData.push({"property" : property, "value" : value, "comment" : comment});
                        //alert(JSON.stringify(jsonData));
                    }
                });
                return jsonData;
            },
            _validate: function (changes, source) {
                this.validateCells(function () {});
                ide.messages.clear(ide.messages.actionEditor);
                ide.service.components.validate(ide.service.components.conf, ide.messages.actionEditor);
                ide.triggerEvent(ide._events.CHANGE, ide.service.components.conf.name);
            }
        },
        getContent: function (component) {
            var value = '';
            if (jQuery(component.div).children().length > 0) {
                value = jQuery(component.div).data('handsontable').getData();
            }
            return value;
        },
        setContent: function (component, data) {
            jQuery(component.div).data('handsontable').loadData(data);
        },
        _nameValidator: function (value, callback) {
            var error = false;
            if (value) {
                if (/^[a-zA-Z_][a-zA-Z0-9_]*$/.test(value) && !/^(SM_|VDI_|APPLICATIONS_|INTERACTIVE_|EF_)/.test(value)) {
                    var array = this.instance.getDataAtCol(this.col);
                    for (var i = 0; i < array.length; i++) {
                        if (array[i] === value.toUpperCase() && i !== this.row) {
                            error = true;
                        }
                    }
                }
                else {
                    error = true;
                }
            }
            callback(!error);
        },
        _valueValidator: function (value, callback) {
            var valid = false;
            var name = this.instance.getDataAtCell(this.row, 0);
            if (name || !value) {
                valid = true;
            }
            callback(valid);
        },
        _uppercaseValue: function (changes, source) {
            for (var i = changes.length - 1; i >= 0; i--) {
                if (changes[i][1] === 0 && changes[i][3].charAt(0)) {
                    changes[i][3] = changes[i][3].toUpperCase();
                }
            }
        },
        _render: function (changes, source) {
            this.render();
        },
        validate: function (tableId, tableName, ideMessage) {
            var message, tableValues, isValid = true;

            tableValues = jQuery(tableId).data('handsontable').getData();
            jQuery.each(tableValues, function (key, columns) {
                if (columns[0]) {
                    if (/^(SM_|VDI_|APPLICATIONS_|INTERACTIVE_|EF_)/.test(columns[0])) {
                        message = tableName + " Name with prefix '" + efEncodeHtml(columns[0].substring(0, 3)) + "' is not allowed.";
                        ide.messages.alert(ideMessage, message);
                        isValid = false;
                        return false;
                    }
                    if (!/^[a-zA-Z_][a-zA-Z0-9_]*$/.test(columns[0])) {
                        message = tableName + " '" + efEncodeHtml(columns[0]) + "' contains invalid characters.";
                        ide.messages.alert(ideMessage, message);
                        isValid = false;
                        return false;
                    }
                    jQuery.each(tableValues, function (key2, columns2) {
                        if (columns[0] === columns2[0] && key !== key2) {
                            message = "Cannot have two " + tableName + " with the same Name '" + efEncodeHtml(columns[0]) + "'.";
                            ide.messages.alert(ideMessage, message);
                            isValid = false;
                            return false;
                        }
                    });
                    if (!isValid) {
                        return false;
                    }
                }
                else {
                    if (columns[1]) {
                        message = efEncodeHtml(tableName) + " with Value '" + efEncodeHtml(columns[1]) + "' cannot have empty Name.";
                        ide.messages.alert(ideMessage, message);
                        isValid = false;
                        return false;
                    }
                }
            });
            return isValid;
        }
    },

    // ------ Utility functions ------ //

    saveFiles: function () {
        var serviceData = {
            serviceJson: ide.service.components.getValue(ide.service.components.core),
            actionScript: ide.service.components.getValue(ide.service.components.action.actionScript),
            jobScript: ide.service.components.getValue(ide.service.components.action.jobScript),
            conf: ide.service.components.getValue(ide.service.components.conf),
            serviceId: ide.service.id,
            serviceName: ide.service.name.value,
            templateId: ide.service.templateId,
            editorId: ide.editorId,
            serviceType: ide.service.type,
            plugin: ide.plugin,
            metadata: ide.service.components.getValue(ide.service.components.metadata),
            serviceJs: ide.service.components.getValue(ide.service.components.serviceJs),
            serviceCss: ide.service.components.getValue(ide.service.components.serviceCss)
        };

        const cluster = ide.service.components.getValue(ide.service.components.action.cluster);
        const jobManager = cluster.split(":")[1];

        if (ide.service.type === 'interactive') {
            serviceData.sessionClass = ide.service.components.getValue(ide.service.components.sessionClass);
            serviceData.maxSessions = ide.service.components.getValue(ide.service.components.maxSessions);
            serviceData.actionOS = ide.service.components.getValue(ide.service.components.action.os);
            if (jobManager === 'dcvsm') {
                serviceData.actionSessionMode = ide.service.components.getValue(ide.service.components.action.sessionMode);
            }
            serviceData.actionRemote = ide.service.components.getValue(ide.service.components.action.remote);
            serviceData.actionCluster = ide.service.components.getValue(ide.service.components.action.cluster);
            serviceData.actionDesktopManager = ide.service.components.getValue(ide.service.components.action.desktopManager);
            serviceData.actionDisplayResolution = ide.service.components.getValue(ide.service.components.action.displayResolution);
        }
        else {
            serviceData.serviceEmbeddable = ide.service.components.getValue(ide.service.components.embeddableAttr);
        }
        if (ide.plugin === 'applications') {
            serviceData.serviceProfile = ide.service.components.getValue(ide.service.components.profile);
            serviceData.serviceResubmit = ide.service.components.getValue(ide.service.components.resubmitAttr);
            serviceData.resubmitReuseSpooler = ide.service.components.getValue(ide.service.components.reuseSpoolerAttr);
            serviceData.resubmitResetTTL = ide.service.components.getValue(ide.service.components.resetTTLAttr);
            serviceData.spoolerTtl = ide.service.components.getValue(ide.service.components.spoolerTTL);
            serviceData.serviceHidden = ide.service.components.getValue(ide.service.components.hiddenAttr);
            serviceData.fileActionEnabled = ide.service.components.getValue(ide.service.components.fileAction.enabled);
            serviceData.fileActionLabel = ide.service.components.getValue(ide.service.components.fileAction.label);
            serviceData.fileActionServiceExecution = ide.service.components.getValue(ide.service.components.fileAction.serviceExecution);
            serviceData.fileActionFileFilter = ide.service.components.getValue(ide.service.components.fileAction.fileFilter);
            serviceData.fileActionOptionId = ide.service.rfbList.getOptionId(ide.service.components.getValue(ide.service.components.fileAction.optionId));
        }

        ide.triggerEvent(ide._events.SAVE, 'Service Editor');
        jQuery.hydrogen.invokeService({
            sdf: ide.sdf,
            uri: ide.serviceManagerUri + 'save.service',
            data: serviceData,
            dataType: "xml",
            complete: function () {
            },
            success: function (xml) {
                // retrieve service Id from the SDF returned by the save service
                ide.service.id = jQuery(xml).find('ef\\:service, service').attr("id");
                // do not trigger LOAD_DONE event for the 'Service Editor' component to keep the current SAVING state
                ide.initServiceEditor();

                ide.service.lock.set();
            },
            error: function () {
                ide.triggerEvent(ide._events.LOAD_ERROR, 'Service Editor');
            }
        });
    },

    // ------ Interactive Editor services ------ //

    OSs: {},

    drawOSs: function () {
        ide.triggerEvent(ide._events.LOAD, ide.service.components.action.os.name);
        jQuery.hydrogen.invokeService({
            sdf: ide.sdf,
            uri: ide.serviceManagerUri + "list.session.os",
            data: {
                plugin: ide.plugin
            },
            dataType: "xml",
            success: function (xml) {
                ide.OSs = {};
                jQuery(xml).find('ef\\:session-os, session-os').each(function (index) {
                    ide.OSs[jQuery(this).attr("id")] = jQuery(this).text();
                });
                ide.updateSelect(ide.OSs, ide.service.components.action.os);
                ide.triggerEvent(ide._events.LOAD_DONE, ide.service.components.action.os.name);
            },
            error: function () {
                ide.triggerEvent(ide._events.LOAD_ERROR, ide.service.components.action.os.name);
            }
        });
    },

    sessionModes: {},

    drawSessionModes: function () {
        const cluster = ide.service.components.getValue(ide.service.components.action.cluster);
        const jobManager = cluster.split(":")[1];

        if (jobManager === 'dcvsm') {
            const os = ide.service.components.getValue(ide.service.components.action.os);
            ide.triggerEvent(ide._events.LOAD, ide.service.components.action.sessionMode.name);
            jQuery.hydrogen.invokeService({
                sdf: ide.sdf,
                uri: ide.serviceManagerUri + "list.session.modes",
                data: {
                    plugin: ide.plugin,
                    os: os ? os : ""
                },
                dataType: "xml",
                success: function (xml) {
                    ide.sessionModes = {};
                    jQuery(xml).find('ef\\:session-mode, session-mode').each(function (index) {
                        ide.sessionModes[jQuery(this).attr("id")] = jQuery(this).text();
                    });
                    ide.updateSelect(ide.sessionModes, ide.service.components.action.sessionMode);
                    ide.triggerEvent(ide._events.LOAD_DONE, ide.service.components.action.sessionMode.name);
                },
                error: function () {
                    ide.triggerEvent(ide._events.LOAD_ERROR, ide.service.components.action.sessionMode.name);
                }
            });

            jQuery('.ef-srv-ide-action-session-mode').show();
            ide.service.components.enable(ide.service.components.action.sessionMode);
        } else {
            jQuery('.ef-srv-ide-action-session-mode').hide();
            // Disable and reset the session mode to the saved value
            ide.service.components.disable(ide.service.components.action.sessionMode, ide.service.components.action.sessionMode.savedValue);
        }
    },

    sessionClusters: {},

    drawClusters: function () {
        var os = ide.service.components.getValue(ide.service.components.action.os);

        ide.triggerEvent(ide._events.LOAD, ide.service.components.action.cluster.name);
        jQuery.hydrogen.invokeService({
            sdf: ide.sdf,
            uri: ide.serviceManagerUri + "list.session.clusters",
            data: {
                plugin: ide.plugin,
                os: os ? os : ""
            },
            dataType: "xml",
            success: function (xml) {
                ide.sessionClusters = {};
                jQuery(xml).find('ef\\:session-cluster, session-cluster').each(function (index) {
                    ide.sessionClusters[jQuery(this).attr("id") + ":" + jQuery(this).attr("type")] = jQuery(this).text();
                });
                ide.updateSelect(ide.sessionClusters, ide.service.components.action.cluster);
                ide.triggerEvent(ide._events.LOAD_DONE, ide.service.components.action.cluster.name);
            },
            error: function () {
                ide.triggerEvent(ide._events.LOAD_ERROR, ide.service.components.action.cluster.name);
            }
        });
    },

    remotes: {},

    drawRemotes: function () {
        var os = ide.service.components.getValue(ide.service.components.action.os);
        var cluster = ide.service.components.getValue(ide.service.components.action.cluster);
        var grid = cluster.split(":")[1];

        ide.triggerEvent(ide._events.LOAD, ide.service.components.action.remote.name);
        jQuery.hydrogen.invokeService({
            sdf: ide.sdf,
            uri: ide.serviceManagerUri + "list.session.remotes",
            data: {
                plugin: ide.plugin,
                os: os ? os : "",
                grid: grid ? grid : ""
            },
            dataType: "xml",
            success: function (xml) {
                ide.remotes = {};
                jQuery(xml).find('ef\\:session-remote, session-remote').each(function (index) {
                    ide.remotes[jQuery(this).attr("id")] = jQuery(this).text();
                });
                ide.updateSelect(ide.remotes, ide.service.components.action.remote);
                ide.triggerEvent(ide._events.LOAD_DONE, ide.service.components.action.remote.name);
            },
            error: function () {
                ide.triggerEvent(ide._events.LOAD_ERROR, ide.service.components.action.remote.name);
            }
        });
    },

    desktopManagers: {},

    drawDesktopManagers: function () {
        ide.triggerEvent(ide._events.LOAD, ide.service.components.action.desktopManager.name);
        jQuery.hydrogen.invokeService({
            sdf: ide.sdf,
            uri: ide.serviceManagerUri + "list.session.desktop.managers",
            data: {
                plugin: ide.plugin
            },
            dataType: "xml",
            success: function (xml) {
                ide.desktopManagers = {};
                jQuery(xml).find('ef\\:session-desktop-manager, session-desktop-manager').each(function (index) {
                    ide.desktopManagers[jQuery(this).attr("id")] = jQuery(this).text();
                });
                ide.updateSelect(ide.desktopManagers, ide.service.components.action.desktopManager);
                ide.triggerEvent(ide._events.LOAD_DONE, ide.service.components.action.desktopManager.name);
            },
            error: function () {
                ide.triggerEvent(ide._events.LOAD_ERROR, ide.service.components.action.desktopManager.name);
            }
        });

        jQuery('.ef-srv-ide-remote-properties').show();
        if (ide.service.components.getValue(ide.service.components.action.remote) === 'dcv2sm') {
            // Disable and reset the display resolution to the saved value
                ide.service.components.disable(ide.service.components.action.displayResolution, ide.service.components.action.displayResolution.savedValue);
            }

        if (ide.service.components.getValue(ide.service.components.action.os) === 'windows') {
            // Disable and reset the desktop manager to the saved value
                ide.service.components.disable(ide.service.components.action.desktopManager, ide.service.components.action.desktopManager.savedValue);
        }
        else {
            // linux
                if (ide.service.components.getValue(ide.service.components.action.remote) === 'dcv2sm') {
                    // Disable and reset the desktop manager to the saved value
                ide.service.components.disable(ide.service.components.action.desktopManager, ide.service.components.action.desktopManager.savedValue);
            }
            else {
              ide.service.components.enable(ide.service.components.action.desktopManager);
            }
        }
    },

    displayResolutions: {},

    drawDisplayResolutions: function () {
        if (ide.service.components.getValue(ide.service.components.action.remote) === 'dcv2sm') {
            // Disable and reset the display resolution to the saved value
            ide.service.components.disable(ide.service.components.action.displayResolution, ide.service.components.action.displayResolution.savedValue);
        }
        else {
            // Initialize Display Resolution selector
            if (jQuery(ide.service.components.action.displayResolution.div).css('display') === "none") {
                // revert case, we need to destroy the select before re-init it
                jQuery(".hy-interactive-geometry-preset-selector").remove();
                jQuery(".hy-interactive-geometry-details").remove();
            }
            jQuery(ide.service.components.action.displayResolution.div).interactiveResolutionExtended(ide.service.components.getValue(ide.service.components.action.remote));
            jQuery('.hy-interactive-geometry-preset-selector.ef-srv-ide-action-properties,' +
                       'input[name^="ef-srv-ide-action-display-resolution"]').bind("keyup change", function () {
                ide.triggerEvent(ide._events.CHANGE, ide.service.components.action.displayResolution.name);
            });

            ide.service.components.enable(ide.service.components.action.displayResolution);
        }
    },

    updateSelect: function (listObj, component) {
        var previousValue, foundId, element;

        previousValue = '';
        if (ide.status.isInitState()) {
            // we are in an initializing state
            previousValue = component.savedValue;
            ide._log("[" + component.name + "] -> previous selected value is the initial savedValue (" + previousValue + ")");
        }
        else {
            // we already initialize the select
            previousValue = ide.service.components.getValue(component);
            ide._log("[" + component.name + "] -> previous selected value is (" + previousValue + ")");
        }

        foundId = false;
        element = jQuery(component.div);
        element.empty();
        jQuery.each(listObj, function (key, val) {
            // append OS icon
            // FIXME not cross-browser compatible, see https://bugs.chromium.org/p/chromium/issues/detail?id=568116
            // if (component.div === ide.service.components.action.os.div) {
            //     if (key === "windows") {
            //         val = "&#xf17a; " + val;
            //     }
            //     else if (key === "linux") {
            //         val = "&#xf17c; " + val;
            //     }
            // }
            element.append('<option value="' + key + '">' + val + '</option>');
            ide._log("[" + component.name + "] -> append option: key (" + key + "), value (" + val + ")");
            if (previousValue === key) {
                ide.service.components.setValue(component, key);
                ide._log("[" + component.name + "] -> previous selected value found.");
                foundId = true;
            }
        });

        if (!foundId) {
            ide._log("[" + component.name + "] -> previous selected value not found.");
            if (element.children().length > 0) {
                ide.service.components.setValue(component, element.val());
            }
            else {
                element.append('<option value="--">--</option>');
                ide.service.components.setValue(component, "--");
            }
        }
        ide._log("[" + component.name + "] -> current selected value is (" + element.val() + ")");
    }
};

/*global DetectZoom, screen, alert*/
(function ($) {
    $.fn.interactiveResolutionExtended = function (remote) {
        return this.each(function () {

            var self, presets, detailsRow, id, devicePxPerCssPx, width, height,
                widthEl, heightEl, fullScreenEl, allMonitorsEl, allMonitorsRow,
                defaultGeometry, autoGeometry;

            function customResolution() {
                var w, h, fs, am, regExpr;
                regExpr = new RegExp("^[1-9][0-9]*$");

                w = $.trim(widthEl.val());
                if (!regExpr.test(w)) {
                    alert("Invalid display width.");
                    w = width;
                    widthEl.val(w);
                }

                h = $.trim(heightEl.val());
                if (!regExpr.test(h)) {
                    alert("Invalid display height.");
                    h = height;
                    heightEl.val(h);
                }

                fs = fullScreenEl.prop("checked");
                if (fs === true) {
                    // Do not use show() since it add "display: block" breaking the tableWrapper
                    allMonitorsRow.css('display', '');
                    am = allMonitorsEl.prop("checked");
                } else {
                    allMonitorsRow.hide();
                    am = false;
                }
                self.val(w + "x" + h + ":" + fs + ":" + am);
            }

            function parseCustomGeometry(customGeometry) {
                var regex, values, geometryValues, w, h, fs, am;

                regex = new RegExp("^[1-9][0-9]*$");
                values = $.trim(customGeometry).split(":", 3);
                geometryValues = values[0].split("x", 2);
                w = geometryValues[0];
                if (!w || !regex.test(w)) {
                    w = width;
                }
                widthEl.val(w);

                h = geometryValues[1];
                if (!h || !regex.test(h)) {
                    h = height;
                }
                heightEl.val(h);

                fs = values[1] === "true";
                am = fs && (values[2] === "true");
                if (fs) {
                    // Do not use show() since it add "display: block" breaking the tableWrapper
                    allMonitorsRow.css('display', '');
                } else {
                    allMonitorsRow.hide();
                }
                fullScreenEl.prop("checked", fs);
                allMonitorsEl.prop("checked", am);

                detailsRow.css('display', '');
                widthEl.focus();
                self.val(w + "x" + h + ":" + fs + ":" + am);
            }

            self = $(this);
            id = self.attr('id') || 'id-' + Math.floor(9999 * (Math.random() % 1));

            devicePxPerCssPx = DetectZoom.device();

            // FIXME: It seems this does not work with iPad
            width = Math.ceil(screen.width * devicePxPerCssPx);
            height = Math.ceil(screen.height * devicePxPerCssPx);

            autoGeometry = width + "x" + height + ":true:false";
            defaultGeometry = $.trim(self.val());
            // TODO: if defaultGeometry is empty check cookie
            if (autoGeometry === defaultGeometry) {
                defaultGeometry = "auto";
            }

            self.hide();
            presets = $('<select class="hy-interactive-geometry-preset-selector ef-srv-ide-action-properties"/>').insertAfter(self);
            detailsRow = $('<div class="hy-interactive-geometry-details input-area">' +
                    '      <table>' +
                    '        <tr>' +
                    '          <td>Resolution:</td>' +
                    '          <td>' +
                    '            <input type="text" size="5" maxlength="5" name="' + id + '-width"/> &#215; ' +
                    '            <input type="text" size="5" maxlength="5" name="' + id + '-height"/>' +
                    '          </td>' +
                    '        </tr>' +
                    '        <tr>' +
                    '          <td><label for="' + id + '-fullscreen">Fullscreen Mode:</label></td>' +
                    '          <td><input type="checkbox" name="' + id + '-fullscreen"/></td>' +
                    '        </tr>' +
                    '        <tr class="hy-interactive-geometry-details-all-monitors">' +
                    '          <td><label for="' + id + '-all-monitors">Use All Monitors:</label></td>' +
                    '          <td><input type="checkbox" name="' + id + '-all-monitors"/></td>' +
                    '        </tr>' +
                    '      </table>' +
                    '    </div>').hide().insertAfter(presets);

            widthEl = $(':input[name="' + id + '-width"]', detailsRow).val(width);
            heightEl = $(':input[name="' + id + '-height"]', detailsRow).val(height);
            fullScreenEl = $(':input[name="' + id + '-fullscreen"]', detailsRow);
            allMonitorsEl = $(':input[name="' + id + '-all-monitors"]', detailsRow);
            allMonitorsRow = $("tr.hy-interactive-geometry-details-all-monitors", detailsRow).hide();

            presets.change(function () {
                var value = $(this).val();
                if (value === "custom") {
                    // Do not use show() since it add "display: block" breaking the tableWrapper
                    detailsRow.css('display', '');
                    customResolution();
                    widthEl.focus();
                } else {
                    detailsRow.hide();
                    if (value === "auto") {
                        self.val(autoGeometry);
                    } else {
                        self.val(value);
                    }
                }
            });

            $('input', detailsRow).bind('change keyup', customResolution);

            ide.triggerEvent(ide._events.LOAD, ide.service.components.action.displayResolution.name);
            $.hydrogen.invokeService({
                sdf: "/" + $.enginframe.rootContext + "/interactive/lib/xml/com.enginframe.interactive.xml",
                uri: "//com.enginframe.interactive/list.resolutions",
                data: {
                    remote: remote
                },
                success: function (data, textStatus) {
                    var lines, hasCustom, firstOption, customOption, hasDefault, userDefaultOption;
                    hasCustom = false;
                    hasDefault = false;

                    userDefaultOption = $('<option value="">User\'s Default</option>').appendTo(presets);
                    $('<option class="hy-interactive-geometry-preset-selector-dash" disabled="disabled">----</option>').appendTo(presets);
                    data = "\r\n" + data;

                    lines = data.match(/[^\r\n]+/g);
                    $.each(lines, function (index, value) {
                        var geometry, label, i, option;
                        i = value.search(/\s/);
                        if (i > 0) {
                            geometry = $.trim(value.substr(0, i));
                            label = $.trim(value.substr(i + 1));
                            option = $('<option value="' + geometry + '">' + efEncodeHtml(label) + '</option>').appendTo(presets);
                            if (geometry === defaultGeometry) {
                                option.attr("selected", "selected");
                                hasDefault = true;
                            }
                            if (index === 0) {
                                firstOption = option;
                            }
                        }
                        else if ($.trim(value) === "custom") {
                            hasCustom = true;
                        }
                        else if ($.trim(value) === "") {
                            if (index === 0) {
                                firstOption = userDefaultOption;
                            }
                        }
                    });

                    if (hasCustom) {
                        $('<option class="hy-interactive-geometry-preset-selector-dash" disabled="disabled">----</option>').appendTo(presets);
                        customOption = $('<option value="custom">Custom...</option>').appendTo(presets);
                    }

                    if (!hasDefault) {
                        if (defaultGeometry === "") {
                            userDefaultOption.attr("selected", "selected");
                        }
                        else if (hasCustom && defaultGeometry) {
                            customOption.attr("selected", "selected");
                            parseCustomGeometry(defaultGeometry);
                        }
                        else {
                            try {
                                firstOption.attr("selected", "selected");
                            }
                            catch (e) {
                                // IE6 throws an exception that we can safely ignore, selected is set anyway
                            }
                            if (firstOption.val() === "auto") {
                                self.val(autoGeometry);
                            }
                            else {
                                self.val(firstOption.val());
                            }
                        }
                    }
                    presets.css('width', 'auto');

                    ide.triggerEvent(ide._events.LOAD_DONE, ide.service.components.action.displayResolution.name);
                },
                error: function () {
                    ide.triggerEvent(ide._events.LOAD_ERROR, ide.service.components.action.displayResolution.name);
                },
                dataType: 'text'
            });
        });
    };
}(jQuery));
