#!/bin/bash

 ################################################################################
 ################################################################################
# Copyright 2023-2025 by NI SP Software GmbH, All rights reserved.
# Copyright 1999-2023 by Nice, srl., All rights reserved.
#
# This software includes confidential and proprietary information
# of NI SP Software GmbH ("Confidential Information").
# You shall not disclose such Confidential Information
# and shall use it only in accordance with the terms of
# the license agreement you entered into with NI SP Software.
################################################################################
################################################################################ ################################################################################
 ################################################################################
 # $EF_ROOT/plugins/sge/bin/functions.lib.sh - common functions for SGE Plugin
################################################################################


# Get GE flavor
# NOTE: you need to source SGE_PROFILE to use this function.
get_sge_flavor() {
    # qstat -help | head -n 1 returns GE version.
    # I.e. "UGE 8.2.1" for UGE, "SGE 8.1.9" for SoGE, "SGE 6.2u5" for SGE, "OGS/GE 2011.11p1" for OGS
    local _flavor=$(qstat -help | head -n 1 | cut -d ' ' -f 1)

    echo "${_flavor}"
}


# Get jobstatus from given jobid
#   $1 jobid
# NOTE: you need to source SGE_PROFILE to use this function.
get_jobstatus() {
    local _jobid=$1

    local _qstat_out=$(qstat)
    local _ccount=$(echo "${_qstat_out}" | sed '1!d;s/ state .*$//' | wc -c)
    local _jobstatus=$(echo "${_qstat_out}" | grep "^\ *${_jobid}" | sed "1!d;s/^.\{${_ccount}\}//g;s/\([^ ]*\) .*/\1/")
    echo "${_jobstatus}"
}

find_spooler_dir() {
    # for running jobs check a variable set from qsub
    _id=$1
    _spooler=`qstat -j $_id 2>/dev/null| grep context | awk -F':' '{print $2}' | sed 's/\,/\n/' | grep EF_SPOOLER | awk -F'=' '{print $2}'`

    # for done jobs we use the workaround to find in spooler file system
    if [ "${_spooler}" = "" ]; then
        __USER="${EF_USER}"
        [ -z "${__USER}" ] && __USER="default"
        EF_SPOOLERS_BASE=`dirname ${EF_SPOOLER}`
        EF_SPOOLERS_BASE=`dirname ${EF_SPOOLERS_BASE}`

        for job in "${EF_SPOOLERS_BASE}"/${__USER}/*/.sgejobs ; do
            if [ -f "${job}" ] ; then
                cat "${job}" | while read line ; do
                    line=`echo $line | sed "s/\..*$//"`
                    if [ "${line}" = "${_id}" ] ; then
                        job_spooler=`dirname "${job}"`
                        echo "${job_spooler}"
                        return 0
                    fi
                done
            fi
        done
    else
        echo $_spooler
        return 0
    fi
}


find_spooler_uri() {
    VALID_SPOOLER=`find_spooler_dir $@`
    [ -x "${VALID_SPOOLER}" ] && echo "spooler://${VALID_SPOOLER}"
}


ef_killjob() {
    _message="`${SGE_BINDIR}/qdel ${_jobid} 2>&1`"
    [ -n "${_message}" ] && echo "<grid:message ${EF_XMLNS_grid}>$(ef_xml_escape_content -i "${_message}")</grid:message>"
    ef_qstat2xml
}


ef_switchjob() {
    _message="`${SGE_BINDIR}/qalter ${_queue} ${_jobid} 2>&1`"
    [ -n "${_message}" ] && echo "<grid:message ${EF_XMLNS_grid}>$(ef_xml_escape_content -i "${_message}")</grid:message>"
    ef_qstat2xml
}


ef_rename() {
    _message="`${SGE_BINDIR}/bmod -J \"${_name}\" ${_jobid} 2>&1`"
    [ -n "${_message}" ] && echo "<grid:message ${EF_XMLNS_grid}>$(ef_xml_escape_content -i "${_message}")</grid:message>"
    ef_qstat2xml
}

ef_changestatus() {
    DETAILS=true

    case "${_action}" in
        kill) _message="`${SGE_BINDIR}/qdel ${_jobid} 2>&1`"
            _exit_code=$?
            DETAILS=false ;;
        stop) _message="`${SGE_BINDIR}/qmod -sj ${_jobid} 2>&1`"
            _exit_code=$? ;;
        resume) _message="`${SGE_BINDIR}/qmod -usj ${_jobid} 2>&1`"
            _exit_code=$? ;;
        hold) _message="`${SGE_BINDIR}/qhold ${_jobid} 2>&1`"
            _exit_code=$? ;;
        release) _message="`${SGE_BINDIR}/qrls ${_jobid} 2>&1`"
            _exit_code=$? ;;
        *) _message="Bad status requested"
            _exit_code=1
    esac

    [ -n "${_message}" -a ! "${_exit_code}" = 0 ] && echo "<grid:message ${EF_XMLNS_grid}>$(ef_xml_escape_content -i "${_message}")</grid:message>"

    [ "${DETAILS}" = "true" ] && ef_qstat2xml
}


jobname_sanitize() {
    # Read from pipe
    local _jobname="${1:-$(</dev/stdin)}"

    local _flavor="$(get_sge_flavor)"
    case "${_flavor}" in
    SGE)
        # SGE: object_name : An object name is a sequence of up to 512 ASCII string characters
        #   except "\n", "\t", "\r", " ", "/", ":", "'", "\", "[", "]", "{", "}", "|", "(", ")", "@", "%", "," or the " character itself.
        # SoGE: object_name : An object name is a sequence of up to 512 ASCII printing characters
        #   except SPACE, "/", ":", "'", "\", "[", "]", "{", "}", "|", "(", ")", "@", "%", "," or the '"' character itself.
        # NOTE: we need to escape "£" and "$" too
        printf "%s" "${_jobname}" | tr '\\/|:@%$,"{}[]()[:space:]'"'" _
        ;;
    UGE|OGS/GE)
        # UGE and OGS/GE: name
        #   The name may be any arbitrary alphanumeric ASCII string, but may not contain  "\n", "\t", "\r", "/", ":", "@", "\", "*",  or "?".
        printf "%s" "${_jobname}" | tr '\n\t\r/:@\\*?' _
        ;;
    *)
        # Apply a white-list instead of a black-list
        printf "%s" "${_jobname}" | tr -c 'a-zA-Z0-9' _
        ;;
    esac
}

#
# vi: ts=4 sw=4 et syntax=sh :
#
