
#
################################################################################
# Copyright 2023-2025 by NI SP Software GmbH, All rights reserved.
# Copyright 1999-2023 by Nice, srl., All rights reserved.
#
# This software includes confidential and proprietary information
# of NI SP Software GmbH ("Confidential Information").
# You shall not disclose such Confidential Information
# and shall use it only in accordance with the terms of
# the license agreement you entered into with NI SP Software.
################################################################################
#################################################################################


# Output format for normal jobs:
#
# 01: id,index
# 02: submission host
# 03: execution hosts
# 04: submission time  (TIME IS IN THE FORMAT YY:MM:DD:HH:MM)
# 05: termination time
# 06: execution time
# 07: job name
# 08: queue
# 09: account / project
# 10: job status (scheduler)
# 11: job status (enginframe)
# 12: job status (compatibility)
# 13: owner / user
# 14: exit code
# 15: user email address
# 16: submission directory
# 17: execution directory
# 18: shell
# 19: command
# 20: total cpu usage (user + system)
# 21: memory usage
# 22: swap usage
# 23: max swap usage (swap or vmem?)
# 24: pids
# >= 25: reasons
#


# Output format for compressed job arrays:
# 01: array marker ("array")
# 02: id
# 03: name
# 04: owner
# 05: account
# 06: queue
# 07: command
# 08: submission-time
# 09: submission-host
# 10: submission-directory
# 11: number of jobs
# 12: number of pending jobs
# 13: number of done jobs
# 14: number of running jobs
# 15: number of exited jobs
# 16: number of interrupted jobs
# 17: number of suspended jobs
# 18: job pids



BEGIN {
    current_jobid = 0
    job_position = 0
    listmonths = "Jan,Feb,Mar,Apr,May,Jun,Jul,Aug,Sep,Oct,Nov,Dec"
    split(listmonths, months, ",")

    # Read status mapping file
    while (getline < statusmap > 0) {
        split($0, fields, ":")
        s = split(fields[2], label, " ")
        for (i=1; i<=s; i++) {
            states[label[i]] = fields[1]
        }
    } 
    close(statusmap);
}

function emitOutput(jobid, array) {
          if (array)
            printf array              ";"
          else
            printf jobid              ";"
          printf \
            SUBMISSION_HOST[jobid]    ";" \
            EXECUTION                 ";" \
            SUBMISSIONDATE[jobid]     ";" \
            TERMINATIONDATE[jobid]    ";" \
            EXECUTIONDATE[jobid]      ";" \
            JOBNAME[jobid]            ";" \
            QUEUE                     ";" \
            PROJECT[jobid]            ";" \
            STATUSCODE                ";" \
            efStatus                  ";" \
            STATUS                    ";" \
            OWNER[jobid]              ";" \
            EXIT_STATUS[jobid]        ";" \
            MAILLIST[jobid]           ";" \
            WORKDIR[jobid]            ";" \
            CWD[jobid]                ";" \
            SHELL[jobid]              ";" \
            COMMAND[jobid]            ";" \
            USAGE_CPU[jobid]          ";" \
            USAGE_MEM[jobid]          ";" \
            USAGE_VMEM[jobid]         ";" \
            USAGE_MAXVMEM[jobid]      ";" \
            PIDLIST                   ";"

          if (ROWREASON[jobid] > 0) {
            # remove last newlines
            sub("\n+ *$", "", REASONS[jobid]);
            # change all newlines into two spaces
            # FIXME: change \n into field separator ';' ? 
            gsub("\n","  ", REASONS[jobid]);
            printf REASONS[jobid] ";"
          }
          printf "\n";
}

# Main
{

    if (/^---joblist/) {
        split($2,JOBLIST,"%")
        next;
    }

    if (/^---tasklist/) {
        split($2,TASKLIST,"%")
        next;
    }

    if (/^---statuslist/) {
        split($2,STATUSLIST,"%")
        next;
    }

    if (/^---queuelist/) {
        split($2,QUEUELIST,"%")
        next;
    }

    if (/^---executionlist/) {
        split($2,EXECUTIONLIST,"%")
        next;
    }

    if (/^---timelist/) {
        split($2,TIMELIST,"%")
        next;
    }

    if (/^---datelist/) {
        split($2,DATELIST,"%")
        next;
    }

    if (/^---pidslist:/) {
        sub("^---pidslist: *", "")
        PIDLIST=$0;
        next;
    }

    if (/^---qacct-zone/) {
        qacct_zone = 1
        next;
    }
    
    if (/Following jobs do not exist/) {
       NJOB = 0;
       job_position++;
       job_map[job_position] = NJOB
       # FIXME: add 'getline' here to discard next line instead of using the next pattern matching?
    }

    if (/^[0-9]*$/) {} # nop

    if (/^job_number/) {
        job_position++

        if (TASKLIST[job_position] != 0) {
            if (COMPRESS_ARRAY != "true") 
                NJOB = $2 "," TASKLIST[job_position]
            else { 
                NJOB = $2
                IS_ARRAY[job_position] = TASKLIST[job_position]
            }
        } else
          NJOB = $2
        job_map[job_position] = NJOB
        ACTIVE[NJOB] = 1
        tasks = IS_ARRAY[job_position]
        #print "          [DEBUG] job_number - NJOB=" NJOB " position=" job_position " tasks=" IS_ARRAY[job_position]
    } else if (/^jobnumber/) {
        # from qacct
        current_jobid = $2
        #print "          [DEBUG] jobnumber - current_job=" current_jobid " position+1=" job_position + 1 " JOBLIST=" JOBLIST[job_position + 1] " tasks=" IS_ARRAY[job_position]
    } else if (/^taskid/ && $2 !~ /undefined/) {
        # from qacct
        job_position++
        if (COMPRESS_ARRAY != "true") 
            NJOB = current_jobid "," $2
        else { 
            NJOB = current_jobid
            IS_ARRAY[job_position] = $2
            FINISHED[NJOB "," $2] = 1
        }
        job_map[job_position] = NJOB
        tasks = IS_ARRAY[job_position]
        FINISHED[NJOB] = 1
        EXEC_HOST[NJOB] = pending_hostname
        OWNER[NJOB] = pending_owner
        PROJECT[NJOB] = pending_project
        JOBNAME[NJOB] = pending_jobname
        QUEUE_LIST[NJOB] = pending_queuename
        #print "          [DEBUG] taskid=" $2 " NJOB=" NJOB " position=" job_position " FINISHED=" FINISHED[NJOB]
    } else if (/^taskid/ && $2 ~ /undefined/) {
        job_position++
        NJOB = current_jobid
        job_map[job_position] = NJOB
        tasks = IS_ARRAY[job_position]
        FINISHED[NJOB] = 1
        EXEC_HOST[NJOB] = pending_hostname
        OWNER[NJOB] = pending_owner
        PROJECT[NJOB] = pending_project
        JOBNAME[NJOB] = pending_jobname
        QUEUE_LIST[NJOB] = pending_queuename
        tasks = "undefined"
        #print "          [DEBUG] taskid=" $2 " NJOB=" NJOB " position=" job_position " FINISHED=" FINISHED[NJOB]
    } else if (/^submission_time/ || /^qsub_time/) {
        if ($2 != "-/-") {
            if ($2 ~ /^[0-9][0-9]*\/[0-9][0-9]*\/[0-9][0-9]*/) { 
                # UGE 8.2 format
                split($2, TMPDATE, "/");
                split($3, TIME, ":");

                # Translate months to numbers
                sub("01", "Jan", TMPDATE[1])
                sub("02", "Feb", TMPDATE[1])
                sub("03", "Mar", TMPDATE[1])
                sub("04", "Apr", TMPDATE[1])
                sub("05", "May", TMPDATE[1])
                sub("06", "Jun", TMPDATE[1])
                sub("07", "Jul", TMPDATE[1])
                sub("08", "Aug", TMPDATE[1])
                sub("09", "Sep", TMPDATE[1])
                sub("10", "Oct", TMPDATE[1])
                sub("11", "Nov", TMPDATE[1])
                sub("12", "Dec", TMPDATE[1])

                SUBMISSIONDATE[NJOB] = TMPDATE[3] ":" TMPDATE[1] ":" TMPDATE[2] ":" TIME[1] ":" TIME[2]                
	    } else if ($2 ~ /^[0-9][0-9]*\-[0-9][0-9]*\-[0-9][0-9]*/) {
		# 2024-07-16 16:31:48.770564
                # Gridware 9 format
                # print "# Gridware 9 format" $0  
                split($2, TMPDATE, "-");
                split($3, TIME, ":");

                # Translate months to numbers
                sub("01", "Jan", TMPDATE[2])
                sub("02", "Feb", TMPDATE[2])
                sub("03", "Mar", TMPDATE[2])
                sub("04", "Apr", TMPDATE[2])
                sub("05", "May", TMPDATE[2])
                sub("06", "Jun", TMPDATE[2])
                sub("07", "Jul", TMPDATE[2])
                sub("08", "Aug", TMPDATE[2])
                sub("09", "Sep", TMPDATE[2])
                sub("10", "Oct", TMPDATE[2])
                sub("11", "Nov", TMPDATE[2])
                sub("12", "Dec", TMPDATE[2])

                SUBMISSIONDATE[NJOB] = TMPDATE[1] ":" TMPDATE[2] ":" TMPDATE[3] ":" TIME[1] ":" TIME[2]                
            } else {
                split($5, TIME, ":")
                SUBMISSIONDATE[NJOB] = $6 ":" $3 ":" $4 ":" TIME[1] ":" TIME[2]
                #print "          [DEBUG] submissiontime=" SUBMISSIONDATE[NJOB] " job=" NJOB
            }
        }
    } else if (/^end_time/) {
        if ($2 != "-/-") {
            if ($2 ~ /^[0-9][0-9]*\/[0-9][0-9]*\/[0-9][0-9]*/) { 
                # UGE 8.2 format
                split($2, TMPDATE, "/");
                split($3, TIME, ":");

                # Translate months to numbers
                sub("01", "Jan", TMPDATE[1])
                sub("02", "Feb", TMPDATE[1])
                sub("03", "Mar", TMPDATE[1])
                sub("04", "Apr", TMPDATE[1])
                sub("05", "May", TMPDATE[1])
                sub("06", "Jun", TMPDATE[1])
                sub("07", "Jul", TMPDATE[1])
                sub("08", "Aug", TMPDATE[1])
                sub("09", "Sep", TMPDATE[1])
                sub("10", "Oct", TMPDATE[1])
                sub("11", "Nov", TMPDATE[1])
                sub("12", "Dec", TMPDATE[1])

                TERMINATIONDATE[NJOB] = TMPDATE[3] ":" TMPDATE[1] ":" TMPDATE[2] ":" TIME[1] ":" TIME[2]
	    } else if ($2 ~ /^[0-9][0-9]*\-[0-9][0-9]*\-[0-9][0-9]*/) {
		# 2024-07-16 16:31:48.770564
                # Gridware 9 format
                # print "# Gridware 9 format" $0  
                split($2, TMPDATE, "-");
                split($3, TIME, ":");

                # Translate months to numbers
                sub("01", "Jan", TMPDATE[2])
                sub("02", "Feb", TMPDATE[2])
                sub("03", "Mar", TMPDATE[2])
                sub("04", "Apr", TMPDATE[2])
                sub("05", "May", TMPDATE[2])
                sub("06", "Jun", TMPDATE[2])
                sub("07", "Jul", TMPDATE[2])
                sub("08", "Aug", TMPDATE[2])
                sub("09", "Sep", TMPDATE[2])
                sub("10", "Oct", TMPDATE[2])
                sub("11", "Nov", TMPDATE[2])
                sub("12", "Dec", TMPDATE[2])

                TERMINATIONDATE[NJOB] = TMPDATE[1] ":" TMPDATE[2] ":" TMPDATE[3] ":" TIME[1] ":" TIME[2]
            } else {
                split($5, TIME, ":")
                TERMINATIONDATE[NJOB] = $6 ":" $3 ":" $4 ":" TIME[1] ":" TIME[2]
            }
        }
    } else if (/^start_time/) {
        if ($2 != "-/-") {
            if ($2 ~ /^[0-9][0-9]*\/[0-9][0-9]*\/[0-9][0-9]*/) {
                # UGE 8.2 format
                split($2, TMPDATE, "/");
                split($3, TIME, ":");

                # Translate months to numbers
                sub("01", "Jan", TMPDATE[1])
                sub("02", "Feb", TMPDATE[1])
                sub("03", "Mar", TMPDATE[1])
                sub("04", "Apr", TMPDATE[1])
                sub("05", "May", TMPDATE[1])
                sub("06", "Jun", TMPDATE[1])
                sub("07", "Jul", TMPDATE[1])
                sub("08", "Aug", TMPDATE[1])
                sub("09", "Sep", TMPDATE[1])
                sub("10", "Oct", TMPDATE[1])
                sub("11", "Nov", TMPDATE[1])
                sub("12", "Dec", TMPDATE[1])

                EXECUTIONDATE[NJOB] = TMPDATE[3] ":" TMPDATE[1] ":" TMPDATE[2] ":" TIME[1] ":" TIME[2]
	    } else if ($2 ~ /^[0-9][0-9]*\-[0-9][0-9]*\-[0-9][0-9]*/) {
		# 2024-07-16 16:31:48.770564
                # Gridware 9 format
                # print "# Gridware 9 format" $0  
                split($2, TMPDATE, "-");
                split($3, TIME, ":");

                # Translate months to numbers
                sub("01", "Jan", TMPDATE[2])
                sub("02", "Feb", TMPDATE[2])
                sub("03", "Mar", TMPDATE[2])
                sub("04", "Apr", TMPDATE[2])
                sub("05", "May", TMPDATE[2])
                sub("06", "Jun", TMPDATE[2])
                sub("07", "Jul", TMPDATE[2])
                sub("08", "Aug", TMPDATE[2])
                sub("09", "Sep", TMPDATE[2])
                sub("10", "Oct", TMPDATE[2])
                sub("11", "Nov", TMPDATE[2])
                sub("12", "Dec", TMPDATE[2])

                # EXECUTIONDATE = TMPDATE[1] ":" TMPDATE[2] ":" TMPDATE[3] ":" TIME[1] ":" TIME[2]
                EXECUTIONDATE[NJOB] = TMPDATE[1] ":" TMPDATE[2] ":" TMPDATE[3] ":" TIME[1] ":" TIME[2]
            } else {
                split($5, TIME, ":")
                # EXECUTIONDATE = $6 ":" $3 ":" $4 ":" TIME[1] ":" TIME[2]
                EXECUTIONDATE[NJOB] = $6 ":" $3 ":" $4 ":" TIME[1] ":" TIME[2]
            }
        }
    } else if (/^sge_o_host/) {
        SUBMISSION_HOST[NJOB] = $2
    } else if (/^submit_host/) {
        SUBMISSION_HOST[NJOB] = $2
    } else if (/^hostname/) {
        pending_hostname = $2
    } else if (/^job_name[:]*/) {
        split($2, JOB, " ")
        sub("^job_name[:]* *", "")
        JOBNAME[NJOB] = $0
    } else if (/^jobname[:]*/) {
        split($2, JOB, " ")
        sub("^jobname[:]* *", "")
        sub(" *$", "")
        pending_jobname = $0
    } else if (/^account/) {
        # FIXME: the use of field separator ';' should be revisited
        gsub(";", ",", $2)
        PROJECT[NJOB] = $2
    } else if (/^project/) {
        pending_project = $2
    } else if (/^qname[:]*/) {
        split($2, JOB, " ")
        sub("^qname[:]* *", "")
        sub("@.*$", "")
        gsub(" ", "")
        pending_queuename = $0
    } else if (/^hard_queue_list[:]*/) {
        split($2, JOB, " ")
        sub("^hard_queue_list[:]* *", "")
        sub("@.*$", "")
        gsub(" ", "")
        QUEUE_LIST[NJOB] = $0
    } else if (/^owner/) {
        if (qacct_zone) 
          pending_owner = $2
        else
          OWNER[NJOB] = $2
    } else if (/^mail_list/) {
        MAILLIST[NJOB] = $2
    } else if (/^sge_o_shell/) {
        SHELL[NJOB] = $2
    } else if (/^sge_o_workdir[:]*/) {
        sub("^sge_o_workdir[:]* *","")
        WORKDIR[NJOB] = $0
    } else if (/^cwd[:]*/) {
        sub("^cwd[:]* *","")
        CWD[NJOB] = $0
    } else if (/^error reason[:]*/) {
        ROWREASON[NJOB]++
        # sub(/error reason\ +[0-9]+[:]*\ +/, "")
        sub(/error reason\s+[0-9]+[:]*\s+/, "")
        REASONS[NJOB] = $0 "\n" REASONS[NJOB]
    } else if (/^scheduling info[:]*/) {
        PENDFLAG[NJOB] = 1
        sub("scheduling info[:]* *","")
        sub("\n","")
        if(sub(" *(Collecting of scheduler job information is turned off)", "") == 0) {
            ROWREASON[NJOB]++;
            if (REASONS[NJOB])
              REASONS[NJOB] = $0 "\n" REASONS[NJOB]
            else
              REASONS[NJOB] = $0
        }
    } else if (/^script_file[:]*/) {
        sub("^script_file[:]* *","")
        COMMAND[NJOB] = $0
    } else if (/^usage[:]* /) {
        #  usage    1:                 cpu=00:00:00, mem=0.00026 GBs, io=0.00000, vmem=10.188M, maxvmem=10.188M
        # sub(/usage\ +[0-9]+[:]*\ +/, "")
        sub(/usage\s+[0-9]+[:]*\s+/, "")
        NGROUPS = split($0, USAGEGROUP, ",")
        for (group = 1; group <= NGROUPS; group++) {
            split(USAGEGROUP[group], USAGE, "=")
            sub("^ +", "", USAGE[1])
            if (USAGE[1] == "cpu")
                USAGE_CPU[NJOB] = USAGE[2]
            if (USAGE[1] == "mem")
                USAGE_MEM[NJOB] = USAGE[2]
            if (USAGE[1] == "vmem")
                USAGE_VMEM[NJOB] = USAGE[2]
            if (USAGE[1] == "maxvmem")
                USAGE_MAXVMEM[NJOB] = USAGE[2] 
            if (USAGE[1] == "io")
                USAGE_IO[NJOB] = USAGE[2]
        }
    } else if(/^cpu/) {
        USAGE_CPU[NJOB] = $2
    } else if(/^mem/) {
        USAGE_MEM[NJOB] = $2
    } else if(/^vmem/) {
        USAGE_VMEM[NJOB] = $2
    } else if(/^io/) {
        USAGE_IO[NJOB] = $2
    } else if(/^iow/) {
        USAGE_IOW[NJOB] = $2
    } else if(/^maxvmem/) {
        USAGE_MAXVMEM[NJOB] = $2
    } else if (/^exit_status/) {
        if (COMPRESS_ARRAY != "true" || tasks == "undefined" )
          EXIT_STATUS[NJOB] = $2
        else
          EXIT_STATUS[NJOB "," tasks] = $2
    } else if(/^failed[:]*/) {
        if (COMPRESS_ARRAY != "true" || tasks == "undefined" ) {
            FAILED_CODE[NJOB] = $2
            # sub(/failed\ +[0-9]+\ +[:]*\ +/,"")
            sub(/failed\s+[0-9]+\s+[:]*\s+/,"")
            FAILED_MSG[NJOB] = $0  
        } else 
            FAILED_CODE[NJOB "," tasks] = $2
    }

} END {

      array_to_print = 0
      for (i=1; i<=job_position; i++) {

        current_jobid = job_map[i]

        #printf "          [DEBUG] cj:" current_jobid " pos:" job_position "\n"

        if (current_jobid == 0) continue

        if (COMPRESS_ARRAY == "true" && IS_ARRAY[i] && IS_ARRAY[i] != 0) {
            # Produce array data in compressed mode

            ntasks = step = 1
            tasks = IS_ARRAY[i]

            if (IS_ARRAY[i] ~ /[0-9][0-9]*-[0-9][0-9]*:[0-9][0-9]*/) {
              # 16,1-5:1 format (x items)
              lower = higher = step = IS_ARRAY[i]
              sub("-.*","", lower);
              sub(".*-","", higher); sub(":.*","", higher);
              sub(".*:","", step);
              ntasks = higher - lower + 1
              FINISHED[current_jobid "," tasks ] = FINISHED[current_jobid "," lower ]
              EXIT_STATUS[current_jobid "," tasks] = EXIT_STATUS[current_jobid "," lower]
              FAILED_CODE[current_jobid "," tasks] = FAILED_CODE[current_jobid "," lower]

              lower = lower + 0
              higher = higher + 0
              step = step + 0
              
              seen_job[current_jobid ",0" ] = 1    # avoid to count single separate array task jobs
              for (j=lower; j<=higher; j += step) {
                seen_job[current_jobid "," j ] = 1
                #print "          [DEBUG] seen " current_jobid "-" j "=1 lower=" lower " higher=" higher " step=" step
              }
            } else if (IS_ARRAY[i] ~ /[0-9][0-9]*,[0-9][0-9]*/) {
              # 16,4,5 (2 items)
              lower = higher = IS_ARRAY[i]
              sub(",.*","", lower);
              sub(".*,","", higher);
              ntasks = 2
              FINISHED[current_jobid "," tasks ] = FINISHED[current_jobid "," lower ]
              EXIT_STATUS[current_jobid "," tasks] = EXIT_STATUS[current_jobid "," lower]
              FAILED_CODE[current_jobid "," tasks] = FAILED_CODE[current_jobid "," lower]

              lower = lower + 0
              higher = higher + 0
              step = step + 0

              seen_job[current_jobid ",0" ] = 1    # avoid to count single separate array task jobs
              for (j=lower; j<=higher; j += 1) {
                seen_job[current_jobid "," j ] = 1
                #print "          [DEBUG] seen " current_jobid "," j "=1 lower=" lower " higher=" higher " step=" step
              }
            } else if (seen_job[current_jobid "," tasks] && seen_job[current_jobid "," tasks] != 0) {
                # doubled item, count only one
                continue
            } else {
                #print "          [DEBUG] seen " current_jobid "," tasks
                seen_job[current_jobid ",0" ] = 1    # avoid to count single separate array task jobs
                seen_job[current_jobid "," tasks] = 1
            }

            if (a_tasks[current_jobid] == "undefined" || a_tasks[current_jobid] == 0) {
              array_to_print++
              a_position[array_to_print] = current_jobid;
              a_queue[current_jobid] = QUEUELIST[i]
            }

            a_tasks[current_jobid] += ntasks
            #printf "          [DEBUG] A jobid=" current_jobid " array_to_print=" array_to_print " i=" i " IS_ARRAY=" IS_ARRAY[i] " ntasks=" ntasks " a_tasks=" a_tasks[current_jobid] " a_position[array_to_print]=" a_position[array_to_print] " FINISHED=" FINISHED[current_jobid "," tasks] " STATUSCODE=" STATUSLIST[i] " EC=" EXIT_STATUS[current_jobid "," tasks] " FC="  FAILED_CODE[NJOB "," tasks] "\n"

            if (FINISHED[current_jobid "," tasks] != 1) {
                # --- Torque job status ---
                STATUSCODE = STATUSLIST[i]

                # --- EF 2013 Job status mapping ---
                STATUSCODE = STATUSLIST[i]
                efStatus = states[STATUSCODE];

                # --- Backward compatibility status ---
                if      (match(STATUSCODE, /[dE]/) >0 ) { STATUS = "EXIT"  }
                else if (match(STATUSCODE, /[w]/)  >0 ) { STATUS = "PEND"  }
                else if (match(STATUSCODE, /[h]/)  >0 ) { STATUS = "HOLD"  }
                else if (match(STATUSCODE, /[rRT]/)>0 ) { STATUS = "RUN"   }
                else if (match(STATUSCODE, /[s]/)  >0 ) { STATUS = "USUSP" }
                else if (match(STATUSCODE, /[St]/) >0 ) { STATUS = "SSUSP" }
                else                                    { STATUS = "UNKWN" }

                if (QUEUE == "") QUEUE = QUEUE_LIST[current_jobid]

            } else {
                if (EXIT_STATUS[current_jobid "," tasks] == "0" && FAILED_CODE[current_jobid "," tasks] == "0") {
                    STATUS = "DONE"
                    efStatus = "Done"
                } else {
                    STATUS = "EXIT"
                    efStatus = "Exit"
                }
            }

            for (k=1; k<=ntasks; k += step) {
                # update array counters
                a_efStatus[current_jobid, efStatus]++
            }

        } else {

            # If this is a task and his array has already been processed, skip
            if (seen_job[current_jobid ",0"] && seen_job[current_jobid ",0"] != 0) continue

            QUEUE = QUEUELIST[i]
            EXECUTION = EXECUTIONLIST[i]
            if (EXECUTION == "" && EXEC_HOST[current_jobid] != "") EXECUTION = EXEC_HOST[current_jobid]

            if (QUEUE == "") QUEUE = QUEUE_LIST[current_jobid]

            if (FINISHED[current_jobid] != 1) {
                # --- Torque job status ---
                STATUSCODE = STATUSLIST[i]

                # --- EF 2013 Job status mapping ---
                efStatus = states[STATUSCODE];
                if (efStatus == "") efStatus = "Unknown"

                # --- Backward compatibility status ---
                if      (match(STATUSCODE, /[dE]/) >0 ) { STATUS = "EXIT"  }
                else if (match(STATUSCODE, /[w]/)  >0 ) { STATUS = "PEND"  }
                else if (match(STATUSCODE, /[h]/)  >0 ) { STATUS = "HOLD"  }
                else if (match(STATUSCODE, /[rRT]/)>0 ) { STATUS = "RUN"   }
                else if (match(STATUSCODE, /[s]/)  >0 ) { STATUS = "USUSP" }
                else if (match(STATUSCODE, /[St]/) >0 ) { STATUS = "SSUSP" }
                else                                    { STATUS = "UNKWN" }


            } else {
                if (EXIT_STATUS[current_jobid] == "0" && FAILED_CODE[current_jobid] == "0") {
                    STATUS = "DONE"
                    efStatus = "Done"
                } else {
                    STATUS = "EXIT"
                    efStatus = "Exit"

                    if (FAILED_CODE[current_jobid] != "0") {
                        ROWREASON[current_jobid]++
                        REASONS[current_jobid] = "Failed: " FAILED_CODE[current_jobid] " - " FAILED_MSG[current_jobid] " - " REASONS[current_jobid]
                    }
                }
            }

            if (STATUS == "RUN" || STATUS == "USUSP" || STATUS == "SSUSP" ) {
                split(DATELIST[i], RAWDATE, "/")
                split(TIMELIST[i], RAWTIME, ":")
                if (RAWDATE[3] != "") {
                    EXECUTIONDATE[NJOB] = RAWDATE[3] ":" months[RAWDATE[1]+0] ":" RAWDATE[2] ":" RAWTIME[1] ":" RAWTIME[2]
                } else {
                    EXECUTIONDATE[NJOB] = ""
                }
            }

            #printf "          [DEBUG] jobid=" current_jobid " array_to_print=" array_to_print " i=" i " IS_ARRAY=" IS_ARRAY[i] " ntasks=" ntasks " a_tasks=" a_tasks[current_jobid] " a_position[array_to_print]=" a_position[array_to_print] " FINISHED=" FINISHED[current_jobid] " STATUSCODE=" STATUSLIST[i] " EC=" EXIT_STATUS[current_jobid] " FC="  FAILED_CODE[NJOB] "\n"

            # Check if its a job array
            if (current_jobid ~ /,[0-9][0-9]*-[0-9][0-9]*:[0-9][0-9]*/) {
              # 16,1-5:1 format (x items)
              # 907,1-3:1,6-10:1
              array_jobid=current_jobid
              sub(",.*", "", array_jobid);

              n_intervals = split(current_jobid, array_intervals, ",")
              # First element of array_intervals[] is the array_jobid
              for (interval_idx = 2; interval_idx <= n_intervals; ++interval_idx) {
                # 1-3:1
                lower = higher = array_intervals[interval_idx];
                sub("-.*","", lower) ; lower = lower + 0
                sub(".*-","", higher); sub(":.*","", higher) ; higher = higher + 0

                for (iterator = lower ; iterator <= higher ; iterator++) {
                    if (!SUBMISSIONDATE[array_jobid "," iterator]) emitOutput(current_jobid, array_jobid "," iterator)
                }
              }

            } else if (current_jobid ~ /,[0-9][0-9]*,[0-9][0-9]*/) {
              # 16,4,5 (2 items)
              array_jobid = higher = current_jobid
              sub(",.*","", array_jobid);
              sub("[0-9][0-9]*,[0-9][0-9]*,","", higher);
              lower = higher - 1
              if (FINISHED[current_jobid] == 1) FINISHED[array_jobid "," lower] = FINISHED[current_jobid]
              if (FINISHED[current_jobid] == 1) FINISHED[array_jobid "," higher] =  FINISHED[current_jobid]

              emitOutput(current_jobid, array_jobid "," lower)
              emitOutput(current_jobid, array_jobid "," higher)

            } else if (current_jobid ~ /[0-9][0-9]*,[0-9][0-9]*/ && ACTIVE[current_jobid] != 1) {
              # 15,6 (already matched item)
              if (FINISHED[current_jobid] != 1) continue
              emitOutput(current_jobid)
            } else {
              if (ACTIVE[current_jobid] != 1) ACTIVE[current_jobid] = 0;
              emitOutput(current_jobid, current_jobid)
            }
        }
      }

      if (array_to_print > 0) {
        for (i=1; i<=array_to_print; i++) {

          jobid = a_position[i];

          printf \
            "array"                          ";" \
            jobid                            ";" \
            JOBNAME[jobid]                   ";" \
            SUBMISSIONDATE[jobid]            ";" \
            PROJECT[jobid]                   ";" \
            a_queue[jobid]                   ";" \
            COMMAND[jobid]                   ";" \
            OWNER[jobid]                     ";" \
            SUBMISSION_HOST[jobid]           ";" \
            CWD[jobid]                       ";" \
            a_tasks[jobid]                   ";" \
            a_efStatus[jobid, "Pending"]     ";" \
            a_efStatus[jobid, "Done"]        ";" \
            a_efStatus[jobid, "Running"]     ";" \
            a_efStatus[jobid, "Exit"]        ";" \
            a_efStatus[jobid, "Interrupted"] ";" \
            a_efStatus[jobid, "Suspended"]   ";" \
            PIDLIST                          ";" \
            "\n"
         }
      }
}


#
# vi: ts=4 sw=4 et syntax=awk :
#

