#!/bin/bash

################################################################################
# Copyright 2023-2025 by NI SP Software GmbH, All rights reserved.
# Copyright 1999-2023 by Nice, srl., All rights reserved.
#
# This software includes confidential and proprietary information
# of NI SP Software GmbH ("Confidential Information").
# You shall not disclose such Confidential Information
# and shall use it only in accordance with the terms of
# the license agreement you entered into with NI SP Software.
################################################################################

################################################################################
# $EF_ROOT/plugins/slurm/bin/functions.lib.sh - common functions for SLURM Plugin
################################################################################

# exec 2>"/tmp/debug.log.$$";set -x

# parse the output of SLURM scontrol show nodes
ef_scontrol_show_nodes_2xml() {
  "${EF_AWK}" -v "slurmclusterid=${cluster}" \
    -f "${EF_ROOT}/plugins/ef/lib/awk/utils.awk" \
    -f "${EF_ROOT}/plugins/slurm/lib/scontrol_show_nodes_2xml.awk"
}

# parse the output of SLURM squeue --noheader -o "${SLURM_JOB_FORMAT}" -j "${jobid}"
ef_squeue_jobs_2xml() {
  # sed scontrol show nodes output to join lines starting with tabs, then produce XML via awk
  [ -n "$host" ] && _host=""
  "${EF_AWK}" \
    -v "hostfilter=$_host" \
    -v "statusmap=${EF_ROOT}/plugins/slurm/conf/grid.job.status.mapping" \
    -v "slurmclusterid=${cluster}" \
    -f "${EF_ROOT}/plugins/ef/lib/awk/utils.awk" \
    -f "${EF_ROOT}/plugins/slurm/lib/squeue_jobs_2xml.awk"
}

# convert sacct output format to the one used with squeue for parsing Slurm jobs
# Note1: the output format of squeue is defined into SLURM_JOB_FORMAT variable
#        inside the ef.slurm.conf file.
# Note2: the output fields have a positional order as defined by SLURM_SACCT_JOB_FORMAT
#        variable inside the ef.slurm.conf file.
ef_sacct2squeue_format() {
  "${EF_AWK}" '{printf \
"JobId="$1"\n"\
"JobName="$2"\n"\
"UserId="$3"\n"\
"Partition="$4"\n"\
"RunTime="$5"\n"\
"JobState="$6"\n"\
"NodeList="$7"\n"\
"SubmitTime="$8"\n"\
"StartTime="$9"\n"\
"NumCPUs="$10"\n"\
"Account="$11"\n"\
"EndTime="$12"\n"
}'
}

jobname_sanitize() {
  local _jobname="${1:-$(</dev/stdin)}"

  printf "%s" "${_jobname}" | tr '[:space:]/:@\\*?' _ | awk '{print substr($1, i, 50)}'
}

spaces_sanitize() {
  local _string="${1:-$(</dev/stdin)}"

  printf "%s" "${_string}" | tr '[:space:]' _
}



# parameters:
#  1: slurm cluster
#  2: slurm client command
slurm_command() {
  local -r cluster_name="$1"

  if [ -z "${cluster_name}" ]; then
    local -r conf_var="SLURM_CLUSTER_CONF"
  else
    # Replace hyphens with underscores only in the variable name
    local -r safe_cluster_name="${cluster_name//-/_}"
    local -r conf_var="SLURM_CLUSTER_${safe_cluster_name}_CONF"
  fi

  if [ -z "${!conf_var}" ]; then
    local -r _slurm_conf_prefix=""
  else
    local -r _slurm_conf_prefix="SLURM_CONF=${!conf_var}"
  fi

  echo "${_slurm_conf_prefix} $2"
}

# parameters:
#  1: slurm cluster id
slurm_cluster_name() {
  local -r _cluster=$1
  _scontrol_out=$(eval "$(slurm_command "${_cluster}" scontrol) show config" 2>&1)
  _exit_code="$?"
  if [ "${_exit_code}" != "0" ]; then
      ef_error "${_scontrol_out} - exit code (${_exit_code})" \
               "EnginFrame SLURM Plugin Error" \
               "scontrol show config"
      exit ${_exit_code}
  fi

  echo "${_scontrol_out}" | sed -n 's/^ClusterName[[:space:]]*=[[:space:]]*//p'
}

# parameters:
#  1: slurm cluster id
slurm_label() {
  local -r _cluster="$1"
  # Replace hyphens with underscores only in the variable name
  local -r safe_cluster_name="${_cluster//-/_}"
  local -r _slurm_label_var="SLURM_CLUSTER_${safe_cluster_name}_LABEL"
  local _slurm_label="${!_slurm_label_var}"
  [ -z "${_slurm_label}" ] && _slurm_label=$(slurm_cluster_name "${_cluster}")
  [ -z "${_slurm_label}" ] && _slurm_label="${_cluster}"
  [ -z "${_slurm_label}" ] && _slurm_label="default"
  echo "${_slurm_label}"
}

# parameters:
#  1: slurm cluster
#  2: caller
slurm_cluster_info() {
  local -r _cluster="$1"

  local -r _xmlcluster=$(ef_xml_escape -i "${_cluster:-$(fallback_cluster_name "${_cluster}")}")
  local -r _slurm_label="$(slurm_label "${_cluster}")"
  local -r _xmllabel=$(ef_xml_escape -i "${_slurm_label}")

  printf '<grid:cluster xmlns:grid="%s" type="%s" id="%s" name="%s" />\n' \
         'http://www.enginframe.com/2000/GRID' 'slurm' \
         "${_xmlcluster}" "${_xmllabel}"
}

# parameters:
#  1: slurm cluster name
fallback_cluster_name() {
  _cluster_first="$(eval "$EF_ROOT/plugins//${EF_CURRENT_GRID_MANAGER}/grid/grid.cluster.first")"
  if [ -n "${_cluster_first}" ]; then
      echo "${_cluster_first}"
  else
      echo "$(slurm_cluster_name $1)"
  fi
}

